"""
    This module provides a stripped version of :py:class:`RequestKey` from bitbucket yasm repo.
"""

import bisect

from infra.yasm.gateway.lib.util.common import cached_property
from infra.yasm.gateway.lib.util.verification import is_correct_tag_name, is_correct_request_tag_value

from base import parse_underscored_tags
from errors import MissingTagNameTagValuesSeparator, EmptyTagName, TagNameError
from errors import EmptyItype, MultipleItypeValues, ItypeError, MissingItype
from errors import EmptyTagValue, TagValueError, ArgumentTypeError, ArgumentUnicodeError
from errors import TagMeetsTwice, ItypeMeetsTwice
from errors import TagsIsNotDict


def is_skippable_tag_value(raw):
    return not raw or raw == "*"


def concatenate_tag_values(tag_values):
    return ",".join(tag_values)


class RequestKey(object):
    """
    Parsed and validated request key.

    This class should not be instantiated via default constructor, use
    method :py:method:`from_string`.

    :param str itype: Instance type
    :param list[(str, Matcher)] tags: Other tags in lexicographical order
    """

    __cache = {}

    def __init__(self, itype, tags):
        self.itype = itype
        self.__tags = tags
        self.__tag_names = set(tag[0] for tag in tags)

    @classmethod
    def from_string(cls, raw_str):
        """
        Parses request key from string.

        >>> RequestKey.from_string("itype=upper;ctype=p*")
        RequestKey(...)
        >>> RequestKey.from_string("upper_self")
        RequestKey(...)

        :param str raw_str: Original request string in undersocred or dynamic format

        :raises T <= TagsError: See errors module for details
        :rtype: RequestKey
        """
        try:
            request = cls.__cache[raw_str]
        except TypeError:
            raise ArgumentTypeError(raw_str, type(raw_str), str)
        except KeyError:
            if not isinstance(raw_str, str):
                if isinstance(raw_str, unicode):
                    try:
                        raw_str = raw_str.encode("ascii")
                    except UnicodeEncodeError:
                        raise ArgumentUnicodeError(raw_str)
                else:
                    if not isinstance(raw_str, str):
                        raise ArgumentTypeError(raw_str, type(raw_str), str)

            if "=" in raw_str:
                request = cls.__from_dynamic_string(raw_str)
            else:
                request = cls.__from_underscored_string(raw_str)

            cls.__cache[raw_str] = request
        return request

    def to_dynamic_string(self):
        """
        Formats request key as dynamic string.

        :rtype: str
        """
        return self.__dynamic

    @staticmethod
    def from_dict(tags):
        """Parses request key from dictionary.

        >>> RequestKey.from_dict({"itype": "base", "ctype": ["prod", "test"]})
        RequestKey(...)

        :param dict tags:
        :return: RequestKey
        """
        if not isinstance(tags, dict):
            raise TagsIsNotDict(repr(tags))
        for tag_name, tag_values in tags.iteritems():
            if not isinstance(tag_name, basestring):
                raise TagNameError(repr(tags), repr(tag_name))
            if not isinstance(tag_values, basestring):
                if not isinstance(tag_values, list) or not all(isinstance(t, basestring) for t in tag_values):
                    raise TagValueError(repr(tags), repr(tag_values))

        return RequestKey.from_string(";".join(
            "{}={}".format(tag_name, (",".join(tag_values) if isinstance(tag_values, list) else tag_values))
            for tag_name, tag_values in tags.iteritems()
        ))

    def to_dict(self):
        """
        Internal format for storing and searching

        >>> list(RequestKey.from_string("upper_prod_imgs-main_sas_tier0").to_dict())
        {'itype': ['upper'], 'ctype': ['prod'], 'geo': ['sas'], 'prj': ['imgs-main'], 'tier': ['tier0']}

        >>> list(RequestKey.from_string("common_self").to_dict())
        {'itype': ['common']}

        :return: {"tagname": [list of tag values]}
        :rtype: dict[str,list[str]]
        """
        tags = {'itype': [self.itype]}
        tags.update((key, value.split(','))  # not so effective, but this method is seldom used
                    for key, value in self.__tags)
        return tags

    @classmethod
    def __from_dynamic_string(cls, raw_str):
        itype = None
        tags = []
        tag_names = set()

        raw_pairs = raw_str.split(";")
        for raw_pair in raw_pairs:
            tag_name, separator, raw_tag_values = raw_pair.strip().partition("=")
            if not separator:
                raise MissingTagNameTagValuesSeparator(raw_str)

            if not is_correct_tag_name(tag_name):
                if not tag_name:
                    raise EmptyTagName(raw_str)
                raise TagNameError(raw_str, tag_name)

            raw_tag_values = [tag_value.strip() for tag_value in raw_tag_values.split(",")]

            if tag_name == "itype":
                if not raw_tag_values[0]:
                    raise EmptyItype(raw_str)
                elif len(raw_tag_values) != 1:
                    raise MultipleItypeValues(raw_str, raw_tag_values)
                if itype:
                    raise ItypeMeetsTwice(raw_str)
                itype = raw_tag_values[0]
                if not is_correct_request_tag_value(itype):
                    raise ItypeError(raw_str, itype)
            else:
                tag_values = cls.__normalize_and_verify_tag_values(tag_name, raw_tag_values, raw_str)
                if tag_values is not None:
                    bisect.insort(tags, (tag_name, tag_values))

                if tag_name in tag_names:
                    raise TagMeetsTwice(raw_str, tag_name)
                tag_names.add(tag_name)

        if itype is None:
            raise MissingItype(raw_str)

        return cls(itype, tags)

    def has_tag_bound(self, tag_name):
        """
        Checks if request key contain any bounds with `tag_name`.

        >>> RequestKey.from_string("itype=upper;ctype=p*").has_tag_bound("prj")
        False
        >>> RequestKey.from_string("itype=upper;ctype=p*").has_tag_bound("ctype")
        True

        :type tag_name: str
        :rtype: bool
        """
        return tag_name in self.__tag_names

    @staticmethod
    def __normalize_and_verify_tag_values(tag_name, tag_values_list, raw_str):
        if tag_name == "tier" and 'self' in tag_values_list:
            return None

        processed_tag_values = []
        for tag_value in tag_values_list:
            if not is_correct_request_tag_value(tag_value):
                if not tag_value:
                    raise EmptyTagValue(raw_str, tag_name)
                raise TagValueError(raw_str, tag_value)

            if is_skippable_tag_value(tag_value):  # skip useless filter
                return None
            else:
                bisect.insort(processed_tag_values, tag_value)

        if len(processed_tag_values) == 1:
            return processed_tag_values[0]
        else:
            return concatenate_tag_values(processed_tag_values)

    @classmethod
    def __from_underscored_string(cls, raw_str):
        itype, tag_set, iter_tag_values = parse_underscored_tags(raw_str)

        tags = []

        for tag_name, tag_value in iter_tag_values:
            if tag_name == "tier" and tag_value == "self":
                break
            else:
                if not is_skippable_tag_value(tag_value):  # skip useless filter
                    bisect.insort(tags, (tag_name, tag_value))

        return cls(itype, tags)

    @cached_property
    def __dynamic(self):
        tags = ";".join("{0}={1}".format(key, value) for key, value in self.__tags)
        return "itype={0}{1}".format(self.itype, ";{0}".format(tags) if tags else "")

    def __hash__(self):
        return self.__dynamic.__hash__()

    def __eq__(self, other):
        return type(self) is type(other) and self.__dynamic == other.__dynamic

    def __repr__(self):
        return "RequestKey(itype={0!r}, tags={1!r})".format(self.itype, self.__tags)
