from infra.yasm.unistat import Unistat, SuffixType
from infra.yasm.gateway.lib.util.stat import TIMING_INTERVALS_LOG_1_3
from library.python.monlib.metric_registry import MetricRegistry, HistogramType
from library.python.monlib.metric import Rate, Histogram
from library.python.monlib.encoder import dumps, Compression


TIMING_EXPONENTIAL_BUCKETS = {"bucket_count": 20, "base": 2, "scale": 1}


class Duostat(object):
    def __init__(self):
        self.unistat = Unistat()
        self.registry = MetricRegistry()

        # Unistat push is untyped, only signal name is supplied
        # MetricRegistry does not support get() method, so keep another registry for that
        self._another_registry = {}

    @staticmethod
    def _make_key(name, tags=(), **kwargs):
        if tags:
            return Unistat._format_tags(name, tags)
        return name

    @staticmethod
    def _make_labels(name, tags=(), **kwargs):
        labels = {"sensor": name}
        if tags:
            labels.update(tags)
        return labels

    def create_float(self, name, **kwargs):
        self.unistat.create_float(name, **kwargs)
        suffix = kwargs.get("suffix", SuffixType.Sum)
        if suffix != SuffixType.Sum:
            raise NotImplementedError("Only summ metrics are supported in duostat, got " + str(suffix))
        key = Duostat._make_key(name, **kwargs)
        self._another_registry[key] = self.registry.rate(Duostat._make_labels(name, **kwargs)), False

    def create_histogram(self, name, **kwargs):
        self.unistat.create_histogram(name, **kwargs)
        key = Duostat._make_key(name, **kwargs)
        intervals = kwargs.get("intervals")
        # XXX: so hack. The only excuse is zero code modification
        is_timing = intervals is TIMING_INTERVALS_LOG_1_3
        if is_timing:
            metric = self.registry.histogram_rate(Duostat._make_labels(name + "_ms", **kwargs),
                                                  HistogramType.Exponential, **TIMING_EXPONENTIAL_BUCKETS)
            self._another_registry[key] = metric, True
        else:
            if intervals[0] == 0:
                intervals = intervals[1:]
            metric = self.registry.histogram_rate(Duostat._make_labels(name, **kwargs),
                                                  HistogramType.Explicit, buckets=intervals)
            self._another_registry[key] = metric, False

    def push(self, name, value, **kwargs):
        self.unistat.push(name, value)
        key = Duostat._make_key(name, **kwargs)
        maybe_metric = self._another_registry.get(key)
        if maybe_metric is None:
            return
        metric, is_timing = maybe_metric
        if isinstance(metric, Rate):
            metric.add(value)
        elif isinstance(metric, Histogram):
            if is_timing:
                value = int(1000 * value)
            else:
                value = int(value)
            metric.collect(value)

    def to_json(self, **kwargs):
        return self.unistat.to_json(**kwargs)

    def solomon_spack(self):
        return dumps(self.registry, format='spack', compression=Compression.Lz4)
