#include "chunk_cleaner.h"
#include "histdb_cleaner.h"

#include <library/cpp/logger/global/global.h>

#include <util/folder/path.h>
#include <util/folder/filelist.h>
#include <util/generic/cast.h>

#include <library/cpp/threading/local_executor/local_executor.h>

using namespace NCleaner;

THistdbCleaner::THistdbCleaner(
    const TString& histdbPath, const TStringBuf groupRegex, const TStringBuf chunkRegex,
    const TVector<TString>& prefixes, const TString& excludedFile, int threadCount, bool backup)
    : Filter(prefixes, excludedFile, backup)
    , HistdbPath(histdbPath)
    , GroupRegex(groupRegex)
    , ChunkRegex(chunkRegex)
    , ThreadCount(threadCount)
{
}

void THistdbCleaner::Start() {
    TFsPath histdbDir(HistdbPath);
    TVector<TString> chunks{};
    TDirsList groups;
    TFileList headers;
    groups.Fill(histdbDir);

    while (const TStringBuf histdbSubdir = groups.Next()) {
        if (AnyOf(histdbSubdir, [](char a) { return IsAsciiLower(a); })) {
            // is host
            continue;
        };

        if (!RE2::PartialMatch(re2::StringPiece(histdbSubdir), GroupRegex)) {
            continue;
        }

        TFsPath groupDir = histdbDir / histdbSubdir;
        headers.Fill(groupDir, "", Filter.GetHeaderSuffix(), 1);

        while (TStringBuf headerBuf = headers.Next()) {
            headerBuf.RNextTok(".");
            if (RE2::PartialMatch(re2::StringPiece(headerBuf), ChunkRegex)) {
                TFsPath targetChunk = groupDir / headerBuf;
                chunks.emplace_back(targetChunk.GetPath());
            }
        }
    }
    INFO_LOG << "Chunks for cleaning:" << Endl;
    for (const auto& chunk: chunks) {
        INFO_LOG << "+ " << chunk << Endl;
    }

    auto& executor = NPar::LocalExecutor();
    executor.RunAdditionalThreads(ThreadCount - 1);
    executor.ExecRange(
        [&](int chunkPosition) {
            TString& targetChunk = chunks[chunkPosition];
            INFO_LOG << "Starting " << targetChunk << " cleaning (" << chunkPosition << "/" << chunks.size() << ")" << Endl;
            try {
                Filter.Start(targetChunk);
                INFO_LOG << targetChunk << " cleaning complete (" << chunkPosition << "/" << chunks.size() << ")" << Endl;
            } catch (std::exception& e) {
                INFO_LOG << e.what() << Endl;
                INFO_LOG << "Chunk " << targetChunk << " cleaning was failed, skip it." << Endl;
            }
        },
        NPar::TLocalExecutor::TExecRangeParams(0, static_cast<int>(chunks.size())),
        NPar::TLocalExecutor::WAIT_COMPLETE | NPar::TLocalExecutor::LOW_PRIORITY);
    INFO_LOG << "Done" << Endl;
}
