#include "chunk_cleaner.h"
#include "histdb_cleaner.h"

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/logger/global/global.h>

int main(int argc, const char** argv) {
    using namespace NLastGetopt;
    TOpts options{TOpts::Default()};

    InitGlobalLog2Console();

    TString optHeaderPath;
    options
        .AddCharOption('c', "--  path to chunk (without '.header' or '.data')")
        .AddLongName("chunk")
        .RequiredArgument("CHUNK")
        .StoreResult(&optHeaderPath);

    TString optHistdbPath;
    options
        .AddCharOption('h', "--  histdb path, all groups will be cleaned")
        .AddLongName("histdb")
        .RequiredArgument("DIR")
        .StoreResult(&optHistdbPath);

    int optThreadCount;
    options
        .AddCharOption('t', "--  thread count")
        .AddLongName("threads")
        .DefaultValue(4)
        .RequiredArgument("COUNT")
        .StoreResult(&optThreadCount);

    TString optGroupRegex;
    options
        .AddCharOption('g', "--  group name regex")
        .AddLongName("group-name-regex")
        .DefaultValue(".*")
        .RequiredArgument("REGEX")
        .StoreResult(&optGroupRegex);

    TString optChunkRegex;
    options
        .AddCharOption('r', "--  chunk name regex")
        .AddLongName("chunk-name-regex")
        .DefaultValue(".*")
        .RequiredArgument("REGEX")
        .StoreResult(&optChunkRegex);

    TVector<TString> optSignalPrefixes;
    options
        .AddCharOption('f', "--  signals prefix for filter")
        .AddLongName("filter-signal-prefix")
        .Required()
        .RequiredArgument("PREFIXES")
        .SplitHandler(&optSignalPrefixes, ',');

    TString optExcludedPrefixesFile;
    options
        .AddCharOption('e', "-- excluded prefixes")
        .AddLongName("excluded-prefix")
        .Optional()
        .RequiredArgument("EXCLUDED")
        .StoreResult(&optExcludedPrefixesFile);

    bool optBackup;
    options
        .AddCharOption('b', "--  create backup")
        .AddLongName("backup")
        .NoArgument()
        .Optional()
        .SetFlag(&optBackup);

    TOptsParseResult optParsing(&options, argc, argv);

    try {
        if (optHeaderPath) {
            NCleaner::TFilter cleaner{optSignalPrefixes, optExcludedPrefixesFile, optBackup};
            cleaner.Start(optHeaderPath);
        } else if (optHistdbPath) {
            if (optThreadCount < 1) {
                Cerr << "Thread count should be 1 or bigger";
                return EXIT_FAILURE;
            }
            NCleaner::THistdbCleaner histdbCleaner {
                optHistdbPath,
                optGroupRegex,
                optChunkRegex,
                optSignalPrefixes,
                optExcludedPrefixesFile,
                optThreadCount,
                optBackup,
            };
            histdbCleaner.Start();
        } else {
            Cerr << "You shoud use '--chunk' or '--histdb' option";
            return EXIT_FAILURE;
        }
    } catch (std::exception& e) {
        Cerr << e.what();
        return EXIT_FAILURE;
    } catch (...) {
        Cerr << "Unknown exception";
        return EXIT_FAILURE;
    }
}
