#include "implementation.h"
#include "metrics.h"

#include <infra/yasm/histdb/components/placements/second.h>
#include <infra/yasm/histdb/components/stockpile/stockpile_cleaner.h>
#include <infra/yasm/histdb/dumper/lib/stockpile_pipeline.h>

#include <infra/monitoring/common/perf.h>

#include <util/string/builder.h>

#include <library/cpp/unistat/pusher/pusher.h>

namespace {
    using namespace NHistDb;
    using namespace NMonitoring;
    using namespace NYasm::NCommon;
    using TAggregationRules = NZoom::NAggregators::TAggregationRules;
    using TCommonRules = NZoom::NAggregators::TCommonRules;
    using THostName = NZoom::NHost::THostName;
    using TInstanceKey = NTags::TInstanceKey;
    using TSignalName = NZoom::NSignal::TSignalName;
    using TValue = NZoom::NValue::TValue;

    class THoleInitializer {
    public:
        THoleInitializer(TUnistat& creator)
            : Creator(creator)
            , Priority(NUnistat::TPriority(10))
        {
        }

        void DrillFloatHole(const TString& name, const TString& suffix = "mhhh") {
            Creator.DrillFloatHole(name, suffix, Priority);
        }

        void DrillMaxFloatHole(const TString& name, const TString& suffix = "xhhh") {
            Creator.DrillFloatHole(name, suffix, Priority, NUnistat::TStartValue(0), EAggregationType::Max);
        }
    private:
        TUnistat& Creator;
        NUnistat::TPriority Priority;
    };

    void StartUnistatPusher(const TVector<TString>& tags) {
        auto initializer = [](TUnistat& creator) {
            THoleInitializer hi(creator);

            hi.DrillFloatHole(NMetrics::UPLOAD_CLEAR_KEY);
            hi.DrillMaxFloatHole(NMetrics::UPLOAD_CLEAR_PERC);

            hi.DrillFloatHole(NMetrics::UPLOAD_HOST_DATA_TIME);
            hi.DrillFloatHole(NMetrics::UPLOAD_PERIOD_DATA_TIME);
            hi.DrillFloatHole(NMetrics::UPLOAD_CHUNK_DATA_TIME);
            hi.DrillFloatHole(NMetrics::UPLOAD_CHUNKS_COUNT);

            hi.DrillFloatHole(NMetrics::UPLOAD_IGNORED_KEYS);
            hi.DrillFloatHole(NMetrics::UPLOAD_UNAGGREGATED_KEYS);
            hi.DrillFloatHole(NMetrics::UPLOAD_TOTAL_KEYS);

            hi.DrillFloatHole(NMetrics::UPLOAD_IGNORED_ROWS);
            hi.DrillFloatHole(NMetrics::UPLOAD_UNAGGREGATED_ROWS);
            hi.DrillFloatHole(NMetrics::UPLOAD_DUMPED_ROWS);

            hi.DrillMaxFloatHole(NMetrics::UPLOAD_STAT_PERC);
            hi.DrillMaxFloatHole(NMetrics::UPLOAD_DUMPED_BYTES, "xmmx");
            hi.DrillMaxFloatHole(NMetrics::UPLOAD_TOTAL_BYTES, "xmmx");
        };

        TUnistatPusher::TOptions options;
        options.ParseTags(JoinSeq(TStringBuf(";"), tags));
        TUnistatPusher::Instance().TUnistatPusher::Instance().Start(initializer, options);
    }

    void StopUnistatPusher() {
        TUnistatPusher::Instance().Stop();
    }

    class TTestVisitor : public ISnapshotVisitor{
    public:
        void OnRecord(const IRecordDescriptor&) override {};

        TMaybe<TInstant> GetLastTime() override {
            return Nothing();
        };
    };
}

void NHistDb::PushSignal(const TString& hole, double signal) {
    TUnistatPusher::Instance().PushSignalUnsafe(hole, signal);
    TUnistat::Instance().PushSignalUnsafe(hole, signal);
}

NHistDb::TUploader::TUploader(TLog& logger, const THistoryUploaderSettings& settings)
    : TBaseApplication(logger, settings)
    , Root(settings.GetRoot())
    , StockpileSettings(settings.GetStockpileSettings())
    , WriteMode(!settings.GetReadOnly())
    , ThreadsCount(settings.GetThreadsCount())
    , StatePath(settings.GetStatePath())
    , Begin(settings.GetBegin())
    , End(settings.GetEnd())
    , TestPeriod(settings.GetTestPeriod())
    , FilterItype(settings.GetTestItype())
    , FastConfigSettings(MakeYasmFastConfigSettings("fast_config.json"))
    , FastConfig(Logger, FastConfigSettings)
    , StockpileState(StockpileLogger, StockpileSettings, FastConfig)
{
    auto backend = CreateRotatingLogBackend(settings.GetLogFileStockpile(), TLOG_INFO, true);
    StockpileLogger.ResetBackend(std::move(backend));
    StockpileLogger.SetFormatter(SimpleLogFormatter);
    NStockpile::TGrpcSettings::Init("yasmuploader", logger);
    if (settings.GetTags()) {
        Tags.emplace_back(settings.GetTags());
    }

    Tags.emplace_back(TString("prj=yasmuploader"));
    Tags.emplace_back(TString("itype=yasmuploader"));
    TCommonRules commonRules;
    NZoom::NAggregators::FillRules(commonRules, AggregationRules, settings.GetRulesPath());
}

void NHistDb::TUploader::FillParser(NLastGetopt::TOpts& options) {
    THistoryUploaderSettings::FillParser(options);
}

void NHistDb::TUploader::ProcessParsedOptions(const NLastGetopt::TOptsParseResult& parsed, THistoryUploaderSettings& settings) {
    settings.ProcessParsedOptions(parsed);
}

THolder<ISnapshotVisitor> TUploader::CreateVisitor() {
    if (WriteMode) {
        TStockpileWriterSettings writerSettings;
        writerSettings.UnavailableMetabaseShardDropRecordsDelay.Clear(); // don't drop anything
        auto pipeline = MakeHolder<TWritingStockpilePipeline>(StockpileLogger, ThreadsCount, StockpileSettings, FastConfigSettings, writerSettings);
        pipeline->Start();
        return pipeline;
    } else {
        return MakeHolder<TTestVisitor>();
    }
}

void NHistDb::TUploader::DumpPeriodData(const THostName& host, const TRecordPeriod& period) {
    TInstant endTime = GetEndTime(host);

    auto times = GetChunkTimes(host.GetName(), period);
    if (times.empty()) {
        Logger << ELogPriority::TLOG_INFO << "No chunks for " << host <<  " " << period.GetTargetPrefix();
        return;
    }

    TStringBuilder startLog;
    startLog << "Dump period data for " << host.GetName() << " " << period.GetTargetPrefix() << " " << times.size() << "chunks";
    TPushMeasuredMethod perf(Logger, startLog, NMetrics::UPLOAD_PERIOD_DATA_TIME);

    Logger << ELogPriority::TLOG_INFO << "For prefix " << period.GetTargetPrefix() << " " << times.size() << " chunks";

    for (auto timeID : xrange(times.size())) {
        auto startTime = times[times.size() - timeID - 1];

        TString timeInfo = TStringBuilder() << "("<< host << ", " << period.GetTargetPrefix() << ", " << startTime.Seconds() << ")";

        auto visitor = CreateVisitor();

        try {
            TPushMeasuredMethod perf(Logger, "Dump chunk data for " + timeInfo, NMetrics::UPLOAD_CHUNK_DATA_TIME);
            size_t perc = (50 * timeID / times.size()) * 2;
            Logger << ELogPriority::TLOG_INFO << "Process " << timeInfo << " " << perc << "%";

            SendChunk(Logger, AggregationRules, Root, host, startTime, endTime, period, KnownItypes,
                             *visitor.Get(), FilterItype);

            if (UseStateFile()) {
                State().ProcessTime(host, startTime);
            }
        } catch (const UnexpectedInsanceKey& e) {
            throw;
        } catch (const yexception& e) {
            Logger << ELogPriority::TLOG_ERR << "chunk " << timeInfo << " have failed with [" << e.what() << "]";
            return;
        }

        AtomicAdd(DumpedSize, GetChunkSize(host, period, startTime));
        auto dumped = AtomicGet(DumpedSize);
        Logger << ELogPriority::TLOG_INFO << "Dumped " << dumped << " from " << TotalSize << " " << int(100 * dumped / TotalSize) << "%";

        if (WriteMode) {
            auto pipeline = static_cast<TWritingStockpilePipeline*>(visitor.Get());
            pipeline->Finish();
        }
    }
}

void NHistDb::TUploader::DumpHostData(const THostName& host) {
    TPushMeasuredMethod perf(Logger, "Dump data for " + host.GetName(), NMetrics::UPLOAD_HOST_DATA_TIME);
    Logger << ELogPriority::TLOG_INFO << "Start dump " << host.GetName();
    if (TestPeriod.Defined()) {
        DumpPeriodData(host, TRecordPeriod::Get(TestPeriod.GetRef()));
    } else {
        if (UseStateFile() && State().IsFinished(host.GetName())) {
            Logger << ELogPriority::TLOG_INFO << "Skip host data" << host.GetName();
            AtomicAdd(DumpedSize, GetHostSize(host));
            return;
        }
        DumpPeriodData(host, TRecordPeriod::Get("m5"));
        DumpPeriodData(host, TRecordPeriod::Get("h"));
        if (UseStateFile()) {
            State().Finish(host.GetName());
        }
    }
}

size_t  NHistDb::TUploader::GetChunkSize(const THostName& host, const TRecordPeriod& period, TInstant startTime) const {
    auto chunkPath = TSecondPlacementReader::GetDataPath(Root, host.GetName(), period, startTime);
    TFileStat stat;
    chunkPath.Stat(stat);
    return stat.Size;
}

void  NHistDb::TUploader::SendStatImpl() const {
    TAutoEvent event;
    while (!AtomicGet(Done)) {
        auto dumped = AtomicGet(DumpedSize);
        if (TotalSize) {
            PushSignal(NMetrics::UPLOAD_STAT_PERC, 100 * dumped / TotalSize);
        }
        PushSignal(NMetrics::UPLOAD_DUMPED_BYTES, dumped);
        PushSignal(NMetrics::UPLOAD_TOTAL_BYTES, TotalSize);
        auto totalShards = AtomicGet(TotalShards);
        if (totalShards) {
            PushSignal(NMetrics::UPLOAD_CLEAR_PERC, 100 * AtomicGet(ClearShards) / totalShards);
        }
        event.WaitT(TDuration::Seconds(1));
    }
}
void*  NHistDb::TUploader::SendStat(void* data) noexcept{
    TUploader* self = static_cast<TUploader*>(data);
    self->SendStatImpl();
    return nullptr;
}

bool NHistDb::TUploader::UseStateFile() const {
    return StatePath.Defined() && !TestPeriod.Defined() && !FilterItype.Defined() && WriteMode && !Begin.Defined();
}
TProtoUploaderState TUploader::State() const {
    return TProtoUploaderState(StatePath.GetRef());
}


TVector<TInstant> NHistDb::TUploader::GetChunkTimes(const THostName& host, const TRecordPeriod& period) const {
    auto times = TSecondPlacementReader::GetChunkTimes(Root, host.GetName(), period);
    if (Begin.Defined()) {
        EraseIf(times, [&](auto time){ return time + period.GetInterval() < Begin.GetRef(); });
    }
    auto stopTime = GetEndTime(host);
    EraseIf(times, [&](auto time){ return time > stopTime; });
    return times;
}

size_t  NHistDb::TUploader::GetPeriodSize(const THostName& host, const TRecordPeriod& period) const {
    auto times = GetChunkTimes(host.GetName(), period);
    size_t totalSize = 0;

    for (auto startTime : times) {
        totalSize += GetChunkSize(host, period, startTime);
    }
    return totalSize;
}

size_t NHistDb::TUploader::GetHostSize(const THostName& host) const {
    if (TestPeriod.Defined()) {
        return GetPeriodSize(host, TRecordPeriod::Get(TestPeriod.GetRef()));
    } else {
        return GetPeriodSize(host, TRecordPeriod::Get("m5")) + GetPeriodSize(host, TRecordPeriod::Get("h"));
    }
}

void NHistDb::TUploader::Prepare() {
    TotalSize = 0;
    AtomicSet(DumpedSize, 0);
    AtomicSet(TotalShards, 0);
    AtomicSet(ClearShards, 0);
    for (const auto& hostName : TSecondPlacementReader::GetHostNames(Root)) {
        THostName group(hostName);
        if (!group.IsGroup()) {
            continue;
        }
        GroupNames.push_back(group);
        TotalSize += GetHostSize(group);
    }
    Logger << ELogPriority::TLOG_INFO << "Total size for dump " << TotalSize << " in " << GroupNames.size() << " groups.";
}

TInstant NHistDb::TUploader::GetEndTime(const THostName& host) const {
    if (UseStateFile()) {
        auto processedTime = State().GetDumpedTime(host);
        return Min(End, processedTime);
    }
    return End;
}

void NHistDb::TUploader::Clear(const THostName& host) {
    TPushMeasuredMethod perf(Logger, "Clear " + host.GetName());

    Logger << ELogPriority::TLOG_INFO << "Start Clear " << host.GetName() << Endl;
    StockpileDataCleaner cleaner(StockpileLogger, StockpileState);

    auto shardKeys = StockpileState.GetMetabaseShardKeys(false);
    AtomicSet(ClearShards, 0);
    AtomicSet(TotalShards, shardKeys.size());

    TInstant clearUntil = GetEndTime(host);

    for (auto shardKey : shardKeys) {
        try {
            cleaner.ClearStockpileData(host, shardKey, clearUntil);
        } catch (const yexception& e) {
            Logger << ELogPriority::TLOG_ERR << "clear on " << shardKey << " failed with " << e.what();
        } catch (...) {
            Logger << ELogPriority::TLOG_ERR << "internal exception catch";
        }

        AtomicAdd(ClearShards, 1);
    }
    AtomicSet(TotalShards, 0);

    if (UseStateFile()) {
        State().ProcessTime(host, clearUntil);
    }
}

void NHistDb::TUploader::Run() {
    HttpServer.StartServing();

    if (UseStateFile() && State().IsFinished()) {
        Logger << ELogPriority::TLOG_INFO << "Everything is already dumped";
        return;
    }

    Logger << ELogPriority::TLOG_INFO << "Start";

    StartUnistatPusher(Tags);

    Prepare();

    TThread thread(SendStat, this);
    AtomicSet(Done, 0);
    thread.Start();

    StockpileState.UpdateShards();
    StockpileState.Start();

    KnownItypes = StockpileState.GetItypes();

    Logger << ELogPriority::TLOG_INFO << "Dump data until " << End;

    bool hasFinished = false;
    try {
        for (const auto& group : GroupNames) {
            Clear(group);
            DumpHostData(group);
        }
        hasFinished = true;
    } catch (const yexception& e) {
        Logger << ELogPriority::TLOG_CRIT << "Uploading failed with: " << e.what();
    }

    if (hasFinished && UseStateFile()) {
        State().Finish();
    }

    AtomicSet(Done, 1);
    StopUnistatPusher();
    StockpileState.Stop();
    Logger << ELogPriority::TLOG_INFO << "Finish";
}
