#pragma once

#include "settings.h"
#include "uploader_state.h"
#include "metrics.h"

#include <infra/yasm/stockpile_client/common/base_types.h>
#include <infra/yasm/histdb/dumper/lib/stockpile_pipeline.h>
#include <infra/yasm/zoom/components/aggregators/preaggregates.h>
#include <infra/yasm/common/labels/host/host.h>

#include <infra/monitoring/common/application.h>
#include <infra/monitoring/common/settings.h>

namespace NHistDb {
    class UnexpectedInsanceKey : public yexception {};

    void SendChunk(
        TLog& logger,
        const NZoom::NAggregators::TAggregationRules& rules,
        const TFsPath& root,
        const NZoom::NHost::THostName& host,
        TInstant startTime,
        TInstant endTime,
        const TRecordPeriod& period,
        const THashSet<TString>& knownItypes,
        ISnapshotVisitor& pipeline,
        TMaybe<TString> itype = Nothing()
    );
    void PushSignal(const TString& hole, double signal);

    class TUploader final: public NMonitoring::TBaseApplication<THistoryUploaderSettings> {
    public:
        TUploader(TLog& logger, const THistoryUploaderSettings& settings);
        void Run() override;
        static void FillParser(NLastGetopt::TOpts& options);
        static void ProcessParsedOptions(const NLastGetopt::TOptsParseResult& parsed, THistoryUploaderSettings& settings);

        static const THashMap<TString, TString> INSTANCE_KEY_HOST_RENAMING_WHITELIST;
    private:
        TInstant GetEndTime(const NZoom::NHost::THostName& host) const;
        void Clear(const NZoom::NHost::THostName& host);
        void DumpHostData(const NZoom::NHost::THostName& host);
        THolder<ISnapshotVisitor> CreateVisitor();
        void DumpPeriodData(const NZoom::NHost::THostName& host, const TRecordPeriod& period);


        static void* SendStat(void* self) noexcept;
        void SendStatImpl() const;

        void Prepare();
        TVector<TInstant> GetChunkTimes(const NZoom::NHost::THostName& host, const TRecordPeriod& period) const;
        size_t GetHostSize(const NZoom::NHost::THostName& host) const;
        size_t GetPeriodSize(const NZoom::NHost::THostName& host, const TRecordPeriod& period) const;
        size_t GetChunkSize(const NZoom::NHost::THostName& host, const TRecordPeriod& period, TInstant startTime) const;

        bool UseStateFile() const;

        TProtoUploaderState State() const;

        THolder<NMonitoring::TStatsInitializer> GetStatsInitializer() override {
            return MakeHolder<NHistDb::NMetrics::THistoryUploaderStatsInitializer>();
        }
    private:
        TLog StockpileLogger;
        TVector<TString> Tags;
        const TFsPath Root;
        const NStockpile::TSettings StockpileSettings;
        const bool WriteMode;
        NZoom::NAggregators::TAggregationRules AggregationRules;
        const size_t ThreadsCount;
        TMaybe<TFsPath> StatePath;
        TMaybe<TInstant> Begin;
        TInstant End;

        TMaybe<TString> TestPeriod;
        TMaybe<TString> FilterItype;
        NYasm::NCommon::TFastConfigSettings FastConfigSettings;
        NYasm::NCommon::TFastConfig FastConfig;
        NStockpile::TStockpileState StockpileState;

        THashSet<TString> KnownItypes;
        TVector<NZoom::NHost::THostName> GroupNames;

        TAtomic Done;

        TAtomic ClearShards;
        TAtomic TotalShards;

        TAtomic DumpedSize;
        size_t TotalSize;
    };

    class TInstanceKeyProcessor {
    public:
        static NTags::TInstanceKey FixInstanceKey(const NTags::TInstanceKey& instanceKey, NZoom::NHost::THostName host, const THashMap<TString, TString>& whitelist) {
            NTags::TInstanceKey fixedInstanceKey;
            if (instanceKey.GetGroupName().Empty()) {
                if (!instanceKey.GetHostName().Empty()) {
                    throw UnexpectedInsanceKey() << "Instance key has host field: " << instanceKey.ToNamed();
                }
                fixedInstanceKey = instanceKey.SetGroupAndHost(host, NZoom::NHost::THostName());
            } else if  (instanceKey.GetGroupName() == host) {
                fixedInstanceKey = instanceKey;
            } else {
                THashMap<TString, TString>::const_iterator expectedGroupNameIt = whitelist.find(instanceKey.GetGroupName().GetName());

                if (expectedGroupNameIt.IsEnd() || expectedGroupNameIt->second != host.GetName()) {
                    throw UnexpectedInsanceKey() << "Shard renaming is not in whitelist. Shard name: " << host.GetName()
                                                 << ", instance key group: " << instanceKey.GetGroupName().GetName();
                }
                fixedInstanceKey = instanceKey.SetGroupAndHost(host, NZoom::NHost::THostName());
            }
            return fixedInstanceKey;
        }
    };
}
