#include "implementation.h"
#include "metrics.h"

#include <infra/yasm/histdb/components/placements/second.h>

#include <infra/monitoring/common/perf.h>


namespace {
    using namespace NHistDb;
    using namespace NMonitoring;
    using namespace NYasm::NCommon;
    using TAggregationRules = NZoom::NAggregators::TAggregationRules;
    using THostName = NZoom::NHost::THostName;
    using TInstanceKey = NTags::TInstanceKey;
    using TSignalName = NZoom::NSignal::TSignalName;
    using TValue = NZoom::NValue::TValue;
    using TSplittedKeysMap = THashMap<TString, THashMap<size_t, THashSet<TInstanceKey>>>;

    const THashSet<TString> IgnoredItypes = {
            "nirvanajob",
    };

    class TRecordDescriptor : public IRecordDescriptor {
    public:
        TRecordDescriptor(
                THostName host,
                TInstanceKey instanceKey,
                TSignalName signal,
                TInstant start,
                TInstant end,
                const TValue& value
        )
                : Host(host)
                , InstanceKey(instanceKey)
                , Signal(signal)
                , Start(start)
                , End(end)
                , Value(value.GetValue())
        {
            //TODO init FlushOffset, Data, Kind if it necessary
        }

        THostName GetHostName() const override {
            return Host;
        }

        TInstanceKey GetInstanceKey() const override {
            return InstanceKey;
        }

        TSignalName GetSignalName() const override {
            return Signal;
        }

        size_t GetValuesCount() const override {
            return 1;
        }

        TInstant GetStartTime() const override {
            return Start;
        }

        TInstant GetEndTime() const override {
            return End;
        }

        TInstant GetFlushOffset() const override {
            return Start;
        }

        NYasmServer::ESeriesKind GetSeriesKind() const override {
            ythrow yexception() << "Kind was not defined";
        };

        const TString& GetData() const override {
            ythrow yexception() << "Data was not defined";
        };

        THolder<IRecordDescriptor> Clone() const override {
            return MakeHolder<TRecordDescriptor>(Host, InstanceKey, Signal, Start, End, Value);
        }

        THolder<IRecordDescriptor> CloneWithNewKey(const TInstanceKey& newKey) const override {
            return MakeHolder<TRecordDescriptor>(Host, newKey, Signal, Start, End, Value);
        }

        void Iterate(IRecordVisitor& visitor) const override {
            visitor.OnValue(Value.GetValue());
        }

        bool operator < (const TRecordDescriptor& other) const noexcept {
            return GetStartTime() < other.GetStartTime();
        }

    private:
        THostName Host;
        TInstanceKey InstanceKey;
        TSignalName Signal;
        TInstant Start;
        TInstant End;
        TValue Value;
    };

    class THistoryDataDumper {
    public:
        THistoryDataDumper(
                TLog& logger,
                const TAggregationRules& rules,
                const TFsPath& root,
                const THostName& host,
                TInstant startTime,
                TInstant endTime,
                const TRecordPeriod& period,
                const THashSet<TString>& knownItypes,
                TMaybe<TString> itype,
                ISnapshotVisitor& pipeline
        )
                : Logger(logger)
                , Rules(rules)
                , Root(root)
                , Host(host)
                , StartTime(startTime)
                , EndTime(endTime)
                , Period(period)
                , HeaderCache(1)
                , Pipeline(pipeline)
                , KnownItypes(knownItypes)
                , FilterItype(itype)
                , TotalRecords(0)
                , IgnoredRows(0)
                , UnAggregatedRows(0)
                , DumpedRows(0)
        {
            Header = HeaderCache.Create(Root, Host.GetName(), Period, StartTime);
        }

        void ReadAndDump() {
            if (!Header) {
                ythrow yexception() << "No header file";
            }
            Prepare();
            SendRecords();
            Finish();
        }

    private:
        void ConvertRow(TSomethingFormat::TIteratorRow& row) {
            auto &[time, instanceKey, record] = row;
            instanceKey = FixedInstanceKeys.at(instanceKey);

            TVector<TInstanceKey> keys;
            if (!Ignored.contains(instanceKey)) {
                keys.push_back(instanceKey);
            }

            if (UnAggregatedKeys.contains(instanceKey)) {
                UnAggregatedRows += 1;
                keys.push_back(UnAggregatedKeys.at(instanceKey));
            }
            if (keys.empty()) {
                IgnoredRows += 1;
                return;
            }
            DumpedRows += 1;

            auto start = Period.FromOffset(StartTime, time);

            TotalRecords += record.GetValues().size();
            for (auto key: keys) {
                for (auto &[signal, value] : record.GetValues()) {
                    Pipeline.OnRecord(
                            TRecordDescriptor(Host, key, signal, start, start + Period.GetResolution(), value));
                }
            }
        }

        TVector<TSomethingFormat::TTimestamp> GetTimes() {
            return Header->GetTimes();
        }

        void ProcessKeys(TSecondPlacementReader::TSomethingKeyCallback callback) {
            Header->IterateSomethingKeys(callback);
        }

        void Process(const TVector<TSomethingFormat::TTimestamp>& timestamps, TSecondPlacementReader::TSomethingCallback callback) {
            Header->IterateSomething(timestamps, callback);
        }

        void Finish() {
            PushSignal(NMetrics::UPLOAD_IGNORED_ROWS, IgnoredRows);
            PushSignal(NMetrics::UPLOAD_UNAGGREGATED_ROWS, UnAggregatedRows);
            PushSignal(NMetrics::UPLOAD_DUMPED_ROWS, DumpedRows);
            PushSignal(NMetrics::UPLOAD_CHUNKS_COUNT, TotalRecords);

            Logger << ELogPriority::TLOG_INFO << "Upload host data finished";
            Logger << ELogPriority::TLOG_INFO << "\tDumped rows " << DumpedRows;
            Logger << ELogPriority::TLOG_INFO << "\tSkip rows " << IgnoredRows;
            Logger << ELogPriority::TLOG_INFO << "\tDumped records " << TotalRecords;
        }

        void SendRecords() {
            TPushMeasuredMethod perf(Logger, "SendRecords");
            for (auto time: Times) {
                Process({time}, [&](TSomethingFormat::TIteratorRow& row) {
                    ConvertRow(row);
                });
            }
        }

        void AddIgnored(const THashSet<TInstanceKey>& instances, const THashSet<TInstanceKey>& actual) {
            for (auto instanceKey : instances) {
                auto itype = instanceKey.GetItype();
                if (IgnoredItypes.contains(itype) ||
                    !KnownItypes.contains(itype) ||
                    (FilterItype.Defined() && itype != FilterItype.GetRef())
                        ) {
                    Ignored.insert(instanceKey);
                    continue;
                }

                if (!actual.contains(instanceKey)) {
                    auto aggrs = instanceKey.GetAggregated();
                    if (aggrs.size() != 2 || aggrs[0] != "host" || aggrs[1] != "tier") {
                        Logger << ELogPriority::TLOG_INFO << "Ignore instance key " << instanceKey;
                    }
                    Ignored.insert(instanceKey);
                }
            }
        }

        TInstanceKey SetTier(TInstanceKey instanceKey) {
            auto keyWithTier = instanceKey.SetTag(TString("tier"), TString("none"));
            UnAggregatedKeys[instanceKey] = keyWithTier;
            return keyWithTier;
        }

        THashSet<TInstanceKey> GetMinAggregated(const THashSet<TInstanceKey>& instances) {
            size_t maxTagsCount = 0;
            for (auto key : instances) {
                maxTagsCount = std::max(maxTagsCount, key.GetTags().size());
            }

            THashSet<TInstanceKey> minAggregated;
            for (auto instanceKey : instances) {
                Y_VERIFY(instanceKey.GetGroupName() == Host, "Host name differs (shard was renamed)");
                auto aggr = instanceKey.GetAggregated();
                Y_VERIFY(Find(aggr, TString("host")) != aggr.end());

                if (instanceKey.GetTags().size() == maxTagsCount) {
                    if ((aggr.size() == 2) && (Find(aggr, TString("tier")) != aggr.end())) {
                        minAggregated.insert(SetTier(instanceKey));
                    } else {
                        minAggregated.insert(instanceKey);
                    }
                }
            }
            return minAggregated;
        }

        THashSet<TInstanceKey> AggregateByRules(const THashSet<TInstanceKey>& instances) const {
            THashSet<TInstanceKey> actual;
            for (auto instanceKey : instances) {
                actual.insert(instanceKey);
                for (const auto rule : Rules.GetRules(instanceKey)) {
                    actual.insert(instanceKey.AggregateBy(rule));
                }
            }
            return actual;
        }

        void PrepareInstanceKeys(const THashSet<TInstanceKey>& instances) {
            auto minAggregated = GetMinAggregated(instances);
            auto actual = AggregateByRules(minAggregated);
            AddIgnored(instances, actual);
        }

        void AddInstanceKeyToKeyMap(const TInstanceKey& instanceKey, TSplittedKeysMap& instanceKeysMap) {
            size_t hash = instanceKey.GetTagsAndAggregatedNameSet().Hash();
            instanceKeysMap[instanceKey.GetItype()][hash].insert(instanceKey);
        }

        TSplittedKeysMap SplitInstanceKeys() {
            TSplittedKeysMap instanceKeysMap;

            ProcessKeys([&](TSomethingFormat::TKeyIteratorRow& row) {
                TInstanceKey instanceKey = std::get<0>(row);
                if (FixedInstanceKeys.contains(instanceKey)) {
                    return;
                }
                TInstanceKey fixedInstanceKey = TInstanceKeyProcessor::FixInstanceKey(instanceKey, Host, TUploader::INSTANCE_KEY_HOST_RENAMING_WHITELIST);
                FixedInstanceKeys[instanceKey] = fixedInstanceKey;
                AddInstanceKeyToKeyMap(fixedInstanceKey, instanceKeysMap);
            });

            return instanceKeysMap;
        }

        void Prepare() {
            TPushMeasuredMethod perf(Logger, "Prepare");
            Times = GetTimes();
            EraseIf(Times, [&](auto time){ return Period.FromOffset(StartTime, time) >= EndTime; });

            size_t totalKeys = 0;
            TSplittedKeysMap instanceKeysMap = SplitInstanceKeys();

            for (const auto &[itype, allInstances] : instanceKeysMap) {
                for (const auto &[hash, instances] : allInstances) {
                    PrepareInstanceKeys(instances);
                    totalKeys += instances.size();
                }
            }

            PushSignal(NMetrics::UPLOAD_IGNORED_KEYS, Ignored.size());
            PushSignal(NMetrics::UPLOAD_TOTAL_KEYS, totalKeys);
            PushSignal(NMetrics::UPLOAD_UNAGGREGATED_KEYS, UnAggregatedKeys.size());
            Logger << ELogPriority::TLOG_INFO << "Prepare Finish ";
            Logger << ELogPriority::TLOG_INFO << "\tTotal keys " << totalKeys;
            Logger << ELogPriority::TLOG_INFO << "\tSkip keys " << Ignored.size();
            Logger << ELogPriority::TLOG_INFO << "\tUnaggregated keys " << UnAggregatedKeys.size();
        }

    private:
        TLog& Logger;
        const TAggregationRules& Rules;
        const TFsPath& Root;
        const THostName& Host;
        const TInstant StartTime;
        TInstant EndTime;
        const TRecordPeriod& Period;

        TSecondHeaderCache HeaderCache;
        THolder<TSecondPlacementReader> Header;
        ISnapshotVisitor& Pipeline;

        THashSet<TInstanceKey> Ignored;
        TVector<TSomethingFormat::TTimestamp> Times;
        const THashSet<TString>& KnownItypes;
        TMaybe<TString> FilterItype;

        size_t TotalRecords;
        size_t IgnoredRows;
        size_t UnAggregatedRows;
        size_t DumpedRows;

        THashMap<TInstanceKey, TInstanceKey> FixedInstanceKeys;
        THashMap<TInstanceKey, TInstanceKey> UnAggregatedKeys;
    };
}


void NHistDb::SendChunk(
        TLog& logger,
        const TAggregationRules& rules,
        const TFsPath& root,
        const THostName& host,
        TInstant startTime,
        TInstant endTime,
        const TRecordPeriod& period,
        const THashSet<TString>& knownItypes,
        ISnapshotVisitor& pipeline,
        TMaybe<TString> itype
) {
    THistoryDataDumper dumper(logger, rules, root, host, startTime, endTime, period, knownItypes, itype, pipeline);
    dumper.ReadAndDump();
}
