#include "settings.h"

using namespace NHistDb;

namespace {
    TString HISTDB_OPT("histdb-dir");
    TString RULES_OPT("rules");
    TString READ_ONLY_OPT("read-only");
    TString THREADS_OPT("threads");
    TString TAGS_OPT("tags");
    TString PERIOD_OPT("period");
    TString ITYPE_OPT("itype");
    TString STATE_OPT("state");
    TString END_OPT("end");
    TString BEGIN_OPT("begin");
    TString LOG_FILE_STOCKPILE_OPT("log-file-stockpile");
}

void THistoryUploaderSettings::ProcessParsedOptions(const NLastGetopt::TOptsParseResult& parsed) {
    if (parsed.Has(HISTDB_OPT)) {
        SetRoot(parsed.Get(HISTDB_OPT));
    }

    if (parsed.Has(PERIOD_OPT)) {
        SetPeriod(parsed.Get(PERIOD_OPT));
    }

    if (parsed.Has(ITYPE_OPT)) {
        SetItype(parsed.Get(ITYPE_OPT));
    }

    if (parsed.Has(TAGS_OPT)) {
        SetTags(parsed.Get(TAGS_OPT));
    }

    if (parsed.Has(STATE_OPT)) {
        SetStatePath(parsed.Get(STATE_OPT));
    }

    SetRulesPath(parsed.Get(RULES_OPT));
    SetReadOnly(parsed.Has(READ_ONLY_OPT));
    if (parsed.Has(THREADS_OPT)) {
        SetThreadsCount(parsed.Get<size_t>(THREADS_OPT));
    }

    SetEnd(parsed.Get<size_t>(END_OPT));

    if (parsed.Has(BEGIN_OPT)) {
        SetBegin(parsed.Get<size_t>(BEGIN_OPT));
    }

    SetLogFileStockpile(parsed.Get(LOG_FILE_STOCKPILE_OPT));

    StockpileSettingsFactory.ProcessParsedOptions(parsed);
}

void THistoryUploaderSettings::FillParser(NLastGetopt::TOpts& options) {
    options.AddLongOption(HISTDB_OPT, "directory with histdb").Optional().RequiredArgument();
    options.AddLongOption(ITYPE_OPT, "test selected itype").Optional().RequiredArgument();
    options.AddLongOption(PERIOD_OPT, "test selected period").Optional().RequiredArgument();
    options.AddLongOption(RULES_OPT, "path to rules.conf").RequiredArgument().DefaultValue(TStringBuf("rules.conf"));
    options.AddLongOption(READ_ONLY_OPT, "use for test").AddShortName('r').NoArgument();
    options.AddLongOption(THREADS_OPT, "count of threads for stockpile pipeline").RequiredArgument();
    options.AddLongOption(TAGS_OPT, "additional tags").RequiredArgument().DefaultValue(TStringBuf(""));
    options.AddLongOption(STATE_OPT, "path to state file").Optional().RequiredArgument();
    options.AddLongOption(END_OPT, "last value for dump").Optional().RequiredArgument();
    options.AddLongOption(BEGIN_OPT, "first value for dump").RequiredArgument();
    options.AddLongOption(LOG_FILE_STOCKPILE_OPT, "path to stockpile log file.").Optional().RequiredArgument().DefaultValue(TStringBuf("console"));
    NStockpile::TStockpileSetingsFactory::FillParser(options);
}

void THistoryUploaderSettings::FillFromFileConfig(const NConfig::TConfig& config) {
    TSettings::FillFromFileConfig(config);
    SetRoot(config["histdb_dir"].As<TString>("histdb"));
    StockpileSettingsFactory.FillFromFileConfig(config);
}
