#include "settings.h"
#include "implementation.h"

#include <infra/yasm/histdb/components/dumper/common_ut.h>
#include <infra/yasm/zoom/components/yasmconf/yasmconf.h>
#include <infra/yasm/histdb/dumper/lib/pipeline.h>
#include <util/folder/tempdir.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/env.h>


using namespace NHistDb;
using namespace NZoom::NHgram;
using TAggregationRules = NZoom::NAggregators::TAggregationRules;
using TCommonRules = NZoom::NAggregators::TCommonRules;
using THostName = NZoom::NHost::THostName;
using TInstanceKey = NTags::TInstanceKey;
using TSignalName = NZoom::NSignal::TSignalName;
using TValue = NZoom::NValue::TValue;

namespace {
    class TRecordSaver : public ISnapshotVisitor {
    public:
        void OnRecord(const IRecordDescriptor& recordDescriptor) override {
            Records.emplace_back(recordDescriptor.Clone());
        }

        TMaybe<TInstant> GetLastTime() override {
            return Nothing();
        }

        const TVector<THolder<IRecordDescriptor>>& GetRecords() const {
            return Records;
        }

    private:
        TVector<THolder<IRecordDescriptor>> Records;
    };

    class TTestValue : public NZoom::NRecord::IRecordVisitor {
    public:
        TTestValue(const TValue& value)
            : Value(value)
        {}
        void OnValue(NZoom::NValue::TValueRef value) override {
            UNIT_ASSERT_VALUES_EQUAL(TValue(value), Value);
        }
    private:
        const TValue& Value;
    };
}

void TestTags(const NZoom::NAggregators::TAggregationRules& rules, const TVector<TInstanceKey>& baseKeys, const TVector<TInstanceKey>& expectedKeys) {
    TLog logger;
    const auto yasmConf(NZoom::NYasmConf::TYasmConf::FromString(TStringBuf("{\"conflist\": {\"common\": {\"signals\": {}, \"patterns\": {}, \"periods\": {}}}}")));
    TTempDir root;
    root.Path().MkDir();

    THostName group(TStringBuf("SAS.000"));
    TRecordPeriod period(TRecordPeriod::Get("m5"));
    TInstant startTime(period.GetStartTime(TInstant::Now()));
    TMockedRecordDescriptor baseRecord;
    baseRecord.SetHostName(group.GetName()).SetStartTime(startTime).SetEndTime(startTime).SetFlushOffset(startTime);

    THashSet<TString> itypes;
    for (auto key: baseKeys) {
        itypes.insert(TString{key.GetItype()});
    }

    {
        TWritingPipeline pipeline(logger, yasmConf, root.Name(), 1, startTime - TDuration::Days(1));

        pipeline.Start();

        for (auto key: baseKeys) {
            pipeline.OnRecord(baseRecord.SetInstanceKey(key));
            itypes.insert(TString{key.GetItype()});
        }

        auto flush = startTime + TDuration::Days(2);
        baseRecord.SetStartTime(flush).SetEndTime(flush).SetFlushOffset(flush);
        pipeline.OnRecord(baseRecord);

        pipeline.Stop();
        pipeline.Finish();
    }

    {
        TRecordSaver pipeline;

        for (auto startTime: TSecondPlacementReader::GetChunkTimes(root.Path(), group.GetName(), period)) {
            NHistDb::SendChunk(logger, rules, root.Path(), group, startTime, TInstant::Now(), period, itypes, pipeline);
        }
        THashSet<TInstanceKey> resultKeys;
        for (const auto& record : pipeline.GetRecords()) {
            resultKeys.insert(record->GetInstanceKey());
        }

        UNIT_ASSERT_VALUES_EQUAL(resultKeys.size(), expectedKeys.size());
        for (const auto& key : expectedKeys) {
            UNIT_ASSERT(resultKeys.contains(key));
        }
    }
}

Y_UNIT_TEST_SUITE(TReadToFileAndDumpToPipline) {
    Y_UNIT_TEST(CheckHostFill) {
        NZoom::NAggregators::TAggregationRules rules;

        TVector<TInstanceKey> baseKeys {
            TInstanceKey::FromNamed("host_not_defined|geo=man;ctype=prod;prj=yasm;tier=some"),
            TInstanceKey::FromNamed("host_defined|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),
            TInstanceKey::FromNamed("host_defined_partly|geo=man;ctype=prod;prj=yasm;tier=some"),
            TInstanceKey::FromNamed("host_defined_partly|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),
        };
        TVector<TInstanceKey> expectedKeys {
            TInstanceKey::FromNamed("host_not_defined|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),
            TInstanceKey::FromNamed("host_defined|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),
            TInstanceKey::FromNamed("host_defined_partly|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),
        };
        TestTags(rules, baseKeys, expectedKeys);
    }

    Y_UNIT_TEST(CheckRules) {
        //for itype=rule1 where is key aggregated by only host
        //for itype=rule2 all keys aggregated by tier
        NZoom::NAggregators::TAggregationRules rules;
        rules.AddCustomRule(NTags::TRequestKey::FromString("itype=rule1"), TVector<TStringBuf>{"tier"});
        rules.AddCustomRule(NTags::TRequestKey::FromString("itype=rule1"), TVector<TStringBuf>{"geo", "ctype", "prj", "tier"});
        rules.AddCustomRule(NTags::TRequestKey::FromString("itype=rule2"), TVector<TStringBuf>{"tier"});
        rules.AddCustomRule(NTags::TRequestKey::FromString("itype=rule2"), TVector<TStringBuf>{"geo", "ctype", "prj", "tier"});

        auto key1 = TInstanceKey::FromNamed("rule1|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host");
        auto key2 = TInstanceKey::FromNamed("rule2|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=none|host");

        TVector<TInstanceKey> baseKeys{
            key1,
            key1.AggregateBy(TVector<TStringBuf>{"geo", "ctype", "prj", "tier"}),
            key1.AggregateBy(TVector<TStringBuf>{"geo", "ctype", "prj"}),
            key1.AggregateBy(TVector<TStringBuf>{"tier"}),
            key2.AggregateBy(TVector<TStringBuf>{"geo", "ctype", "prj", "tier"}),
            key2.AggregateBy(TVector<TStringBuf>{"geo", "ctype", "prj"}),
            key2.AggregateBy(TVector<TStringBuf>{"tier"}),
        };
        TVector<TInstanceKey> expectedKeys{
            key1,
            key1.AggregateBy(TVector<TStringBuf>{"geo", "ctype", "prj", "tier"}),
            key1.AggregateBy(TVector<TStringBuf>{"tier"}),
            key2,
            key2.AggregateBy(TVector<TStringBuf>{"tier"}),
            key2.AggregateBy(TVector<TStringBuf>{"geo", "ctype", "prj", "tier"}),
        };
        TestTags(rules, baseKeys, expectedKeys);
    }

    Y_UNIT_TEST(CheckUnaggregate) {
        // for itype = bytier? thea are agregation rule by tier
        // for itype = notier? thea are no agregation rules by tier
        // for itype = *1 thea is only key without aggregation by tier
        // for itype = *2 thea is only key with aggregation by tier
        // for itype = *3 thea are both keys with and without aggregation by tier
        NZoom::NAggregators::TAggregationRules rules;
        rules.AddCustomRule(NTags::TRequestKey::FromString("itype=bytier1"), TVector<TStringBuf>{"tier"});
        rules.AddCustomRule(NTags::TRequestKey::FromString("itype=bytier2"), TVector<TStringBuf>{"tier"});
        rules.AddCustomRule(NTags::TRequestKey::FromString("itype=bytier3"), TVector<TStringBuf>{"tier"});

        TVector<TInstanceKey> baseKeys {
            TInstanceKey::FromNamed("notier1|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),
            TInstanceKey::FromNamed("notier2|group=SAS.000;geo=man;ctype=prod;prj=yasm|host,tier"),
            TInstanceKey::FromNamed("notier3|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),
            TInstanceKey::FromNamed("notier3|group=SAS.000;geo=man;ctype=prod;prj=yasm|host,tier"),
            TInstanceKey::FromNamed("bytier1|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),
            TInstanceKey::FromNamed("bytier2|group=SAS.000;geo=man;ctype=prod;prj=yasm|host,tier"),
            TInstanceKey::FromNamed("bytier3|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),
            TInstanceKey::FromNamed("bytier3|group=SAS.000;geo=man;ctype=prod;prj=yasm|host,tier"),

            TInstanceKey::FromNamed("common|group=SAS.000|ctype,geo,host,prj,tier"),
        };

        TVector<TInstanceKey> expectedKeys {
            TInstanceKey::FromNamed("notier1|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),
            TInstanceKey::FromNamed("notier2|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=none|host"),
            TInstanceKey::FromNamed("notier3|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),

            TInstanceKey::FromNamed("bytier1|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),
            TInstanceKey::FromNamed("bytier2|group=SAS.000;geo=man;ctype=prod;prj=yasm|host,tier"),
            TInstanceKey::FromNamed("bytier2|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=none|host"),
            TInstanceKey::FromNamed("bytier3|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),
            TInstanceKey::FromNamed("bytier3|group=SAS.000;geo=man;ctype=prod;prj=yasm|host,tier"),

            TInstanceKey::FromNamed("common|group=SAS.000|ctype,geo,host,prj,tier"),
        };
        TestTags(rules, baseKeys, expectedKeys);
    }

    Y_UNIT_TEST(TestValues) {
        THashMap<TSignalName, TValue> baseValues;
        baseValues[TSignalName(TStringBuf("some_summ"))] = TValue(2);
        baseValues[TSignalName(TStringBuf("some_max"))] = TValue(2);
        // baseValues[TSignalName(TStringBuf("small_hgram"))] = TValue(THgram::Small({1.0, 2.0, 3.0}, 0)); // small hgram converts to Ugram
        baseValues[TSignalName(TStringBuf("ugram_hgram"))] = TValue(THgram::Ugram({TUgramBucket(2.0, 3.0, 10.0)}));
        // baseValues[TSignalName(TStringBuf("normal_hgram"))] = TValue(THgram::Normal({0.0, 51.0, 2.0, 2.0, 50.0}, 0, -1));

        TLog logger;
        const auto yasmConf(NZoom::NYasmConf::TYasmConf::FromString(TStringBuf("{\"conflist\": {\"common\": {\"signals\": {}, \"patterns\": {}, \"periods\": {}}}}")));
        NZoom::NAggregators::TAggregationRules rules;
        TTempDir root;
        root.Path().MkDir();

        TRecordPeriod period(TRecordPeriod::Get("m5"));
        TInstant startTime(period.GetStartTime(TInstant::Now()));

        TMockedRecordDescriptor baseRecord;
        baseRecord.SetStartTime(startTime).SetEndTime(startTime).SetFlushOffset(startTime).SetInstanceKey(TInstanceKey::FromNamed("common||ctype,geo,prj,tier"));
        THashSet<TString> itypes = {TString{baseRecord.GetInstanceKey().GetItype()}};
        {
            TWritingPipeline pipeline(logger, yasmConf, root.Name(), 1, startTime - TDuration::Days(1));

            pipeline.Start();

            for (const auto& [signal, value]: baseValues) {
                TVector<TValue> values;
                values.emplace_back(value.GetValue());
                baseRecord.SetSignalName(signal).SetValues(values);
                pipeline.OnRecord(baseRecord);
            }

            auto flush = startTime + TDuration::Days(2);
            baseRecord.SetStartTime(flush).SetEndTime(flush).SetFlushOffset(flush);
            pipeline.OnRecord(baseRecord);

            pipeline.Stop();
            pipeline.Finish();
        }

        {
            TRecordSaver pipeline;
            THashSet<TString> itypes = {
                TString{baseRecord.GetInstanceKey().GetItype()}
            };
            for (auto startTime: TSecondPlacementReader::GetChunkTimes(root.Path(), baseRecord.GetHostName().GetName(), period)) {
                NHistDb::SendChunk(logger, rules, root.Path(), baseRecord.GetHostName(), startTime, TInstant::Now(), period, itypes, pipeline);
            }

            UNIT_ASSERT_VALUES_EQUAL(baseValues.size(), pipeline.GetRecords().size());

            for (const auto& record : pipeline.GetRecords()) {
                auto base = baseValues.find(record->GetSignalName());
                UNIT_ASSERT(base != baseValues.end());
                TTestValue tester(base->second);
                record->Iterate(tester);
            }
        }
    }

    Y_UNIT_TEST(IgnoreNoFile) {
        TLog logger;
        NZoom::NAggregators::TAggregationRules rules;
        TFsPath root;
        THostName host;
        TInstant startTime = TInstant::Zero();
        TRecordPeriod period(TRecordPeriod::Get("s5"));
        THashSet<TString> itypes;
        TRecordSaver pipeline;
        UNIT_CHECK_GENERATED_EXCEPTION(NHistDb::SendChunk(logger, rules, root, host, startTime, TInstant::Now(), period, itypes, pipeline), yexception);
    }

    Y_UNIT_TEST(IgnoreNirvanajob) {
        NZoom::NAggregators::TAggregationRules rules;

        TVector<TInstanceKey> baseKeys {
            TInstanceKey::FromNamed("nirvanajob|group=SAS.000;geo=man;ctype=prod;prj=yasm;tier=some|host"),
        };

        TVector<TInstanceKey> expectedKeys {
        };
        TestTags(rules, baseKeys, expectedKeys);
    }
}

THashMap<TString, TString> whitelist = {{"testhost_bad" , "testhost"}};

Y_UNIT_TEST_SUITE(InstanceKeyProcessorTests) {
    Y_UNIT_TEST(EmptyGroupAndHostFilled) {
        TInstanceKey instanceKey = TInstanceKey::FromNamed("balancer|ctype=prod;prj=yasm|host");
        TInstanceKey expectedInstanceKey = TInstanceKey::FromNamed("balancer|group=testhost;ctype=prod;prj=yasm|host");
        TInstanceKey fixedInstanceKey = TInstanceKeyProcessor::FixInstanceKey(instanceKey, TString("testhost"), whitelist);
        UNIT_ASSERT_VALUES_EQUAL(fixedInstanceKey, expectedInstanceKey);
    }

    Y_UNIT_TEST(EmptyHostFilled) {
        TInstanceKey instanceKey = TInstanceKey::FromNamed("balancer|group=testhost;ctype=prod;prj=yasm|host");
        TInstanceKey expectedInstanceKey = TInstanceKey::FromNamed("balancer|group=testhost;ctype=prod;prj=yasm|host");
        TInstanceKey fixedInstanceKey = TInstanceKeyProcessor::FixInstanceKey(instanceKey, TString("testhost"), whitelist);
        UNIT_ASSERT_VALUES_EQUAL(fixedInstanceKey, expectedInstanceKey);
    }

    Y_UNIT_TEST(WrongWhitelistedGroupRenamed) {
        TInstanceKey instanceKey = TInstanceKey::FromNamed("balancer|group=testhost_bad;ctype=prod;prj=yasm|host");
        TInstanceKey expectedInstanceKey = TInstanceKey::FromNamed("balancer|group=testhost;ctype=prod;prj=yasm|host");
        TInstanceKey fixedInstanceKey = TInstanceKeyProcessor::FixInstanceKey(instanceKey, TString("testhost"), whitelist);
        UNIT_ASSERT_VALUES_EQUAL(fixedInstanceKey, expectedInstanceKey);
    }

    Y_UNIT_TEST(ValidKeyNotChanged) {
        TInstanceKey instanceKey = TInstanceKey::FromNamed("balancer|group=testhost;ctype=prod;prj=yasm|host");
        TInstanceKey fixedInstanceKey = TInstanceKeyProcessor::FixInstanceKey(instanceKey, TString("testhost"), whitelist);
        UNIT_ASSERT_VALUES_EQUAL(fixedInstanceKey, instanceKey);
    }

    Y_UNIT_TEST(NotWhitelistedRenamingThrows1) {
        TInstanceKey instanceKey = TInstanceKey::FromNamed("balancer|group=testhost_bad2;ctype=prod;prj=yasm|host");
        UNIT_ASSERT_EXCEPTION(TInstanceKeyProcessor::FixInstanceKey(instanceKey, TString("testhost"), whitelist), UnexpectedInsanceKey);
    }

    Y_UNIT_TEST(NotWhitelistedRenamingThrows2) {
        TInstanceKey instanceKey = TInstanceKey::FromNamed("balancer|group=testhost_bad;ctype=prod;prj=yasm|host");
        UNIT_ASSERT_EXCEPTION(TInstanceKeyProcessor::FixInstanceKey(instanceKey, TString("testhost2"), whitelist), UnexpectedInsanceKey);
    }

    Y_UNIT_TEST(NotEmptyHostFixThrows) {
        TInstanceKey instanceKey = TInstanceKey::FromNamed("balancer|host=some_host;ctype=prod;prj=yasm");
        UNIT_ASSERT_EXCEPTION(TInstanceKeyProcessor::FixInstanceKey(instanceKey, TString("testhost"), whitelist), UnexpectedInsanceKey);
    }
}
