#include <infra/yasm/histdb/components/placements/second.h>

#include <library/cpp/getopt/last_getopt.h>

#include <util/generic/yexception.h>

using namespace NLastGetopt;
using namespace NHistDb;
using namespace NTags;
using namespace NZoom::NSignal;
using namespace NZoom::NYasmConf;

int main(int argc, const char** argv) {
    TOpts options{TOpts::Default()};

    TString optHistdbPath;
    options
        .AddCharOption('d', "histdb path")
        .AddLongName("histdb")
        .RequiredArgument("DIR")
        .Required()
        .StoreResult(&optHistdbPath);

    TString optHostName;
    options
        .AddCharOption('n', "host name")
        .AddLongName("hostname")
        .RequiredArgument("HOST")
        .Required()
        .StoreResult(&optHostName);

    TString optPeriod;
    options
        .AddCharOption('p', "period name")
        .AddLongName("period")
        .RequiredArgument("PERIOD")
        .Required()
        .StoreResult(&optPeriod);

    ui64 optStartTime;
    options
        .AddCharOption('t', "start time")
        .AddLongName("start-time")
        .RequiredArgument("TIME")
        .Required()
        .StoreResult(&optStartTime);

    TString optRequestKey;
    options
        .AddCharOption('r', "request key")
        .AddLongName("request-key")
        .RequiredArgument("REQUEST_KEY")
        .Required()
        .StoreResult(&optRequestKey);

    TVector<TString> optSignals;
    options
        .AddCharOption('s', "signal list")
        .AddLongName("signals")
        .RequiredArgument("SIGNALS")
        .Required()
        .SplitHandler(&optSignals, ',');

    options.AddHelpOption('h');
    options.AddVersionOption('v');

    TOptsParseResult optParsing(&options, argc, argv);

    try {
        TRecordPeriod period(TRecordPeriod::Get(optPeriod));
        TInstant startTime(period.GetStartTime(TInstant::Seconds(optStartTime)));

        TVector<TSignalName> signals;
        signals.insert(signals.end(), optSignals.begin(), optSignals.end());

        TSomethingFormat::TTagSignals tags;
        tags.emplace_back(TRequestKey::FromString(optRequestKey), std::move(signals));

        TYasmConf yasmConf = TYasmConf::FromString("{\n"
                                                   "  \"conflist\": {\n"
                                                   "    \"common\": {\n"
                                                   "      \"patterns\": {},\n"
                                                   "      \"periods\": {},\n"
                                                   "      \"signals\": {}\n"
                                                   "    }\n"
                                                   "  }\n"
                                                   "}");

        Cerr << "reading from root=" << optHistdbPath
             << "\thost=" << optHostName
             << "\tperiod=" << period.GetResolution()
             << "\tstart=" << startTime
             << Endl;

        TSecondPlacementReader reader(optHistdbPath, optHostName, period, startTime);
        const auto reply(reader.ReadAggregated(0, 300, tags, yasmConf));

        for (const auto& row : reply) {
            Cout << "ts=" << period.FromOffset(startTime, row.Timestamp) << Endl;
            for (const auto& signalValue : row.Record.GetValues()) {
                Cout << "signal=" << signalValue.first << "\tvalue=" << signalValue.second << Endl;
            }
        }

    } catch (...) {
        Cerr << CurrentExceptionMessage() << Endl;
        return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}
