import requests
import argparse

# https://solomon.yandex-team.ru/swagger-ui.html#!/clusters/updateClusterUsingPUT

API_URLS = {
    "prod" : "https://solomon.yandex-team.ru/api/v2/projects",
    "test" : "http://solomon-test.yandex.net/api/v2/projects",
    "prestable": "http://solomon-prestable.yandex.net/api/v2/projects",
}

class SolomonAPI(object):
    def __init__(self, contour):
        self._base_url = API_URLS[contour]
        token = open('token').read().strip()

        self.header = {
            'Authorization': 'OAuth {}'.format(token),
            'Accept' : 'application/json',
        }

        self.put_header = self.header.copy()
        self.put_header['Content-Type'] = 'application/json'

    def _put_request(self, req, data):
        url = self._base_url + req
        response = requests.put(url, headers=self.put_header, json=data)
        return response.json()

    def _del_request(self, req):
        url = self._base_url + req
        return requests.delete(url, headers=self.header)

    def _get_request(self, req):
        url = self._base_url + req
        response = requests.get(url, headers=self.header)
        return response.json()

    def get_all_yasm_projects(self):
        request = self._get_request("?text=yasm_")
        return [project['id'] for project in request]

    def get_clusters(self, project_id):
        clusters = []
        page = 0
        while True:
            request = "/{}/clusters?page={}".format(project_id, page)
            result = self._get_request(request)['result']
            if (len(result) == 0):
                break
            page += 1
            clusters.extend(shard['id'] for shard in result)
        return clusters

    def get_shards(self, project_id):
        shards = []
        page = 0
        while True:
            request = "/{}/shards?page={}".format(project_id, page)
            result = self._get_request(request)['result']
            if (len(result) == 0):
                break
            page += 1
            shards.extend(shard['id'] for shard in result)
        return shards

    def del_shard(self, project_id, shard_id):
        return self._del_request("/{}/shards/{}".format(project_id, shard_id))

    def get_cluster(self, project_id, cluster_id):
        return self._get_request("/{}/clusters/{}".format(project_id, cluster_id))

    def get_sensors_count(self, project_id):
        return self._get_request("/{}/sensors".format(project_id))['page']['totalCount']

    def set_ttl(self, project_id, cluster_id, ttl):
        data = self.get_cluster(project_id, cluster_id)
        if data.get('sensorsTtlDays') != ttl:
            data['sensorsTtlDays'] = ttl
            self._put_request("/{}/clusters/{}".format(project_id, cluster_id), data)
            return True
        return False

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument('--type', type=str, default='test', help="cluster type")
    parser.add_argument('--host-ttl', type=int, default=30)
    parser.add_argument('--group-ttl', type=int, default=365)
    settings = parser.parse_args()

    api = SolomonAPI(settings.type)
    for project in api.get_all_yasm_projects():
        for cluster in api.get_clusters(project):
            cluster_name = api.get_cluster(project, cluster)['name']
            cluster_type = cluster_name.split("_")[0]
            if cluster_type == "group":
                api.set_ttl(project, cluster, settings.group_ttl)
            elif cluster_type == "host":
                api.set_ttl(project, cluster, settings.host_ttl)
            else:
                print "Unknown type", project, cluster
