#pragma once

#include <infra/yasm/histdb/components/placements/periods.h>
#include <infra/yasm/common/labels/host/host.h>

#include <infra/monitoring/common/settings.h>
#include <infra/yasm/stockpile_client/settings.h>

#include <util/datetime/base.h>

#include <library/cpp/getopt/last_getopt.h>

namespace NHistDb {
    class TStockpileRequestSettings final: public NMonitoring::TSettings {
        using THostName = NZoom::NHost::THostName;
    public:
        void ProcessParsedOptions(const NLastGetopt::TOptsParseResult& parsed) {
            SetPeriod(parsed.Get("period"));
            SetHost(parsed.Get("host"));
            SetSignal(parsed.Get("signal"));
            SetRequest(parsed.Get("request"));
            SetStart(parsed.Get<size_t>("start"));
            SetEnd(parsed.Get<size_t>("end"));
            StockpileSettingsFactory.ProcessParsedOptions(parsed);
        }

        static void FillParser(NLastGetopt::TOpts& options) {
            options.AddLongOption("period", "request period").Optional().RequiredArgument().DefaultValue("m5");
            options.AddLongOption("host", "request host").Optional().RequiredArgument().DefaultValue("MYT.000");
            options.AddLongOption("signal", "request signal").Optional().RequiredArgument().DefaultValue("counter-instance_tmmv");
            options.AddLongOption("request", "request key").Optional().RequiredArgument().DefaultValue("itype=yasmagentmetrics;ctype=prod");
            options.AddLongOption("start", "request start").Optional().RequiredArgument().DefaultValue("1561248000");
            options.AddLongOption("end", "request end").Optional().RequiredArgument().DefaultValue("1561248300");
            NStockpile::TStockpileSetingsFactory::FillParser(options);
        }

        inline TDuration GetPeriod() const {
            return Period;
        }

        inline const THostName& GetHost() const {
            return Host;
        }

        inline const TString& GetSignal() const {
            return Signal;
        }

        inline const TString& GetRequest() const {
            return RequestKey;
        }

        inline TInstant GetStart() const {
            return Start;
        }

        inline TInstant GetEnd() const {
            return End;
        }

        NStockpile::TSettings CreateStockpileSettings() const {
            return StockpileSettingsFactory.CreateStockpileSettings(true);
        }

    private:
        inline void SetPeriod(const TString& value) {
            Period = TRecordPeriod::Get(value).GetResolution();
        }

        inline void SetHost(const TString& value) {
            Host = value;
        }

        inline void SetSignal(const TString& value) {
            Signal = value;
        }

        inline void SetRequest(const TString& value) {
            RequestKey = value;
        }

        inline void SetStart(size_t value) {
            Start = TInstant::Seconds(value);
        }

        inline void SetEnd(size_t value) {
            End = TInstant::Seconds(value);
        }

    private:
        TString Signal;
        TString RequestKey;
        TDuration Period;
        THostName Host;
        TInstant Start;
        TInstant End;
        NStockpile::TStockpileSetingsFactory StockpileSettingsFactory;
    };

    inline TStockpileRequestSettings GetStockpileRequestSettings(int argc, const char** argv) {
        TStockpileRequestSettings settings;
        auto options = NLastGetopt::TOpts::Default();
        settings.FillParser(options);

        NLastGetopt::TOptsParseResult parsed(&options, argc, argv);

        settings.ProcessParsedOptions(parsed);
        return settings;
    }
}
