#pragma once

#include "ifaces.h"

#include <infra/yasm/zoom/components/compression/chunk.h>
#include <infra/yasm/histdb/components/formats/something.h>
#include <infra/yasm/histdb/components/placements/second.h>
#include <infra/yasm/histdb/components/placements/transpiler.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NHistDb {
    class TMockedRecordDescriptor final: public IRecordDescriptor {
    public:
        TMockedRecordDescriptor()
            : HostName("SAS.000")
            , InstanceKey(NTags::TInstanceKey::FromNamed("testing||geo,ctype,prj,tier"))
            , SignalName(TStringBuf("disk_usage_summ"))
            , ValuesCount(0)
        {
        }

        NZoom::NHost::THostName GetHostName() const override {
            return {HostName};
        }

        TMockedRecordDescriptor& SetHostName(TStringBuf hostName) {
            HostName = hostName;
            return *this;
        }

        NTags::TInstanceKey GetInstanceKey() const override {
            return InstanceKey;
        }

        TMockedRecordDescriptor& SetInstanceKey(NTags::TInstanceKey instanceKey) {
            InstanceKey = instanceKey;
            return *this;
        }

        NZoom::NSignal::TSignalName GetSignalName() const override {
            return SignalName;
        }

        TMockedRecordDescriptor& SetSignalName(NZoom::NSignal::TSignalName signalName) {
            SignalName = signalName;
            return *this;
        }

        TInstant GetStartTime() const override {
            return StartTime;
        }

        TMockedRecordDescriptor& SetStartTime(TInstant startTime) {
            StartTime = startTime;
            return *this;
        }

        TInstant GetEndTime() const override {
            return EndTime;
        }

        TMockedRecordDescriptor& SetEndTime(TInstant endTime) {
            EndTime = endTime;
            return *this;
        }

        TInstant GetFlushOffset() const override {
            return FlushOffset;
        }

        TMockedRecordDescriptor& SetFlushOffset(TInstant flushOffset) {
            FlushOffset = flushOffset;
            return *this;
        }

        void Iterate(IRecordVisitor& visitor) const override {
            for (const auto& value : Values) {
                visitor.OnValue(value.GetValue());
            }
        }

        TMockedRecordDescriptor& SetValues(const TVector<NZoom::NValue::TValue>& values) {
            Values.clear();
            for (const auto& value : values) {
                Values.emplace_back(value.GetValue());
            }

            NYasmServer::TDoubleChunk chunk(0);
            for (const auto idx : xrange(Values.size())) {
                chunk.SetTimestamp(idx);
                Values[idx].Update(chunk);
            }
            Data = chunk.GetData();
            ValuesCount = chunk.GetValuesWritten();

            return *this;
        }

        size_t GetValuesCount() const override {
            return ValuesCount;
        }

        NYasmServer::ESeriesKind GetSeriesKind() const override {
            return NYasmServer::ESeriesKind::Double;
        }

        const TString& GetData() const override {
            return Data;
        }

        THolder<IRecordDescriptor> Clone() const override {
            auto other(MakeHolder<TMockedRecordDescriptor>());
            (*other)
                .SetHostName(HostName)
                .SetInstanceKey(InstanceKey)
                .SetSignalName(SignalName)
                .SetStartTime(StartTime)
                .SetEndTime(EndTime)
                .SetFlushOffset(FlushOffset)
                .SetValues(Values);
            return other;
        }

        THolder<IRecordDescriptor> CloneWithNewKey(const NTags::TInstanceKey& newKey) const override {
            auto other(MakeHolder<TMockedRecordDescriptor>());
            (*other)
                .SetHostName(HostName)
                .SetInstanceKey(newKey)
                .SetSignalName(SignalName)
                .SetStartTime(StartTime)
                .SetEndTime(EndTime)
                .SetFlushOffset(FlushOffset)
                .SetValues(Values);
            return other;
        };

    private:
        TString HostName;
        NTags::TInstanceKey InstanceKey;
        NZoom::NSignal::TSignalName SignalName;
        TInstant StartTime;
        TInstant EndTime;
        TInstant FlushOffset;
        TVector<NZoom::NValue::TValue> Values;
        TString Data;
        size_t ValuesCount;
    };

    class TMockedSnapshotDescriptor final: public ISnapshotDescriptor {
    public:
        TMockedSnapshotDescriptor() {
            RecordDescriptor.ConstructInPlace();
        }

        void Iterate(ISnapshotVisitor& visitor) const override {
            if (RecordDescriptor.Defined()) {
                visitor.OnRecord(*RecordDescriptor);
            }
        }

        TMockedRecordDescriptor& GetRecordDescriptor() {
            return *RecordDescriptor;
        }

        TMockedSnapshotDescriptor& ResetRecordDescriptor() {
            RecordDescriptor.Clear();
            return *this;
        }

    private:
        TMaybe<TMockedRecordDescriptor> RecordDescriptor;
    };

    class TResponseChecker {
    public:
        static TResponseChecker Read(TSecondPlacementReader& reader,
                                     TVector<TSomethingFormat::TTimestamp> timestamps,
                                     NTags::TRequestKey requestKey,
                                     TVector<NZoom::NSignal::TSignalName> signals) {
            TSomethingFormat::TTagSignals tags;
            tags.emplace_back(std::move(requestKey), std::move(signals));
            return {reader.Read(timestamps, tags)};
        }

        TResponseChecker(TVector<TSomethingFormat::TReadData>&& response)
            : Response(std::move(response))
            , CurrentRow(Response.end())
        {
        }

        TResponseChecker& CheckTimestampIsMissing(TSomethingFormat::TTimestamp timestamp) {
            for (auto it(Response.begin()); it != Response.end(); ++it) {
                if (it->Timestamp == timestamp) {
                    UNIT_FAIL("timestamp found");
                    return *this;
                }
            }
            return *this;
        }

        TResponseChecker& SeekForTimestamp(TSomethingFormat::TTimestamp timestamp) {
            for (auto it(Response.begin()); it != Response.end(); ++it) {
                if (it->Timestamp == timestamp) {
                    CurrentRow = it;
                    return *this;
                }
            }
            UNIT_FAIL("no timestamp found");
            return *this;
        }

        TResponseChecker& CheckSignal(NZoom::NSignal::TSignalName signalName, const NZoom::NValue::TValue& value) {
            UNIT_ASSERT(CurrentRow != Response.end());
            const auto& values(CurrentRow->Record.GetValues());
            for (const auto& pair : values) {
                if (pair.first == signalName) {
                    UNIT_ASSERT_VALUES_EQUAL(pair.second, value);
                    return *this;
                }
            }
            UNIT_FAIL("no signal found");
            return *this;
        }

    private:
        TVector<TSomethingFormat::TReadData> Response;
        TVector<TSomethingFormat::TReadData>::iterator CurrentRow;
    };
}
