#pragma once

#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/yasm/zoom/components/record/record.h>
#include <infra/yasm/common/labels/host/host.h>
#include <infra/yasm/zoom/components/compression/series.h>

#include <util/datetime/base.h>

namespace NHistDb {
    class ITimeFinder {
    public:
        virtual ~ITimeFinder() = default;
        virtual void Add(TMaybe<TInstant> other) = 0;
        virtual TMaybe<TInstant> Result() = 0;
    };

    class TMinTimeFinder: public ITimeFinder {
    public:
        void Add(TMaybe<TInstant> other) override {
            if (LastTime.Defined() && other.Defined()) {
                LastTime = Min(*LastTime, *other);
            } else if (!LastTime.Defined() && other.Defined()) {
                LastTime = other;
            }
        }

        TMaybe<TInstant> Result() override {
            return LastTime;
        }

    private:
        TMaybe<TInstant> LastTime;
    };

    class TMinTimeFinderWithLimit: public ITimeFinder {
    public:
        explicit TMinTimeFinderWithLimit(TInstant limit)
            : Limit(limit)
            , AcceptedCount(0)
            , IgnoredCount(0) {
        }

        void Add(TMaybe<TInstant> otherLastTime) override {
            if (otherLastTime.Defined()) {
                if (*otherLastTime > Limit) {
                    ++AcceptedCount;
                    LastTime = (LastTime.Defined()) ? Min(*LastTime, *otherLastTime) : *otherLastTime;
                } else {
                    ++IgnoredCount;
                }
            }
        }

        TMaybe<TInstant> Result() override {
            return LastTime;
        }

        size_t GetAcceptedCount() const {
            return AcceptedCount;
        }

        size_t GetIgnoredCount() const {
            return IgnoredCount;
        }

    private:
        TMaybe<TInstant> LastTime;
        const TInstant Limit;
        size_t AcceptedCount;
        size_t IgnoredCount;
    };

    using IRecordVisitor = NZoom::NRecord::IRecordVisitor;

    class IRecordDescriptor : public NZoom::NRecord::IRecordIterable {
    public:
        virtual NZoom::NHost::THostName GetHostName() const = 0;
        virtual NTags::TInstanceKey GetInstanceKey() const = 0;
        virtual NZoom::NSignal::TSignalName GetSignalName() const = 0;

        virtual TInstant GetStartTime() const = 0;
        virtual TInstant GetEndTime() const = 0;

        virtual TInstant GetFlushOffset() const = 0;

        virtual size_t GetValuesCount() const = 0;
        virtual NYasmServer::ESeriesKind GetSeriesKind() const = 0;
        virtual const TString& GetData() const = 0;

        virtual THolder<IRecordDescriptor> Clone() const = 0;
        virtual THolder<IRecordDescriptor> CloneWithNewKey(const NTags::TInstanceKey& newKey) const = 0;
    };

    class ISnapshotVisitor {
    public:
        virtual ~ISnapshotVisitor() = default;

        virtual void OnRecord(const IRecordDescriptor& recordDescriptor) = 0;

        virtual TMaybe<TInstant> GetLastTime() = 0;
        virtual void AddLastTimesToFinder(ITimeFinder& finder);

        virtual void Finish();

        virtual bool IsActive() const;

        virtual void Update();

        virtual void CollectStats();

        virtual void PreloadHostName(NZoom::NHost::THostName hostName);
    };

    class ISnapshotDescriptor {
    public:
        virtual ~ISnapshotDescriptor() = default;

        virtual void Iterate(ISnapshotVisitor& visitor) const = 0;
    };
}
