#include "ifaces.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NHistDb;

Y_UNIT_TEST_SUITE(TTMinTimeFinderTests) {

    Y_UNIT_TEST(NoVisitors) {
        TMinTimeFinder finder;
        auto result = finder.Result();
        UNIT_ASSERT(!result.Defined());
    }

    Y_UNIT_TEST(OnlyNoTimeVisitors) {
        TMinTimeFinder finder;
        finder.Add(TMaybe<TInstant>());
        finder.Add(TMaybe<TInstant>());
        finder.Add(TMaybe<TInstant>());
        auto result = finder.Result();
        UNIT_ASSERT(!result.Defined());
    }

    Y_UNIT_TEST(NestedMinTimeFinders) {
        TMinTimeFinder topLevelFinder;
        TMinTimeFinder allValuesFinder;

        size_t startDays = 10000;
        for (size_t t = startDays; t < startDays + 100; ++t) {
            TMinTimeFinder localFinder;
            auto time1 = TInstant::Days(t);
            auto time2 = TInstant::Days(t + 10);
            auto time3 = TMaybe<TInstant>();

            localFinder.Add(time1);
            allValuesFinder.Add(time1);
            localFinder.Add(time2);
            allValuesFinder.Add(time2);
            localFinder.Add(time3);
            allValuesFinder.Add(time3);

            topLevelFinder.Add(localFinder.Result());
        }

        auto topLevelResult = topLevelFinder.Result();
        auto allValuesResult = allValuesFinder.Result();
        UNIT_ASSERT(topLevelResult.Defined() && allValuesResult.Defined());
        UNIT_ASSERT_VALUES_EQUAL(*topLevelResult, *allValuesResult);
        UNIT_ASSERT_VALUES_EQUAL(*topLevelResult, TInstant::Days(startDays));
    }
}

Y_UNIT_TEST_SUITE(TTMinTimeFinderWithLimitTests) {

    Y_UNIT_TEST(NoVisitors) {
        TMinTimeFinderWithLimit finder(TInstant::Hours(100));
        auto result = finder.Result();
        UNIT_ASSERT(!result.Defined());
        UNIT_ASSERT_VALUES_EQUAL(0, finder.GetAcceptedCount());
        UNIT_ASSERT_VALUES_EQUAL(0, finder.GetIgnoredCount());
    }

    Y_UNIT_TEST(OnlyNoTimeVisitors) {
        TMinTimeFinderWithLimit finder(TInstant::Hours(1000));
        finder.Add(TMaybe<TInstant>());
        finder.Add(TMaybe<TInstant>());
        finder.Add(TMaybe<TInstant>());
        auto result = finder.Result();
        UNIT_ASSERT(!result.Defined());
        UNIT_ASSERT_VALUES_EQUAL(0, finder.GetAcceptedCount());
        UNIT_ASSERT_VALUES_EQUAL(0, finder.GetIgnoredCount());
    }

    Y_UNIT_TEST(OverLimitAndNoTimeVisitors) {
        auto limit = TInstant::Hours(1000);
        TMinTimeFinderWithLimit finder(limit);
        finder.Add(TMaybe<TInstant>());
        finder.Add(limit - TDuration::Minutes(10));
        finder.Add(limit - TDuration::Minutes(40));
        finder.Add(TMaybe<TInstant>());
        finder.Add(limit - TDuration::Minutes(60));
        auto result = finder.Result();
        UNIT_ASSERT(!result.Defined());
        UNIT_ASSERT_VALUES_EQUAL(0, finder.GetAcceptedCount());
        UNIT_ASSERT_VALUES_EQUAL(3, finder.GetIgnoredCount());
    }

    Y_UNIT_TEST(SomeOverLimitVisitors) {
        auto limit = TInstant::Hours(1000);
        TMinTimeFinderWithLimit finder(limit);
        auto underLimit1 = limit + TDuration::Minutes(60);
        auto underLimit2 = limit + TDuration::Minutes(45);
        auto underLimit3 = limit + TDuration::Minutes(90);

        finder.Add(limit - TDuration::Minutes(10));
        finder.Add(TMaybe<TInstant>());
        finder.Add(underLimit1);
        finder.Add(limit - TDuration::Minutes(40));
        finder.Add(TMaybe<TInstant>());
        finder.Add(underLimit2);
        finder.Add(limit - TDuration::Minutes(60));
        finder.Add(TMaybe<TInstant>());
        finder.Add(underLimit3);
        auto result = finder.Result();
        UNIT_ASSERT(result.Defined());
        UNIT_ASSERT_VALUES_EQUAL(result.GetRef(), underLimit2);
        UNIT_ASSERT_VALUES_EQUAL(3, finder.GetAcceptedCount());
        UNIT_ASSERT_VALUES_EQUAL(3, finder.GetIgnoredCount());
    }

    Y_UNIT_TEST(NestedFinders) {
        const size_t startDays = 10000;
        auto limit = TInstant::Days(startDays + 50);
        TMinTimeFinderWithLimit topLevelFinder(limit);
        TMinTimeFinderWithLimit allValuesFinder(limit);

        const size_t step = 5;
        for (size_t t = startDays; t < startDays + 100; t += step) {
            TMinTimeFinderWithLimit localFinder(limit);
            auto time1 = TInstant::Days(t);
            auto time2 = TInstant::Days(t + 10);
            auto time3 = TMaybe<TInstant>();

            localFinder.Add(time1);
            allValuesFinder.Add(time1);
            localFinder.Add(time2);
            allValuesFinder.Add(time2);
            localFinder.Add(time3);
            allValuesFinder.Add(time3);

            topLevelFinder.Add(localFinder.Result());
        }

        auto topLevelResult = topLevelFinder.Result();
        auto allValuesResult = allValuesFinder.Result();
        UNIT_ASSERT(topLevelResult.Defined() && allValuesResult.Defined());
        UNIT_ASSERT_VALUES_EQUAL(*topLevelResult, *allValuesResult);
        UNIT_ASSERT_VALUES_EQUAL(*topLevelResult, limit + TDuration::Days(step));
        UNIT_ASSERT_VALUES_EQUAL(20, allValuesFinder.GetAcceptedCount());
        UNIT_ASSERT_VALUES_EQUAL(20, allValuesFinder.GetIgnoredCount());
        UNIT_ASSERT_VALUES_EQUAL(11, topLevelFinder.GetAcceptedCount());
        UNIT_ASSERT_VALUES_EQUAL(0, topLevelFinder.GetIgnoredCount());
    }
}
