#include "compact.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/memory/blob.h>
#include <util/generic/xrange.h>

#include <infra/yasm/zoom/components/compression/chunk.h>

using namespace NHistDb;
using namespace NTags;
using namespace NZoom::NRecord;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;
using namespace NZoom::NYasmConf;

namespace {
    const static TString TESTING_DATA = "testing.data";

    class TCompactHelper {
    public:
        TCompactFormat& Writer() noexcept {
            if (!OutputFormat.Defined()) {
                OutputFormat.ConstructInPlace();
            }
            if (!OutputStream.Defined()) {
                OutputStream.ConstructInPlace(TESTING_DATA);
            }
            return OutputFormat.GetRef();
        }

        void StartWritingKeyData(const TInstanceKey& key) {
            Writer().Start(key, OutputStream.GetRef());
        }

        void CommitCurrentKeyData() {
            Writer().Commit(OutputStream.GetRef());
        }

        void WriteSignalValues(TAbstractFormat::TTimestamp timestamp, const TSignalName& signal,
            const TVector<TValue>& values) noexcept {
            TMaybe<NYasmServer::ESeriesKind> seriesKind;
            TString data = NYasmServer::EncodeChunk(values, seriesKind, false);

            Writer().Append(signal, timestamp, values.size(), seriesKind.GetRef(), data);
        }

        void Finish() noexcept {
            UNIT_ASSERT(Header.empty());
            Writer();
            OutputStream->Finish();
            OutputFormat->SaveBlocks(OutputStream->Blocks());
            Header = OutputFormat->Dump();
        }

        TCompactFormat& Reader() noexcept {
            UNIT_ASSERT(!Header.empty());
            if (!InputFormat.Defined()) {
                InputFormat.ConstructInPlace(TStringBuf(Header));
            }
            if (!InputStream.Defined()) {
                InputStream.ConstructInPlace(TESTING_DATA, InputFormat->GetBlocks());
            }
            return InputFormat.GetRef();
        }

        TVector<TAbstractFormat::TReadData> Read(
            const TVector<TAbstractFormat::TTimestamp>& times,
            TVector<std::pair<TRequestKey, TVector<TSignalName>>>& tags
        ) {
            return Reader().Read(times, tags, InputStream.GetRef());
        }

        TVector<TAbstractFormat::TReadAggregatedData> ReadAggregated(
            size_t offset, size_t limit,
            const TAbstractFormat::TTagSignals& tags,
            const TYasmConf& conf
        ) {
            return Reader().ReadAggregated(offset, limit, tags, conf, InputStream.GetRef());
        }

    private:
        TString Header;

        TMaybe<TCompactFormat> OutputFormat;
        TMaybe<TSnappyOutputStream> OutputStream;

        TMaybe<TCompactFormat> InputFormat;
        TMaybe<TSnappyInputStream> InputStream;
    };

    class TRecordBuilder {
    public:
        TRecord Build() {
            return {std::move(Values)};
        }

        TRecordBuilder& Add(TSignalName signal, TValue value) {
            Values.emplace_back(std::move(signal), std::move(value));
            return *this;
        }

    private:
        TVector<std::pair<TSignalName, TValue>> Values;
    };
}

Y_UNIT_TEST_SUITE(TCompactFormatTest) {
    Y_UNIT_TEST(ReadSeveralRecords) {
        const TVector<TString> signalStrings = {
            "unistat-ok_subsource_VIDEO_dmmm",
            "unistat-init_subsource_VIDEO_dmmm",
            "unistat-unanswers_subsource_VIDEO_dmmm",
        };
        const TVector<TSignalName> signals(signalStrings.begin(), signalStrings.end());

        const TVector<TString> requestStrings = {
            "itype=testing",
            "itype=testing;geo=man",
            "itype=testing;geo=sas",
        };

        const TVector<TAbstractFormat::TTimestamp> timestamps = {1, 2, 3, 4, 5};

        TCompactHelper helper;
        const TVector<TInstanceKey> instanceKeysGeo = {
            TInstanceKey::FromNamed("testing|ctype=prestable;prj=test|tier"),
            TInstanceKey::FromNamed("testing|ctype=prestable;geo=man;prj=test|tier"),
            TInstanceKey::FromNamed("testing|ctype=prestable;geo=sas;prj=test|tier"),
        };

        for (const auto& instanceKey: instanceKeysGeo) {
            helper.StartWritingKeyData(instanceKey);
            for (const auto& signal: signals) {
                TVector<TValue> values;
                for (auto ts: timestamps) {
                    Y_UNUSED(ts);
                    values.emplace_back(1);
                }
                helper.WriteSignalValues(timestamps.front(), signal, values);
            }
            helper.CommitCurrentKeyData();
        }
        helper.Finish();

        TVector<std::pair<TRequestKey, TVector<TSignalName>>> tags;
        for (const auto& requestString: requestStrings) {
            tags.emplace_back(
                TRequestKey::FromString(requestString),
                signals
            );
        }
        auto res = helper.Read(timestamps, tags);

        UNIT_ASSERT_VALUES_EQUAL(res.size(), 25);
    }

    Y_UNIT_TEST(ReadDifferentPreaggrsAndNoDataSignal) {
        TSignalName signalName = TStringBuf("unistat-ok_subsource_VIDEO_dmmm");

        const TVector<TString> requestStrings = {
            "itype=testing",
        };

        const TVector<TAbstractFormat::TTimestamp> timestamps = {1, 2, 3};


        TCompactHelper helper;
        {
            // full pre-aggregate at timestamp 0, with no data at timestamps 1 and 2
            helper.StartWritingKeyData(TInstanceKey::FromNamed("testing||geo,ctype,prj,tier"));
            TVector<TValue> values;
            values.emplace_back(4);
            helper.WriteSignalValues(timestamps[0], signalName, values);
            helper.CommitCurrentKeyData();
        }
        {
            // no pre-aggregates starting at timestamp 1
            const TVector<TInstanceKey> instanceKeysGeo = {
                TInstanceKey::FromNamed("testing|ctype=prestable;geo=man;prj=test|tier"),
                TInstanceKey::FromNamed("testing|ctype=prestable;geo=sas;prj=test|tier"),
            };

            for (const auto &instanceKey: instanceKeysGeo) {
                helper.StartWritingKeyData(instanceKey);
                TVector<TValue> values;
                values.emplace_back(1);
                values.emplace_back(2);
                helper.WriteSignalValues(timestamps[1], signalName, values);
                helper.CommitCurrentKeyData();
            }
        }
        {
            // partial pre-aggregate starting at timestamp 2
            helper.StartWritingKeyData(TInstanceKey::FromNamed("testing|ctype=prestable;prj=test|geo,tier"));
            TVector<TValue> values;
            values.emplace_back(4);
            helper.WriteSignalValues(timestamps[2], signalName, values);
            helper.CommitCurrentKeyData();
        }
        helper.Finish();

        TVector<TSignalName> signals(1, signalName);
        signals.emplace_back(TStringBuf("unistat-no_data_signal_dmmm"));
        TAbstractFormat::TTagSignals tags;
        for (const auto &requestString: requestStrings) {
            tags.emplace_back(
                TRequestKey::FromString(requestString),
                signals
            );
        }
        auto res = helper.Read(timestamps, tags);

        TVector<std::tuple<
            TAbstractFormat::TTimestamp,
            TVector<std::pair<TString, size_t>>,
            TString,
            TString
        >> expectedRes = {
            {
                1, {{"unistat-ok_subsource_VIDEO_dmmm", 4},}, "itype=testing", "testing||ctype,geo,prj,tier",
            }
        }; // we no longer try to read anything other than the best instance key group. in this case it is a full pre-aggregate.

        UNIT_ASSERT_VALUES_EQUAL(res.size(), expectedRes.size());

        for (size_t id : xrange(res.size())){
            const auto& row = res[id];
            const auto& expectedRow = expectedRes[id];

            UNIT_ASSERT_VALUES_EQUAL(row.Timestamp, std::get<0>(expectedRow));
            UNIT_ASSERT_VALUES_EQUAL(row.RequestKey.ToNamed(), std::get<2>(expectedRow));
            UNIT_ASSERT_VALUES_EQUAL(row.InstanceKey.ToNamed(), std::get<3>(expectedRow));

            const auto& vals = row.Record.GetValues();
            const auto& avals = std::get<1>(expectedRow);
            UNIT_ASSERT_VALUES_EQUAL(vals.size(), avals.size());
            for (const size_t vid : xrange(vals.size())) {
                UNIT_ASSERT_VALUES_EQUAL(vals[vid].first, avals[vid].first);
                UNIT_ASSERT_VALUES_EQUAL(vals[vid].second, TValue(avals[vid].second));
            }
        }
    }

    Y_UNIT_TEST(ReadMatchOnSinglePoint) {
        TSignalName signalName = TStringBuf("unistat-ok_subsource_VIDEO_dmmm");

        const TVector<TString> requestStrings = {
            "itype=testing;ctype=prod",
        };

        TVector<TAbstractFormat::TTimestamp> timestamps = {1, 2, 3};

        TCompactHelper helper;
        {
            helper.StartWritingKeyData(TInstanceKey::FromNamed("testing|ctype=prod;prj=test|geo,tier"));
            TVector<TValue> values;
            values.emplace_back(4);
            helper.WriteSignalValues(timestamps[0], signalName, values);
            helper.CommitCurrentKeyData();
        }
        {
            const TVector<TInstanceKey> instanceKeysGeo = {
                TInstanceKey::FromNamed("testing|ctype=prestable;geo=man;prj=test|tier"),
                TInstanceKey::FromNamed("testing|ctype=prestable;geo=sas;prj=test|tier"),
            };

            for (const auto &instanceKey: instanceKeysGeo) {
                helper.StartWritingKeyData(instanceKey);
                TVector<TValue> values;
                values.emplace_back(1);
                values.emplace_back(2);
                helper.WriteSignalValues(timestamps[1], signalName, values);
                helper.CommitCurrentKeyData();
            }
        }
        {
            // partial pre-aggregate starting at timestamp 2
            helper.StartWritingKeyData(TInstanceKey::FromNamed("testing|ctype=prod|geo,prj,tier"));
            TVector<TValue> values;
            values.emplace_back(4);
            helper.WriteSignalValues(timestamps[2], signalName, values);
            helper.CommitCurrentKeyData();
        }
        helper.Finish();

        TVector<TSignalName> signals(1, signalName);
        TAbstractFormat::TTagSignals tags;
        for (const auto &requestString: requestStrings) {
            tags.emplace_back(
                TRequestKey::FromString(requestString),
                signals
            );
        }
        auto res = helper.Read(timestamps, tags);
        TVector<std::tuple<
            TAbstractFormat::TTimestamp,
            TVector<std::pair<TString, size_t>>,
            TString,
            TString
        >> expectedRes = {
            {
                3, {{"unistat-ok_subsource_VIDEO_dmmm", 4},}, "itype=testing;ctype=prod", "testing|ctype=prod|geo,prj,tier",
            }
        };

        UNIT_ASSERT_VALUES_EQUAL(res.size(), expectedRes.size());

        for (size_t id : xrange(res.size())){
            const auto& row = res[id];
            const auto& expectedRow = expectedRes[id];

            UNIT_ASSERT_VALUES_EQUAL(row.Timestamp, std::get<0>(expectedRow));
            UNIT_ASSERT_VALUES_EQUAL(row.RequestKey.ToNamed(), std::get<2>(expectedRow));
            UNIT_ASSERT_VALUES_EQUAL(row.InstanceKey.ToNamed(), std::get<3>(expectedRow));

            const auto& vals = row.Record.GetValues();
            const auto& avals = std::get<1>(expectedRow);
            UNIT_ASSERT_VALUES_EQUAL(vals.size(), avals.size());
            for (const size_t vid : xrange(vals.size())) {
                UNIT_ASSERT_VALUES_EQUAL(vals[vid].first, avals[vid].first);
                UNIT_ASSERT_VALUES_EQUAL(vals[vid].second, TValue(avals[vid].second));
            }
        }
    }

    Y_UNIT_TEST(ReadNoMatch) {
        TSignalName signalName = TStringBuf("unistat-ok_subsource_VIDEO_dmmm");

        const TVector<TString> requestStrings = {
            "itype=testing;ctype=prod",
        };

        TVector<TAbstractFormat::TTimestamp> timestamps = {1, 2, 3};

        TCompactHelper helper;
        {
            // full pre-aggregate at timestamp 0, with no data at timestamps 1 and 2
            helper.StartWritingKeyData(TInstanceKey::FromNamed("testing||geo,ctype,prj,tier"));
            TVector<TValue> values;
            values.emplace_back(4);
            helper.WriteSignalValues(timestamps[0], signalName, values);
            helper.CommitCurrentKeyData();
        }
        {
            // no pre-aggregates starting at timestamp 1
            const TVector<TInstanceKey> instanceKeysGeo = {
                TInstanceKey::FromNamed("testing|ctype=prestable;geo=man;prj=test|tier"),
                TInstanceKey::FromNamed("testing|ctype=prestable;geo=sas;prj=test|tier"),
            };

            for (const auto &instanceKey: instanceKeysGeo) {
                helper.StartWritingKeyData(instanceKey);
                TVector<TValue> values;
                values.emplace_back(1);
                values.emplace_back(2);
                helper.WriteSignalValues(timestamps[1], signalName, values);
                helper.CommitCurrentKeyData();
            }
        }
        {
            // partial pre-aggregate starting at timestamp 2
            helper.StartWritingKeyData(TInstanceKey::FromNamed("testing|ctype=prod;prj=test|geo,tier"));
            TVector<TValue> values;
            values.emplace_back(4);
            helper.WriteSignalValues(timestamps[2], signalName, values);
            helper.CommitCurrentKeyData();
        }
        helper.Finish();

        TVector<TSignalName> signals(1, signalName);
        TAbstractFormat::TTagSignals tags;
        for (const auto &requestString: requestStrings) {
            tags.emplace_back(
                TRequestKey::FromString(requestString),
                signals
            );
        }
        timestamps.pop_back(); // don't request the last timestamp -- the one where we have a match
        auto res = helper.Read(timestamps, tags);
        UNIT_ASSERT(res.empty());
    }

    Y_UNIT_TEST(ReadPreaggrKillsTheMatch) {
        TSignalName signalName = TStringBuf("unistat-ok_subsource_VIDEO_dmmm");

        const TVector<TString> requestStrings = {
            "itype=testing",
        };

        TVector<TAbstractFormat::TTimestamp> timestamps = {1, 2, 3};

        TCompactHelper helper;
        {
            // no pre-aggregates starting at timestamp 0
            const TVector<TInstanceKey> instanceKeysGeo = {
                TInstanceKey::FromNamed("testing|ctype=prestable;geo=man;prj=test|tier"),
                TInstanceKey::FromNamed("testing|ctype=prestable;geo=sas;prj=test|tier"),
            };

            for (const auto &instanceKey: instanceKeysGeo) {
                helper.StartWritingKeyData(instanceKey);
                TVector<TValue> values;
                values.emplace_back(1);
                values.emplace_back(2);
                values.emplace_back(2);
                helper.WriteSignalValues(timestamps[0], signalName, values);
                helper.CommitCurrentKeyData();
            }
        }
        {
            // partial pre-aggregate starting at timestamp 1
            helper.StartWritingKeyData(TInstanceKey::FromNamed("testing|ctype=prod;prj=test|geo,tier"));
            TVector<TValue> values;
            values.emplace_back(4);
            helper.WriteSignalValues(timestamps[1], signalName, values);
            helper.CommitCurrentKeyData();
        }
        {
            // full pre-aggregate at timestamp 2
            helper.StartWritingKeyData(TInstanceKey::FromNamed("testing||geo,ctype,prj,tier"));
            TVector<TValue> values;
            values.emplace_back(4);
            helper.WriteSignalValues(timestamps[2], signalName, values);
            helper.CommitCurrentKeyData();
        }
        helper.Finish();

        TVector<TSignalName> signals(1, signalName);
        TAbstractFormat::TTagSignals tags;
        for (const auto &requestString: requestStrings) {
            tags.emplace_back(
                TRequestKey::FromString(requestString),
                signals
            );
        }
        timestamps.pop_back(); // don't request the last timestamp -- the one where we have a best match
        auto res = helper.Read(timestamps, tags);
        UNIT_ASSERT(res.empty()); // we do not complicate things, just return empty result
    }
}
