#include "something.h"
#include "something_loader.h"
#include "something_saver.h"
#include "something_reader.h"
#include "something_matcher.h"
#include "something_iterator.h"
#include "something_key_iterator.h"
#include "something_writer.h"

#include <infra/yasm/zoom/components/serialization/deserializers/msgpack.h>

#include <util/generic/xrange.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>

using namespace NHistDb;

namespace {
    static constexpr size_t MAX_INSTANCES_TO_READ = 1000;
    static constexpr THeaderVersionStorage HEADER_SOMETHING_VERSION {{1, 0, 0, 0}};
    static constexpr TDuration READ_TIMEOUT = TDuration::Seconds(25);

    inline size_t GetUncompressedSize(const TVector<TSnappyBlock>& blocks) noexcept {
        size_t size = 0;
        for (const auto& block : blocks) {
            size += block.UncompressedSize;
        }
        return size;
    }

    inline TMaybe<TStringBuf> BlobToBuf(TMaybe<TBlob> incoming) {
        if (incoming) {
            return TStringBuf(incoming->AsCharPtr(), incoming->Size());
        } else {
            return Nothing();
        }
    }
}

TSomethingFormat::TSomethingFormat()
{
}

TSomethingFormat::TSomethingFormat(TMaybe<TStringBuf> incoming)
{
    if (incoming) {
        TLoader reader(*incoming, *this);
        reader.Read();
        FillSizes();
    }
}

TSomethingFormat::TSomethingFormat(TMaybe<TBlob> incoming)
    : TSomethingFormat(BlobToBuf(incoming))
{
}

void TSomethingFormat::FillSizes() {
    size_t elements = 0;
    for (const auto& row : TimeOffsets) {
        elements += row.size();
    }

    SizeOffsets.resize(elements);
    auto it(SizeOffsets.begin());
    for (const auto& row : TimeOffsets) {
        for (const auto position : row) {
            *it = position;
            ++it;
        }
    }
    Sort(SizeOffsets);

    TotalSize = GetUncompressedSize(Blocks);
}

TString TSomethingFormat::Dump() const {
    TSaver writer(*this);
    return writer.Dump();
}

void TSomethingFormat::WriteRecord(
    TTimestamp timestamp,
    const TInstanceKey key,
    const TRecord& record,
    TSnappyOutputStream& stream
) {
    TWriter writer(*this, stream);
    writer.WriteRecord(timestamp, key, record);
}

TSomethingBulkWriter TSomethingFormat::BulkWriter(TSnappyOutputStream& stream) {
    return {*this, stream};
}

TVector<bool> TSomethingFormat::HasRecords(
    const TVector<TTimestamp>& timestamps,
    const TVector<TInstanceKey>& keys
) const {
    TMatcher matcher(*this, timestamps, keys);
    return matcher.Match();
}

void TSomethingFormat::HasRecords(
    const TVector<TTimestamp>& timestamps,
    const TVector<TInstanceKey>& keys,
    THashSet<TTimestamp>& result
) const {
    TMatcher matcher(*this, timestamps, keys);
    return matcher.Match(result);
}

TVector<TSomethingFormat::TReadRow> TSomethingFormat::ReadRecords(
    const TVector<TTimestamp>& timestamps,
    const TVector<std::pair<TInstanceKey, TVector<TSignalName>>>& keysAndSignals,
    TSnappyInputStream& stream,
    TInstant deadline
) const {
    TReader extractor(*this, timestamps, keysAndSignals, stream, deadline);
    return extractor.Read();
}

TVector<TSomethingFormat::TReadRow> TSomethingFormat::ReadRecords(
    const TVector<TTimestamp>& timestamps,
    const TVector<std::pair<TInstanceKey, TVector<TSignalName>>>& keysAndSignals,
    TSnappyInputStream& stream
) const {
    return ReadRecords(timestamps, keysAndSignals, stream, READ_TIMEOUT.ToDeadLine());
}

TSomethingIterator TSomethingFormat::IterateRecords(TSnappyInputStream& stream) const {
    TVector<TTimestamp> timestamps;
    for (const auto ts : xrange(TimeOffsets.size())) {
        timestamps.emplace_back(ts);
    }
    return {*this, timestamps, stream};
}

TSomethingIterator TSomethingFormat::IterateRecords(const TVector<TTimestamp>& timestamps, TSnappyInputStream& stream) const {
    return {*this, timestamps, stream};
}

TSomethingKeyIterator TSomethingFormat::IterateKeys() const {
    return {*this};
}

TVector<TSomethingFormat::TTimestamp> TSomethingFormat::GetTimes() const {
    TVector<TTimestamp> times;
    times.reserve(TimeOffsets.size());

    for (const auto timeIndex : xrange(TimeOffsets.size())) {
        if (!TimeOffsets[timeIndex].empty()) {
            times.emplace_back(timeIndex);
        }
    }
    return times;
}

void TSomethingFormat::SaveBlocks(const TVector<TSnappyBlock>& blocks) {
    Blocks = blocks;
}

const TVector<TSnappyBlock>& TSomethingFormat::GetBlocks() const {
    return Blocks;
}

size_t TSomethingFormat::FindSize(size_t offset) const {
    const auto it(LowerBound(SizeOffsets.begin(), SizeOffsets.end(), offset));
    if (it != SizeOffsets.end() && *it == offset) {
        const auto nextIt(it + 1);
        if (nextIt != SizeOffsets.end()) {
            return *nextIt - *it;
        } else if (nextIt == SizeOffsets.end()) {
            return TotalSize - *it;
        }
    }
    ythrow yexception() << "can't find given offset";
}

TMaybe<TSomethingFormat::TTimestamp> TSomethingFormat::FirstRecordTime() const {
    for (const auto timeIndex : xrange(TimeOffsets.size())) {
        if (!TimeOffsets[timeIndex].empty()) {
            return timeIndex;
        }
    }
    return Nothing();
}

TMaybe<TSomethingFormat::TTimestamp> TSomethingFormat::LastRecordTime() const {
    if (!TimeOffsets.empty()) {
        return TimeOffsets.size() - 1;
    } else {
        return Nothing();
    }
}

TVector<TSomethingFormat::TReadData> TSomethingFormat::Read(
    const TVector<TTimestamp>& times,
    const TTagSignals& tags,
    TSnappyInputStream& stream
) const {

    using TRequestData = std::tuple<
            const TRequestKey&,
            TInstanceKey,
            size_t,
            TRequestKeyIndex>; //request name, instance name, matched set id, request key index

    const auto deadline = READ_TIMEOUT.ToDeadLine();

    TVector<TRequestData> requests;
    TVector<std::pair<TInstanceKey, TVector<TSignalName>>> rawKeysAndSignals;
    TVector<THashSet<TTimestamp>> matchSets;

    for (const auto requestKeyIndex: xrange(tags.size())) {
        const auto& tagVal = tags[requestKeyIndex];
        const auto& tag = tagVal.first;
        NTags::TDynamicFilter filter(tag);
        filter.FeedMany(Keys);

        auto resolvedInstanceKeys = filter.Resolve();
        if (resolvedInstanceKeys.size() > MAX_INSTANCES_TO_READ) {
            ythrow TSignalLimitExceeded() << "can't read more than " << MAX_INSTANCES_TO_READ << " instances per query ("
                << resolvedInstanceKeys.size() << " to read)";
        }

        // ReadRecords
        const auto matcherId = matchSets.size();
        auto& matchesSet = matchSets.emplace_back();

        HasRecords(times, resolvedInstanceKeys, matchesSet);

        if (matchesSet.empty()) {
            matchSets.pop_back();
            continue;
        }

        for (const auto& instanceKey : resolvedInstanceKeys) {
            requests.emplace_back(tag, instanceKey, matcherId, requestKeyIndex);
            rawKeysAndSignals.emplace_back(instanceKey, tagVal.second);
        }
    }

    TVector<TSomethingFormat::TReadData> result;

    if (requests.empty()) {
        return result;
    }

    for (auto& smth : ReadRecords(times, rawKeysAndSignals, stream, deadline)) {
        auto timestamp = std::get<0>(smth);
        auto requestedIndex = std::get<1>(smth);
        auto record = std::move(std::get<2>(smth));

        const TRequestData& request = requests[requestedIndex];

        auto& requestKey = std::get<0>(request);
        auto& instanceKey = std::get<1>(request);
        auto matchedSetId = std::get<2>(request);
        auto requestKeyIndex = std::get<3>(request);

        auto& matchesSet = matchSets[matchedSetId];
        if (matchesSet.contains(timestamp)) {
            result.emplace_back(requestKey, std::move(record), timestamp, instanceKey, requestKeyIndex);
        }
    }
    return result;
}

THeaderVersionStorage TSomethingFormat::GetVersion() const {
    return GetSomethingVersion();
}

THeaderVersionStorage TSomethingFormat::GetSomethingVersion() {
    return HEADER_SOMETHING_VERSION;
}

TSomethingIterator::TSomethingIterator(
    const TSomethingFormat& format,
    const TVector<TSomethingFormat::TTimestamp>& timestamps,
    TSnappyInputStream& stream
)
    : Impl(MakeHolder<TSomethingFormat::TIterator>(format, timestamps, stream))
{
}

TSomethingIterator::TSomethingIterator(TSomethingIterator&& other)
    : Impl(std::move(other.Impl))
{
}

TSomethingIterator::~TSomethingIterator() {
}

bool TSomethingIterator::Next() {
    return Impl->Next();
}

TSomethingFormat::TIteratorRow& TSomethingIterator::Get() {
    return Impl->Get();
}

TSomethingIterator& TSomethingIterator::operator=(TSomethingIterator&& other) {
    Impl = std::move(other.Impl);
    return *this;
}

TSomethingKeyIterator::TSomethingKeyIterator(const TSomethingFormat& format)
    : Impl(MakeHolder<TSomethingFormat::TKeyIterator>(format))
{
}

TSomethingKeyIterator::TSomethingKeyIterator(TSomethingKeyIterator&& other)
    : Impl(std::move(other.Impl))
{
}

TSomethingKeyIterator::~TSomethingKeyIterator() {
}

bool TSomethingKeyIterator::Next() {
    return Impl->Next();
}

TSomethingFormat::TKeyIteratorRow& TSomethingKeyIterator::Get() {
    return Impl->Get();
}

TSomethingKeyIterator& TSomethingKeyIterator::operator=(TSomethingKeyIterator&& other) {
    Impl = std::move(other.Impl);
    return *this;
}

TSomethingBulkCursor::TSomethingBulkCursor(const size_t keyIndex, const size_t signalIndex)
    : KeyIndex(keyIndex)
    , SignalIndex(signalIndex)
{
}

TSomethingBulkCursor::TSomethingBulkCursor(const TSomethingBulkCursor& other)
    : TSomethingBulkCursor(other.KeyIndex, other.SignalIndex)
{
}

void TSomethingBulkCursor::Insert(TSomethingBulkWriter& writer, TSomethingFormat::TTimestamp timestamp, NZoom::NValue::TValueRef value) const {
    writer.InsertValue(KeyIndex, SignalIndex, timestamp, value);
}

TSomethingBulkWriter::TSomethingBulkWriter(TSomethingFormat& format, TSnappyOutputStream& stream)
    : Impl(MakeHolder<TSomethingFormat::TBulkWriter>(format, stream))
{
}

TSomethingBulkWriter::TSomethingBulkWriter(TSomethingBulkWriter&& other)
    : Impl(std::move(other.Impl))
{
}

TSomethingBulkWriter::~TSomethingBulkWriter() {
}

TSomethingBulkCursor TSomethingBulkWriter::CreateSeries(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal) {
    return Impl->CreateSeries(key, signal);
}

void TSomethingBulkWriter::InsertValue(const size_t keyIndex, const size_t signalIndex,
                                        TSomethingFormat::TTimestamp timestamp, NZoom::NValue::TValueRef value) {
    Impl->InsertValue(keyIndex, signalIndex, timestamp, value);
}

void TSomethingBulkWriter::Flush(TSomethingFormat::TTimestamp upperBorder) {
    Impl->Flush(upperBorder);
}

void TSomethingBulkWriter::Flush() {
    Impl->Flush();
}
