#pragma once

#include "base.h"

#include <util/generic/strbuf.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>

#include <contrib/libs/msgpack/include/msgpack.hpp>

namespace NHistDb {
    class TSomethingIterator;
    class TSomethingKeyIterator;
    class TSomethingBulkWriter;

    class TSomethingFormat final : public TAbstractFormat {
    private:
        friend TSomethingIterator;
        friend TSomethingKeyIterator;
        friend TSomethingBulkWriter;

        using TInstanceKey = NTags::TInstanceKey;
        using TRequestKey = NTags::TRequestKey;
        using TSignalName = NZoom::NSignal::TSignalName;
        using TValue = NZoom::NValue::TValue;
        using TRecord = NZoom::NRecord::TRecord;
        using TYasmConf = NZoom::NYasmConf::TYasmConf;
        using TMatchedInstances = NZoom::NAggregators::TMetagroupAggregator::TMatchedInstances;

    public:
        using TTimestamp = ui64;
        using TRequestedIndex = ui64;
        using TRequestKeyIndex = size_t;

        using TTagSignals = TVector<std::pair<TRequestKey, TVector<TSignalName>>>;

        // timestamp, position in keysAndSignals, record
        using TReadRow = std::tuple<TTimestamp, TRequestedIndex, TRecord>;
        using TIteratorRow = std::tuple<TTimestamp, TInstanceKey, TRecord>;
        using TKeyIteratorRow = std::pair<TInstanceKey, const TVector<TSignalName>*>;

        TSomethingFormat();
        TSomethingFormat(TMaybe<TStringBuf> incoming);
        TSomethingFormat(TMaybe<TBlob> incoming);

        TString Dump() const override;

        void WriteRecord(
            TTimestamp timestamp,
            const TInstanceKey key,
            const TRecord& record,
            TSnappyOutputStream& stream
        );

        TSomethingBulkWriter BulkWriter(TSnappyOutputStream& stream);

        TVector<bool> HasRecords(
            const TVector<TTimestamp>& timestamps,
            const TVector<TInstanceKey>& keys
        ) const;

        void HasRecords(
            const TVector<TTimestamp>& timestamps,
            const TVector<TInstanceKey>& keys,
            THashSet<TTimestamp>& result
        ) const;

        TVector<TReadRow> ReadRecords(
            const TVector<TTimestamp>& timestamps,
            const TVector<std::pair<TInstanceKey, TVector<TSignalName>>>& keysAndSignals,
            TSnappyInputStream& stream,
            TInstant deadline
        ) const;

        TVector<TReadRow> ReadRecords(
            const TVector<TTimestamp>& timestamps,
            const TVector<std::pair<TInstanceKey, TVector<TSignalName>>>& keysAndSignals,
            TSnappyInputStream& stream
        ) const;

        TSomethingIterator IterateRecords(TSnappyInputStream& stream) const;
        TSomethingIterator IterateRecords(const TVector<TTimestamp>& timestamps, TSnappyInputStream& stream) const;

        TSomethingKeyIterator IterateKeys() const;
        TVector<TTimestamp> GetTimes() const;

        void SaveBlocks(const TVector<TSnappyBlock>& blocks) override;
        const TVector<TSnappyBlock>& GetBlocks() const override;

        size_t FindSize(size_t offset) const;

        TMaybe<TTimestamp> FirstRecordTime() const override;
        TMaybe<TTimestamp> LastRecordTime() const override;

        TVector<TReadData> Read(
                const TVector<TTimestamp>& times,
                const TTagSignals& tags,
                TSnappyInputStream& stream
        ) const override;

        THeaderVersionStorage GetVersion() const override;
        static THeaderVersionStorage GetSomethingVersion();

    private:
        class TLoader;
        class TSaver;
        class TReader;
        class TMatcher;
        class TIterator;
        class TKeyIterator;
        class TWriter;
        class TBulkWriter;

        using TOffsets = TVector<TVector<ui64>>;

        void FillSizes();
        bool OffsetsContains(const TVector<ui64>& needle, const TVector<ui64>& offsets);

        // See algorithm description:
        // https://wiki.yandex-team.ru/users/knsd/histdb/another-format/final/#algoritmdopisyvanijavsushhestvujushhijjchanklibosozdanijanovogochankaszapisjamirkeytsrecordsv

        TVector<TSnappyBlock> Blocks;

        TVector<TInstanceKey> Keys;
        THashMap<TInstanceKey, size_t> KeyToIndex;

        TVector<std::pair<
            THashMap<TSignalName, size_t>,
            TVector<TSignalName>
        >> Signals;

        TOffsets KeyOffsets;
        TOffsets TimeOffsets;

        TVector<ui64> SizeOffsets;
        ui64 TotalSize = 0;

        msgpack::sbuffer Buffer;
    };

    class TSomethingIterator {
    public:
        TSomethingIterator(
            const TSomethingFormat& format,
            const TVector<TSomethingFormat::TTimestamp>& timestamps,
            TSnappyInputStream& stream
        );
        TSomethingIterator(TSomethingIterator&& other);

        ~TSomethingIterator();

        bool Next();
        TSomethingFormat::TIteratorRow& Get();

        TSomethingIterator& operator=(TSomethingIterator&&);

    private:
        THolder<TSomethingFormat::TIterator> Impl;
    };

    class TSomethingKeyIterator {
    public:
        TSomethingKeyIterator(const TSomethingFormat& format);
        TSomethingKeyIterator(TSomethingKeyIterator&& other);

        ~TSomethingKeyIterator();

        bool Next();
        TSomethingFormat::TKeyIteratorRow& Get();

        TSomethingKeyIterator& operator=(TSomethingKeyIterator&&);

    private:
        THolder<TSomethingFormat::TKeyIterator> Impl;
    };

    class TSomethingBulkCursor {
    public:
        TSomethingBulkCursor(const size_t keyIndex, const size_t signalIndex);
        TSomethingBulkCursor(const TSomethingBulkCursor& other);

        void Insert(TSomethingBulkWriter& writer, TSomethingFormat::TTimestamp timestamp, NZoom::NValue::TValueRef value) const;

    private:
        const size_t KeyIndex;
        const size_t SignalIndex;
    };

    class TSomethingBulkWriter {
    public:
        TSomethingBulkWriter(TSomethingFormat& format, TSnappyOutputStream& stream);
        TSomethingBulkWriter(TSomethingBulkWriter&& other);
        ~TSomethingBulkWriter();

        TSomethingBulkCursor CreateSeries(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signal);
        void InsertValue(const size_t keyIndex, const size_t signalIndex,
                         TSomethingFormat::TTimestamp timestamp, NZoom::NValue::TValueRef value);
        void Flush(TSomethingFormat::TTimestamp upperBorder);
        void Flush();

    private:
        THolder<TSomethingFormat::TBulkWriter> Impl;
    };
}
