#pragma once

#include "something.h"

#include <infra/yasm/zoom/components/serialization/deserializers/msgpack.h>

#include <util/generic/xrange.h>

namespace NHistDb {
    class TSomethingFormat::TIterator {
    public:
        TIterator(
            const TSomethingFormat& format,
            const TVector<TTimestamp>& timestamps,
            TSnappyInputStream& stream
        )
            : Format(format)
            , Timestamps(timestamps)
            , Stream(stream)
            , KeyIterator(Format.KeyOffsets.cbegin())
            , OffsetKeyTimeIterator(OffsetKeyTime.cend())
        {
            ComputeTimeOffsets();
            ComputeSizes();
            FillOffsetKeyTime();
        }

        bool Next() {
            do {
                if (!AdvanceIterator()) {
                    return false;
                }

                const TKeyIndex keyIndex(std::get<0>(*OffsetKeyTimeIterator));
                const TDataOffset offset(std::get<1>(*OffsetKeyTimeIterator));
                const TTimeIndex timeIndex(std::get<2>(*OffsetKeyTimeIterator));
                const TSize size(std::get<3>(*OffsetKeyTimeIterator));
                Y_VERIFY(size > 0);

                DataBuffer.Resize(size);
                Stream.Seek(offset);
                Stream.Read(DataBuffer.Data(), DataBuffer.Size());

                size_t unpacked = 0;
                const auto message(msgpack::unpack(DataBuffer.Data(), DataBuffer.Size(), unpacked));
                if (unpacked != DataBuffer.Size()) {
                    ythrow yexception() << "extra bytes in buffer";
                }

                const msgpack::object_array messageArray(message.get().via.array);

                TVector<std::pair<TSignalName, TValue>> values;
                values.reserve(messageArray.size);
                for (const auto signalIndex : xrange(messageArray.size)) {
                    values.emplace_back(
                        Format.Signals[keyIndex].second[signalIndex],
                        NZoom::NPython::TMsgPackValueHierarchy::DeserializeValue(messageArray.ptr[signalIndex], true)
                    );
                }

                CurrentRow.ConstructInPlace(timeIndex, Format.Keys[keyIndex], std::move(values));
                ++OffsetKeyTimeIterator;

            } while (std::get<1>(*CurrentRow).Empty());

            return true;
        }

        TIteratorRow& Get() {
            return CurrentRow.GetRef();
        }

    private:
        using TKeyIndex = ui64;
        using TTimeIndex = ui64;
        using TDataOffset = ui64;
        using TSize = ui64;

        using TOffsetKeyTimeVector = TVector<std::tuple<TKeyIndex, TDataOffset, TTimeIndex, TSize>>;

        void ComputeTimeOffsets() {
            OffsetMap.reserve(Format.SizeOffsets.size());

            for (const auto timeIndex : Timestamps) {
                if (timeIndex < Format.TimeOffsets.size()) {
                    for (const auto dataOffset : Format.TimeOffsets[timeIndex]) {
                        OffsetMap[dataOffset].TimeIndex = timeIndex;
                    }
                }
            }
        }

        void ComputeSizes() {
            if (Format.SizeOffsets.size() >= 2) {
                auto first(Format.SizeOffsets.begin());
                auto second(Format.SizeOffsets.begin());
                ++second;

                for (; second != Format.SizeOffsets.end(); ++first, ++second) {
                    auto dest(OffsetMap.find(*first));
                    if (dest != OffsetMap.end()) {
                        dest->second.Size = *second - *first;
                    }
                }
            }

            if (!Format.SizeOffsets.empty()) {
                auto dest(OffsetMap.find(Format.SizeOffsets.back()));
                if (dest != OffsetMap.end()) {
                    dest->second.Size = Format.TotalSize - Format.SizeOffsets.back();
                }
            }
        }

        void FillOffsetKeyTime() {
            if (KeyIterator == Format.KeyOffsets.cend()) {
                return;
            }

            OffsetKeyTime.clear();

            TKeyIndex keyIndex = std::distance(Format.KeyOffsets.cbegin(), KeyIterator);
            for (const auto offset : *KeyIterator) {
                const auto offsetInfo(OffsetMap.find(offset));
                if (offsetInfo != OffsetMap.end()) {
                    OffsetKeyTime.emplace_back(keyIndex, offset, offsetInfo->second.TimeIndex, offsetInfo->second.Size);
                }
            }

            Sort(OffsetKeyTime);
            OffsetKeyTimeIterator = OffsetKeyTime.cbegin();
        }

        bool AdvanceIterator() {
            while (OffsetKeyTimeIterator == OffsetKeyTime.cend() && KeyIterator != Format.KeyOffsets.cend()) {
                ++KeyIterator;
                FillOffsetKeyTime();
            }
            return OffsetKeyTimeIterator != OffsetKeyTime.cend();
        }

        const TSomethingFormat& Format;
        const TVector<TTimestamp>& Timestamps;
        TSnappyInputStream& Stream;

        struct TOffsetInfo {
            ui64 Size = 0;
            TTimeIndex TimeIndex = 0;
        };

        THashMap<TDataOffset, TOffsetInfo> OffsetMap;

        TSomethingFormat::TOffsets::const_iterator KeyIterator;
        TOffsetKeyTimeVector OffsetKeyTime;
        TOffsetKeyTimeVector::const_iterator OffsetKeyTimeIterator;

        TBuffer DataBuffer;
        TMaybe<TIteratorRow> CurrentRow;
    };
}
