#pragma once

#include "something.h"

#include <infra/monitoring/common/msgpack.h>

#include <library/cpp/blockcodecs/codecs.h>

namespace NHistDb {
    class TSomethingFormat::TLoader {
    public:
        TLoader(TStringBuf incoming, TSomethingFormat& header)
            : Decoded(NBlockCodecs::Codec("snappy")->Decode(incoming))
            , Message(msgpack::unpack(Zone, Decoded.data(), Decoded.size(), Unpacked))
            , Format(header)
        {
            if (Unpacked != Decoded.size()) {
                ythrow yexception() << "extra bytes in buffer";
            }
        }

        void Read() {
            NMonitoring::EnsureIs(Message, msgpack::type::object_type::ARRAY);
            const auto root(Message.via.array);
            if (root.size == 5) {
                ReadBlocks(root.ptr[0]);
                ReadKeys(root.ptr[1]);
                ReadSignals(root.ptr[2]);
                ReadKeyOffset(root.ptr[3]);
                ReadTimeOffset(root.ptr[4]);
            } else if (root.size == 6) {
                ReadBlocks(root.ptr[0]);
                ReadKeys(root.ptr[1]);
                ReadSignalsTable(root.ptr[3]);
                ReadSignalsUnified(root.ptr[2]);
                ReadKeyOffset(root.ptr[4]);
                ReadTimeOffset(root.ptr[5]);
            } else {
                ythrow yexception() << "not enough elements in array";
            }
        }

    private:
        void ReadBlocks(const msgpack::object blocks) {
            NMonitoring::EnsureIs(blocks, msgpack::type::object_type::ARRAY);
            const auto incoming(blocks.via.array);
            Format.Blocks.resize(incoming.size);
            auto it(Format.Blocks.begin());
            for (const auto& block : NMonitoring::TArrayIterator(incoming)) {
                const auto element(block.via.array);
                it->UncompressedSize = element.ptr[0].via.u64;
                it->CompressedSize = element.ptr[1].via.u64;
                ++it;
            }
        }

        inline TInstanceKey ParseKey(const msgpack::object rawKey) {
            if (rawKey.is_nil()) {
                return {};
            } else if (rawKey.type == msgpack::type::object_type::BIN || rawKey.type == msgpack::type::object_type::STR) {
                try {
                    return TInstanceKey::FromNamed(rawKey.as<TStringBuf>());
                } catch(...) {
                    return {};
                }
            } else if (rawKey.type == msgpack::type::object_type::ARRAY) {
                TVector<std::pair<TStringBuf, TStringBuf>> pairs;
                pairs.reserve(rawKey.via.array.size);
                for (const auto& tagValue : NMonitoring::TArrayIterator(rawKey.via.array)) {
                    if (tagValue.type != msgpack::type::object_type::ARRAY || tagValue.via.array.size != 2) {
                        return {};
                    }
                    pairs.emplace_back(
                        tagValue.via.array.ptr[0].as<TStringBuf>(),
                        tagValue.via.array.ptr[1].as<TStringBuf>()
                    );
                }
                try {
                    return TInstanceKey::FromHistDb(pairs);
                } catch(...) {
                    return {};
                }
            } else {
                return {};
            }
        }

        void ReadKeys(const msgpack::object keys) {
            NMonitoring::EnsureIs(keys, msgpack::type::object_type::ARRAY);
            const auto incoming(keys.via.array);
            Format.Keys.reserve(incoming.size);
            Format.KeyToIndex.reserve(incoming.size);
            for (const auto& rawKey : NMonitoring::TArrayIterator(incoming)) {
                TInstanceKey key(ParseKey(rawKey));
                Format.KeyToIndex.emplace(key, Format.Keys.size());
                Format.Keys.emplace_back(key);
            }
        }

        void ReadSignals(const msgpack::object signals) {
            NMonitoring::EnsureIs(signals, msgpack::type::object_type::ARRAY);
            const auto incoming(signals.via.array);
            Format.Signals.resize(incoming.size);
            auto it(Format.Signals.begin());
            for (const auto& signalsInKey : NMonitoring::TArrayIterator(incoming)) {
                it->first.reserve(signalsInKey.via.array.size);
                it->second.reserve(signalsInKey.via.array.size);
                for (const auto& signalName : NMonitoring::TArrayIterator(signalsInKey.via.array)) {
                    const TSignalName signal(signalName.as<TStringBuf>());
                    it->first.emplace(signal, it->second.size());
                    it->second.emplace_back(signal);
                }
                ++it;
            }
        }

        void ReadSignalsTable(const msgpack::object signals) {
            NMonitoring::EnsureIs(signals, msgpack::type::object_type::ARRAY);
            const auto incoming(signals.via.array);
            KnownSignals.reserve(incoming.size);
            for (const auto& signalName : NMonitoring::TArrayIterator(incoming)) {
                KnownSignals.emplace_back(signalName.as<TStringBuf>());
            }
        }

        void ReadSignalsUnified(const msgpack::object signals) {
            NMonitoring::EnsureIs(signals, msgpack::type::object_type::ARRAY);
            const auto incoming(signals.via.array);
            Format.Signals.resize(incoming.size);
            auto it(Format.Signals.begin());
            for (const auto& signalsInKey : NMonitoring::TArrayIterator(incoming)) {
                it->first.reserve(signalsInKey.via.array.size);
                it->second.reserve(signalsInKey.via.array.size);
                for (const auto& signalIndex : NMonitoring::TArrayIterator(signalsInKey.via.array)) {
                    const auto signal(KnownSignals[signalIndex.as<size_t>()]);
                    it->first.emplace(signal, it->second.size());
                    it->second.emplace_back(signal);
                }
                ++it;
            }
        }

        void FillOffsets(const msgpack::object offsets, TOffsets& outgoing) {
            NMonitoring::EnsureIs(offsets, msgpack::type::object_type::ARRAY);
            const auto incoming(offsets.via.array);
            outgoing.resize(incoming.size);
            auto rootIt(outgoing.begin());
            for (const auto& row : NMonitoring::TArrayIterator(incoming)) {
                rootIt->resize(row.via.array.size);
                //std::memcpy(rootIt->data(), reinterpret_cast<void*>(row.via.array.ptr), row.via.array.size * sizeof(ui64));
                auto rowIt(rootIt->begin());
                for (const auto& offset : NMonitoring::TArrayIterator(row.via.array)) {
                    *rowIt = offset.via.u64;
                    ++rowIt;
                }
                ++rootIt;
            }
        }

        void ReadKeyOffset(const msgpack::object offsets) {
            FillOffsets(offsets, Format.KeyOffsets);
        }

        void ReadTimeOffset(const msgpack::object offsets) {
            FillOffsets(offsets, Format.TimeOffsets);
        }

        const TString Decoded;
        size_t Unpacked = 0;
        msgpack::zone Zone;
        msgpack::object Message;
        TSomethingFormat& Format;
        TVector<TSignalName> KnownSignals;
    };
}
