#pragma once

#include "something.h"

#include <infra/monitoring/common/msgpack.h>
#include <infra/yasm/zoom/components/serialization/deserializers/msgpack.h>

namespace NHistDb {
    class TSomethingFormat::TReader {
    public:
        TReader(
            const TSomethingFormat& header,
            const TVector<TTimestamp>& timestamps,
            const TVector<std::pair<TInstanceKey, TVector<TSignalName>>>& keysAndSignals,
            TSnappyInputStream& stream,
            TInstant deadline
        )
            : Format(header)
            , Timestamps(timestamps)
            , KeysAndSignals(keysAndSignals)
            , Stream(stream)
            , Deadline(deadline)
        {
            FillKeyIndices();
            FillSignalIndices();
            FillOffsetKeyTimeIndices();
        }

        TVector<std::tuple<TTimestamp, TRequestedIndex, TRecord>> Read() {
            for (const auto& quadruple : OffsetKeyTimeIndices) {
                const TDataOffset offset(std::get<0>(quadruple));
                const TKeyIndex keyIndex(std::get<1>(quadruple));
                const TRequestedIndex requestedIndex(std::get<2>(quadruple));
                const TTimeIndex timeIndex(std::get<3>(quadruple));

                const auto signalIt(KeySignalIndices.find(std::make_pair(keyIndex, requestedIndex)));
                if (signalIt != KeySignalIndices.end() && !signalIt->second.empty()) {
                    const auto& values(ReadFromOffset(offset));
                    NMonitoring::EnsureIs(values, msgpack::type::object_type::ARRAY);
                    DecodeValues(timeIndex, keyIndex, requestedIndex, signalIt->second, values.via.array);
                }
            }

            return std::move(Records);
        }

    private:
        using TKeyIndex = ui64;
        using TTimeIndex = ui64;
        using TDataOffset = ui64;
        using TSignalOffset = ui64;

        void FillKeyIndices() {
            // find appropriate key indices
            ui64 position = 0;
            for (const auto& key : KeysAndSignals) {
                const auto it(Format.KeyToIndex.find(key.first));
                if (it != Format.KeyToIndex.end()) {
                    KeySignalIndices.emplace(std::make_pair(it->second, position), TVector<TSignalOffset>());
                }
                ++position;
            }
        }

        void FillSignalIndices() {
            for (auto& pair : KeySignalIndices) {
                const auto& keyPair(pair.first);
                const auto& existingSignals(Format.Signals[keyPair.first].first);
                for (const auto& signal : KeysAndSignals[keyPair.second].second) {
                    const auto it(existingSignals.find(signal));
                    if (it != existingSignals.end()) {
                        pair.second.emplace_back(it->second);
                    }
                }
                Sort(pair.second);
            }
        }

        void FillOffsetKeyTimeIndices() {
            TVector<TDataOffset> intersectedOffsets;
            for (const auto timeIndex : Timestamps) {
                if (timeIndex >= Format.TimeOffsets.size()) {
                    continue;
                }
                for (const auto& pair : KeySignalIndices) {
                    const auto& keyPair(pair.first);
                    const auto& keyOffsets(Format.KeyOffsets[keyPair.first]);
                    const auto& timeOffsets(Format.TimeOffsets[timeIndex]);
                    intersectedOffsets.clear();
                    SetIntersection(keyOffsets.cbegin(), keyOffsets.cend(),
                                    timeOffsets.cbegin(), timeOffsets.cend(),
                                    std::back_inserter(intersectedOffsets));
                    for (const auto offset : intersectedOffsets) {
                        OffsetKeyTimeIndices.emplace_back(offset, keyPair.first, keyPair.second, timeIndex);
                    }
                }
            }
            if (OffsetKeyTimeIndices.size() > 100000) {
                ythrow TSignalLimitExceeded() << "too many offsets (" << OffsetKeyTimeIndices.size() << ") to read";
            }
            Sort(OffsetKeyTimeIndices);
        }

        const msgpack::object& ReadFromOffset(TDataOffset offset) {
            if (Deadline < TInstant::Now()) {
                ythrow TTimeLimitExceeded() << "no time left";
            }
            if (CurrentOffset.Defined() && CurrentOffset.GetRef() == offset) {
                return Message;
            } else {
                DataBuffer.Resize(Format.FindSize(offset));
                Stream.Seek(offset);
                Stream.Read(DataBuffer.Data(), DataBuffer.Size());
                Zone.clear();
                Unpacked = 0;
                Message = msgpack::unpack(Zone, DataBuffer.Data(), DataBuffer.Size(), Unpacked);
                if (Unpacked != DataBuffer.Size()) {
                    ythrow yexception() << "extra bytes in buffer";
                }
                CurrentOffset = offset;
                return Message;
            }
        }

        void DecodeValues(TTimeIndex timeIndex, TKeyIndex keyIndex, TRequestedIndex requestedIndex,
                          const TVector<TSignalOffset>& signalIndices, const msgpack::object_array& values) {

            TVector<std::pair<TSignalName, TValue>> recordValues;
            recordValues.reserve(signalIndices.size());
            for (const auto signalIndex : signalIndices) {
                if (signalIndex >= values.size) {
                    recordValues.emplace_back(Format.Signals[keyIndex].second[signalIndex], TValue());
                } else {
                    recordValues.emplace_back(
                        Format.Signals[keyIndex].second[signalIndex],
                        NZoom::NPython::TMsgPackValueHierarchy::DeserializeValue(values.ptr[signalIndex], true)
                    );
                }
            }

            Records.emplace_back(
                timeIndex,
                requestedIndex,
                std::move(recordValues)
            );
        }

        const TSomethingFormat& Format;
        const TVector<TTimestamp>& Timestamps;
        const TVector<std::pair<TInstanceKey, TVector<TSignalName>>>& KeysAndSignals;
        TSnappyInputStream& Stream;
        const TInstant Deadline;

        size_t Unpacked = 0;
        TBuffer DataBuffer;
        msgpack::zone Zone;
        msgpack::object Message;
        TMaybe<TDataOffset> CurrentOffset;

        // key index, position in KeysAndSignals
        THashMap<std::pair<TKeyIndex, TRequestedIndex>, TVector<TSignalOffset>> KeySignalIndices;
        // data offset, key index, position in KeysAndSignals, time index
        TVector<std::tuple<TDataOffset, TKeyIndex, TRequestedIndex, TTimeIndex>> OffsetKeyTimeIndices;
        // result
        TVector<std::tuple<TTimestamp, TRequestedIndex, TRecord>> Records;
    };
}
