#pragma once

#include "something.h"

#include <infra/monitoring/common/msgpack.h>

#include <library/cpp/blockcodecs/codecs.h>

namespace NHistDb {
    class TSomethingFormat::TSaver {
    public:
        TSaver(const TSomethingFormat& header)
            : Format(header)
            , Packer(Buffer)
        {
        }

        TString Dump() {
            Packer.pack_array(6);
            DumpBlocks();
            DumpKeys();
            DumpSignalsUnified();
            DumpSignalsTable();
            DumpKeyOffset();
            DumpTimeOffset();
            return NBlockCodecs::Codec("snappy")->Encode(TStringBuf(Buffer.data(), Buffer.size()));
        }

    private:
        void DumpBlocks() {
            Packer.pack_array(Format.Blocks.size());
            for (const auto& block : Format.Blocks) {
                Packer.pack_array(2);
                Packer.pack_uint64(block.UncompressedSize);
                Packer.pack_uint64(block.CompressedSize);
            }
        }

        void DumpKeys() {
            Packer.pack_array(Format.Keys.size());
            for (const auto& key : Format.Keys) {
                NMonitoring::PackString(Packer, key.ToNamed());
            }
        }

        void DumpSignalsUnified() {
            Packer.pack_array(Format.Signals.size());
            for (const auto& signalsInKey : Format.Signals) {
                Packer.pack_array(signalsInKey.second.size());
                for (const auto signal : signalsInKey.second) {
                    Packer.pack_uint64(GetSignalOffset(signal));
                }
            }
        }

        void DumpSignalsTable() {
            Packer.pack_array(KnownSignals.size());
            for (const auto signal : KnownSignals) {
                NMonitoring::PackString(Packer, signal.GetName());
            }
        }

        void DumpOffsets(const TOffsets& offsets) {
            Packer.pack_array(offsets.size());
            for (const auto& row : offsets) {
                Packer.pack_array(row.size());
                for (const auto& offset : row) {
                    Packer.pack_uint64(offset);
                }
            }
        }

        void DumpKeyOffset() {
            DumpOffsets(Format.KeyOffsets);
        }

        void DumpTimeOffset() {
            DumpOffsets(Format.TimeOffsets);
        }

        size_t GetSignalOffset(TSignalName signal) {
            THashMap<TSignalName, size_t>::insert_ctx context;
            const auto it(KnownSignalsOffsets.find(signal, context));
            if (it != KnownSignalsOffsets.end()) {
                return it->second;
            }
            KnownSignals.emplace_back(signal);
            return KnownSignalsOffsets.emplace_direct(context, signal, KnownSignalsOffsets.size())->second;
        }

        const TSomethingFormat& Format;
        msgpack::sbuffer Buffer;
        msgpack::packer<msgpack::sbuffer> Packer;

        TVector<TSignalName> KnownSignals;
        THashMap<TSignalName, size_t> KnownSignalsOffsets;
    };
}
