#include "something.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/memory/blob.h>
#include <util/generic/xrange.h>

using namespace NHistDb;
using namespace NTags;
using namespace NZoom::NRecord;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;
using namespace NZoom::NYasmConf;

const size_t HEADER_OFFSET = sizeof(ui32);
const static TString TESTING_HEADER = "s5_1524985500.header";
const static TString TESTING_DATA = "testing.data";

class TSomethingHelper {
public:
    TSomethingFormat& Writer() noexcept {
        if (!OutputFormat.Defined()) {
            OutputFormat.ConstructInPlace();
        }
        if (!OutputStream.Defined()) {
            OutputStream.ConstructInPlace(TESTING_DATA);
        }
        return OutputFormat.GetRef();
    }

    void WriteRecord(TSomethingFormat::TTimestamp timestamp, const TInstanceKey key, const TRecord& record) noexcept {
        Writer().WriteRecord(timestamp, key, record, OutputStream.GetRef());
    }

    TSomethingBulkWriter BulkWriter() {
        return Writer().BulkWriter(OutputStream.GetRef());
    }

    void Finish() noexcept {
        UNIT_ASSERT(Header.empty());
        Writer();
        OutputStream->Finish();
        OutputFormat->SaveBlocks(OutputStream->Blocks());
        Header = OutputFormat->Dump();
    }

    TSomethingFormat& Reader() noexcept {
        UNIT_ASSERT(!Header.empty());
        if (!InputFormat.Defined()) {
            InputFormat.ConstructInPlace(Header);
        }
        if (!InputStream.Defined()) {
            InputStream.ConstructInPlace(TESTING_DATA, InputFormat->GetBlocks());
        }
        return InputFormat.GetRef();
    }

    TVector<TSomethingFormat::TReadRow> ReadRecords(
        const TVector<TSomethingFormat::TTimestamp>& timestamps,
        const TVector<std::pair<TInstanceKey, TVector<TSignalName>>>& keysAndSignals
    ) noexcept {
        return Reader().ReadRecords(timestamps, keysAndSignals, InputStream.GetRef());
    }

    TVector<TSomethingFormat::TReadData> Read(
        const TVector<TSomethingFormat::TTimestamp>& times,
        TVector<std::pair<TRequestKey, TVector<TSignalName>>>& tags
    ){
        return Reader().Read(times, tags, InputStream.GetRef());
    }

    TVector<TSomethingFormat::TReadAggregatedData> ReadAggregated(
            size_t offset, size_t limit,
            const TSomethingFormat::TTagSignals& tags,
            const TYasmConf& conf
    ){
        return Reader().ReadAggregated(offset, limit, tags, conf, InputStream.GetRef());
    }

    TSomethingIterator Iterate(const TVector<TSomethingFormat::TTimestamp>& timestamps) noexcept {
        return Reader().IterateRecords(timestamps, InputStream.GetRef());
    }

    TSomethingKeyIterator IterateKeys() noexcept {
        return Reader().IterateKeys();
    }

private:
    TString Header;

    TMaybe<TSomethingFormat> OutputFormat;
    TMaybe<TSnappyOutputStream> OutputStream;

    TMaybe<TSomethingFormat> InputFormat;
    TMaybe<TSnappyInputStream> InputStream;
};

class TRecordBuilder {
public:
    TRecord Build() {
        return {std::move(Values)};
    }

    TRecordBuilder& Add(TSignalName signal, TValue value) {
        Values.emplace_back(std::move(signal), std::move(value));
        return *this;
    }

private:
    TVector<std::pair<TSignalName, TValue>> Values;
};

Y_UNIT_TEST_SUITE(TSomethingFormatTest) {

    TBlob LoadHeader() {
        TBlob source(TBlob::PrechargedFromFileSingleThreaded(TESTING_HEADER));
        UNIT_ASSERT(source.Size() >= sizeof(i32));
        UNIT_ASSERT_VALUES_EQUAL(*reinterpret_cast<const i32*>(source.Data()), 1);
        return source.SubBlob(HEADER_OFFSET, source.Size());
    }

    Y_UNIT_TEST(DumpRestore) {
        TBlob source(LoadHeader());

        TSomethingFormat header(TStringBuf(source.AsCharPtr(), source.Size()));
        const auto before(header.FindSize(0));
        const auto encoded(header.Dump());

        TSomethingFormat restored(encoded);
        const auto after(restored.FindSize(0));

        UNIT_ASSERT_VALUES_EQUAL(before, after);
    }

    Y_UNIT_TEST(Match) {
        TBlob source(LoadHeader());
        TSomethingFormat header(TStringBuf(source.AsCharPtr(), source.Size()));

        UNIT_ASSERT_VALUES_EQUAL(
            header.HasRecords({100, 999}, {TInstanceKey::FromNamed("mdscloud||geo,ctype,prj,tier")}),
            TVector<bool>({true, false})
        );
    }

    Y_UNIT_TEST(NoRecords) {
        TSomethingHelper helper;
        helper.Finish();

        const auto reply(helper.ReadRecords({0, 1, 2, 3, 4}, {}));
        UNIT_ASSERT_VALUES_EQUAL(reply.size(), 0);
    }

    Y_UNIT_TEST(ReadWriteRecord) {
        const auto instanceKey(TInstanceKey::FromNamed("testing||geo,ctype,prj,tier"));
        const auto signalName(TSignalName(TStringBuf("disk_usage_summ")));

        auto record(TRecordBuilder().Add(signalName, TValue(1)).Build());

        // write something
        TSomethingHelper helper;
        helper.WriteRecord(5, instanceKey, record);
        helper.Finish();

        // read what we wrote
        const auto reply(helper.ReadRecords({5}, {{instanceKey, {signalName}}}));
        UNIT_ASSERT_VALUES_EQUAL(reply.size(), 1);
        const auto& row(reply.back());
        UNIT_ASSERT_VALUES_EQUAL(std::get<0>(row), 5);
        UNIT_ASSERT_VALUES_EQUAL(std::get<1>(row), 0);
        UNIT_ASSERT(std::get<2>(row) == record);

        // iterate over all records
        auto iterator(helper.Iterate({0, 1, 2, 3, 4, 5, 6, 7, 8, 9}));
        size_t iterations = 0;
        while (iterator.Next()) {
            const auto& row(iterator.Get());
            UNIT_ASSERT_VALUES_EQUAL(std::get<0>(row), 5);
            UNIT_ASSERT_VALUES_EQUAL(std::get<1>(row), instanceKey);
            UNIT_ASSERT(std::get<2>(row) == record);
            ++iterations;
        }
        UNIT_ASSERT_VALUES_EQUAL(iterations, 1);

        // iterate over all tags and signals
        auto keyIterator(helper.IterateKeys());
        iterations = 0;
        while (keyIterator.Next()) {
            const auto& row(keyIterator.Get());
            UNIT_ASSERT_VALUES_EQUAL(std::get<0>(row), instanceKey);
            UNIT_ASSERT_VALUES_EQUAL(std::get<1>(row)->size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(std::get<1>(row)->back(), signalName);
            ++iterations;
        }
        UNIT_ASSERT_VALUES_EQUAL(iterations, 1);

        // test chunk interval
        UNIT_ASSERT_VALUES_EQUAL(helper.Reader().FirstRecordTime(), 5UL);
        UNIT_ASSERT_VALUES_EQUAL(helper.Reader().LastRecordTime(), 5UL);
    }

    Y_UNIT_TEST(WriteDifferentRecords) {
        const auto instanceKey(TInstanceKey::FromNamed("testing||geo,ctype,prj,tier"));
        const auto firstSignalName(TSignalName(TStringBuf("disk_usage_summ")));
        const auto secondSignalName(TSignalName(TStringBuf("place_usage_summ")));

        auto firstRecord(TRecordBuilder().Add(firstSignalName, TValue(1)).Build());
        auto secondRecord(TRecordBuilder().Add(secondSignalName, TValue(1)).Build());

        TSomethingHelper helper;
        helper.WriteRecord(5, instanceKey, firstRecord);
        helper.WriteRecord(10, instanceKey, secondRecord);
        helper.Finish();

        const auto reply(helper.ReadRecords({5, 10}, {{instanceKey, {firstSignalName, secondSignalName}}}));
        UNIT_ASSERT_VALUES_EQUAL(reply.size(), 2);

        UNIT_ASSERT_VALUES_EQUAL(std::get<0>(reply.front()), 5);
        UNIT_ASSERT_VALUES_EQUAL(std::get<2>(reply.front()).GetValues().size(), 2);

        UNIT_ASSERT_VALUES_EQUAL(std::get<0>(reply.back()), 10);
        UNIT_ASSERT_VALUES_EQUAL(std::get<2>(reply.back()).GetValues().size(), 2);
    }

    Y_UNIT_TEST(BulkWriter) {
        const auto alphaInstanceKey(TInstanceKey::FromNamed("testing||geo,ctype,prj,tier"));
        const auto betaInstanceKey(TInstanceKey::FromNamed("mdscloud||geo,ctype,prj,tier"));
        const TSignalName gammaSignalName(TStringBuf("disk_usage_summ"));
        const TSignalName thetaSignalName(TStringBuf("place_usage_summ"));

        const TValue one(1);
        const TValue two(2);
        const TValue three(2);

        TSomethingHelper helper;
        auto writer(helper.BulkWriter());
        const auto firstCursor(writer.CreateSeries(alphaInstanceKey, gammaSignalName));
        firstCursor.Insert(writer, 0, one.GetValue());
        firstCursor.Insert(writer, 1, one.GetValue());
        const auto secondCursor(writer.CreateSeries(alphaInstanceKey, thetaSignalName));
        secondCursor.Insert(writer, 1, two.GetValue());
        secondCursor.Insert(writer, 2, two.GetValue());
        const auto thirdCursor(writer.CreateSeries(betaInstanceKey, thetaSignalName));
        thirdCursor.Insert(writer, 0, three.GetValue());
        writer.Flush();
        helper.Finish();

        {
            const auto reply(helper.ReadRecords({0, 1, 2}, {{alphaInstanceKey, {gammaSignalName, thetaSignalName}}}));
            UNIT_ASSERT_VALUES_EQUAL(reply.size(), 3);
            UNIT_ASSERT_EQUAL(
                std::get<2>(reply[0]),
                TRecordBuilder().Add(gammaSignalName, one.GetValue()).Add(thetaSignalName, TValue()).Build()
            );
            UNIT_ASSERT_EQUAL(
                std::get<2>(reply[1]),
                TRecordBuilder().Add(gammaSignalName, one.GetValue()).Add(thetaSignalName, two.GetValue()).Build()
            );
            UNIT_ASSERT_EQUAL(
                std::get<2>(reply[2]),
                TRecordBuilder().Add(gammaSignalName, TValue()).Add(thetaSignalName, two.GetValue()).Build()
            );
        }

        {
            const auto reply(helper.ReadRecords({0, 1, 2}, {{betaInstanceKey, {gammaSignalName, thetaSignalName}}}));
            UNIT_ASSERT_VALUES_EQUAL(reply.size(), 1);
            UNIT_ASSERT_EQUAL(
                std::get<2>(reply[0]),
                TRecordBuilder().Add(thetaSignalName, three.GetValue()).Build()
            );
        }
    }

    Y_UNIT_TEST(ReadSeveralRecords) {
        const TVector<TString> signalStrings = {
            "unistat-ok_subsource_VIDEO_dmmm",
            "unistat-init_subsource_VIDEO_dmmm",
            "unistat-unanswers_subsource_VIDEO_dmmm",
        };
        const TVector<TSignalName> signals(signalStrings.begin(), signalStrings.end());

        const TVector<TString> requestStrings = {
            "itype=testing",
            "itype=testing;geo=man",
            "itype=testing;geo=sas",
        };

        const TVector<TSomethingFormat::TTimestamp> timestamps = {1, 2, 3, 4, 5};

        TVector<std::pair<TRequestKey, TVector<TSignalName>>> tags;
        for (const auto& requestString: requestStrings) {
            tags.emplace_back(
                TRequestKey::FromString(requestString),
                signals
            );
        }

        TSomethingHelper helper;
        const TVector<TInstanceKey> instanceKeysGeo = {
            TInstanceKey::FromNamed("testing|ctype=prestable;prj=test|tier"),
            TInstanceKey::FromNamed("testing|ctype=prestable;geo=man;prj=test|tier"),
            TInstanceKey::FromNamed("testing|ctype=prestable;geo=sas;prj=test|tier"),
        };

        for (const auto& signal: signals) {
            for (const auto& instanceKey: instanceKeysGeo) {
                for (const auto& timestamp: timestamps) {
                    auto record(TRecordBuilder().Add(signal, TValue(1)).Build());
                    helper.WriteRecord(timestamp, instanceKey, record);
                }
            }
        }
        helper.Finish();

        auto res = helper.Read(timestamps, tags);

        UNIT_ASSERT_VALUES_EQUAL(res.size(), 25);
    }

    Y_UNIT_TEST(ReadResultTest) {
        const TVector<TString> signalStrings = {
            "unistat-ok_subsource_VIDEO_dmmm",
        };
        const TVector<TSignalName> signals(signalStrings.begin(), signalStrings.end());

        const TVector<TString> requestStrings = {
            "itype=testing",
        };

        const TVector<TSomethingFormat::TTimestamp> timestamps = {1, 2, 3};

        TSomethingFormat::TTagSignals tags;
        for (const auto &requestString: requestStrings) {
            tags.emplace_back(
                TRequestKey::FromString(requestString),
                signals
            );
        }

        TSomethingHelper helper;
        size_t value = 0;
        {
            const TVector<TInstanceKey> instanceKeysGeo = {
                TInstanceKey::FromNamed("testing|ctype=prestable;geo=man;prj=test|tier"),
                TInstanceKey::FromNamed("testing|ctype=prestable;geo=sas;prj=test|tier"),
            };

            for (const auto &instanceKey: instanceKeysGeo) {
                for (const auto &timestamp: timestamps) {
                    auto record(TRecordBuilder().Add(signals[0], TValue(value++)).Build());
                    helper.WriteRecord(timestamp, instanceKey, record);
                }
            }
        }
        {
            auto instanceKey = TInstanceKey::FromNamed("testing||geo,ctype,prj,tier");

            auto record(TRecordBuilder().Add(signals[0], TValue(value++)).Build());
            helper.WriteRecord(1, instanceKey, record);
        }
        {
            auto instanceKey = TInstanceKey::FromNamed("testing|ctype=prestable;prj=test|geo,tier");

            auto record(TRecordBuilder().Add(signals[0], TValue(value++)).Build());
            helper.WriteRecord(3, instanceKey, record);
        }
        helper.Finish();

        auto res = helper.Read(timestamps, tags);

        TVector<std::tuple<
            TSomethingFormat::TTimestamp,
            TVector<std::pair<TString, size_t>>,
            TString,
            TString
        >> expectedRes = {
            {
                1, {{"unistat-ok_subsource_VIDEO_dmmm", 6},}, "itype=testing", "testing||ctype,geo,prj,tier",
            }
        }; // we no longer try to read anything other than the best instance key group. in this case it is a full pre-aggregate.

        UNIT_ASSERT_VALUES_EQUAL(res.size(), expectedRes.size());

        for (size_t id : xrange(res.size())){
            const auto& row = res[id];
            const auto& expectedRow = expectedRes[id];

            UNIT_ASSERT_VALUES_EQUAL(row.Timestamp, std::get<0>(expectedRow));
            UNIT_ASSERT_VALUES_EQUAL(row.RequestKey.ToNamed(), std::get<2>(expectedRow));
            UNIT_ASSERT_VALUES_EQUAL(row.InstanceKey.ToNamed(), std::get<3>(expectedRow));

            const auto& vals = row.Record.GetValues();
            const auto& avals = std::get<1>(expectedRow);
            UNIT_ASSERT_VALUES_EQUAL(vals.size(), avals.size());
            for (const size_t vid : xrange(vals.size())) {
                UNIT_ASSERT_VALUES_EQUAL(vals[vid].first, avals[vid].first);
                UNIT_ASSERT_VALUES_EQUAL(vals[vid].second, TValue(avals[vid].second));
            }
        }
    }

    Y_UNIT_TEST(ReadRecordsWithSpell) {
        const TVector<TString> signalStrings = {
            "unistat-ok_subsource_VIDEO_dmmm",
            "unistat-init_subsource_VIDEO_dmmm",
            "unistat-unanswers_subsource_VIDEO_dmmm",
        };
        const TVector<TSignalName> signals(signalStrings.begin(), signalStrings.end());

        const TVector<TString> readSignalStrings = {
            "unistad-ok_subsource_VIDEO_dmmm",
            "unistot-init_subsource_VIDEO_dmmm",
            "nistat-unanswers_subsource_VIDEO_dmmm",
        };
        const TVector<TSignalName> readSignals(readSignalStrings .begin(), readSignalStrings .end());

        const TVector<TString> requestStrings = {
            "itype=testing",
            "itype=testing;geo=man",
            "itype=testing;geo=sas",
        };

        const TVector<TSomethingFormat::TTimestamp> timestamps = {1, 2, 3, 4, 5};

        TVector<std::pair<TRequestKey, TVector<TSignalName>>> tags;
        for (const auto& requestString: requestStrings) {
            tags.emplace_back(
                TRequestKey::FromString(requestString),
                readSignals
            );
        }

        TSomethingHelper helper;
        const TVector<TInstanceKey> instanceKeys = {
            TInstanceKey::FromNamed("testing|ctype=prestable;prj=test|tier"),
            TInstanceKey::FromNamed("testing|ctype=prestable;geo=man;prj=test|tier"),
            TInstanceKey::FromNamed("testing|ctype=prestable;geo=sas;prj=test|tier"),
        };

        for (const auto& signal: signals) {
            for (const auto& instanceKey: instanceKeys) {
                for (const auto& timestamp: timestamps) {
                    auto record(TRecordBuilder().Add(signal, TValue(1)).Build());
                    helper.WriteRecord(timestamp, instanceKey, record);
                }
            }
        }
        helper.Finish();

        auto res = helper.Read(timestamps, tags);
        UNIT_ASSERT_VALUES_EQUAL(res.size(), 0);
    }

    Y_UNIT_TEST(AggregtedReadTest) {
        const TVector<TString> signalStrings = {
            "module-suffix_signal_summ",
            "module-conf_signal"
        };
        const TVector<TSignalName> signals(signalStrings.begin(), signalStrings.end());

        const TVector<TSomethingFormat::TTimestamp> timestamps = {0, 1, 2};

        TSomethingHelper helper;
        size_t value = 0;
        {
            const TVector<TInstanceKey> instanceKeys = {
                    TInstanceKey::FromNamed("testing|tag=value1"),
                    TInstanceKey::FromNamed("testing|tag=value2"),
                    TInstanceKey::FromNamed("testing|tag=value3")
            };

            for (const auto &instanceKey: instanceKeys) {
                for (const auto &timestamp: timestamps) {
                    auto record(TRecordBuilder().Add(signals[0], TValue(value++)).Build());
                    helper.WriteRecord(timestamp, instanceKey, record);
                }
            }
        }
        {
            auto instanceKey = TInstanceKey::FromNamed("testing|tag=value4");

            auto record(TRecordBuilder().Add(signals[1], TValue(10)).Build());
            helper.WriteRecord(1, instanceKey, record);
        }

        {
            auto instanceKey = TInstanceKey::FromNamed("testing|tag=value5");

            auto record(TRecordBuilder().Add(signals[1], TValue(11)).Build());
            helper.WriteRecord(1, instanceKey, record);
        }
        helper.Finish();

        const TRequestKey requestKey = TRequestKey::FromString("itype=testing;tag=value1,value2,value4,value5");

        TSomethingFormat::TTagSignals tags;
        tags.emplace_back(TRequestKey::FromString(requestKey.GetDynamicString()), signals);

        TYasmConf yasmConf = TYasmConf::FromString("{\n"
                                                   "  \"conflist\": {\n"
                                                   "    \"testing\": {\n"
                                                   "      \"patterns\": {},\n"
                                                   "      \"periods\": {},\n"
                                                   "      \"signals\": {\"module-conf_signal\": [false, [\"summ\", true]]}\n"
                                                   "    },\n"
                                                   "    \"common\": {\n"
                                                   "      \"patterns\": {},\n"
                                                   "      \"periods\": {},\n"
                                                   "      \"signals\": {}\n"
                                                   "    }\n"
                                                   "  }\n"
                                                   "}");

        auto res = helper.ReadAggregated(1, 4, tags, yasmConf);

        UNIT_ASSERT_VALUES_EQUAL(res.size(), 2);

        UNIT_ASSERT_VALUES_EQUAL(res[0].RequestKey, requestKey);
        UNIT_ASSERT_VALUES_EQUAL(res[1].RequestKey, requestKey);

        UNIT_ASSERT_VALUES_EQUAL(res[0].Timestamp, 1);
        UNIT_ASSERT_VALUES_EQUAL(res[1].Timestamp, 2);

        UNIT_ASSERT_VALUES_EQUAL(res[0].MatchedInstances, 4);
        UNIT_ASSERT_VALUES_EQUAL(res[1].MatchedInstances, 2);

        {
            auto values = std::move(res[0].Record.GetValues());
            TVector<std::pair<TSignalName, TValue>> expectedValues;
            expectedValues.emplace_back(TSignalName(TString("module-conf_signal")), TValue(21));
            expectedValues.emplace_back(TSignalName(TString("module-suffix_signal_summ")), TValue(5));
            UNIT_ASSERT_VALUES_EQUAL(values, expectedValues);
        }

        {
            auto values = std::move(res[1].Record.GetValues());
            TVector<std::pair<TSignalName, TValue>> expectedValues;
            expectedValues.emplace_back(TSignalName(TString("module-suffix_signal_summ")), TValue(7));
            UNIT_ASSERT_VALUES_EQUAL(values, expectedValues);
        }
    }
}
