#pragma once

#include "something.h"

#include <infra/yasm/zoom/components/serialization/common/msgpack_utils.h>

#include <util/memory/pool.h>
#include <util/generic/xrange.h>

namespace NHistDb {
    class TSomethingFormat::TWriter {
    public:
        TWriter(TSomethingFormat& header, TSnappyOutputStream& stream)
            : Header(header)
            , Stream(stream)
        {
        }

        inline size_t AllocateKey(const TInstanceKey key) noexcept {
            const auto keyIt(Header.KeyToIndex.emplace(key, Header.Keys.size()));
            const auto keyIndex(keyIt.first->second);
            if (keyIt.second) {
                Header.Keys.emplace_back(key);
                Header.Signals.emplace_back();
                Header.KeyOffsets.emplace_back();
            }
            return keyIndex;
        }

        inline size_t AllocateSignal(size_t keyIndex, NZoom::NSignal::TSignalName signalName) noexcept {
            auto& signals(Header.Signals[keyIndex]);
            const auto it(signals.first.emplace(signalName, signals.second.size()));
            if (it.second) {
                signals.second.emplace_back(signalName);
            }
            return it.first->second;
        }

        inline void AllocateTimestamp(TTimestamp timestamp) {
            Header.TimeOffsets.resize(Max(Header.TimeOffsets.size(), timestamp + 1));
        }

        inline void WriteValues(TTimestamp timestamp, size_t keyIndex, const msgpack::sbuffer& buffer) {
            const auto position(Stream.Position());
            Stream.Write(buffer.data(), buffer.size());
            Header.KeyOffsets[keyIndex].emplace_back(position);
            Header.TimeOffsets[timestamp].emplace_back(position);
        }

        void WriteRecord(TTimestamp timestamp, const TInstanceKey key, const TRecord& record) {
            AllocateTimestamp(timestamp);

            const auto keyIndex(AllocateKey(key));

            // Check if record with same time and same instance key already
            // in chunk. See GOLOVAN-2352.
            if (!Header.TimeOffsets[timestamp].empty() && !Header.KeyOffsets[keyIndex].empty()) {
                if (OffsetsContains(Header.KeyOffsets[keyIndex], Header.TimeOffsets[timestamp])) {
                    return;
                }
            }

            TVector<const TSomethingFormat::TValue*> orderedValues;
            const auto& values(record.GetValues());
            auto& signals(Header.Signals[keyIndex]);
            if (signals.second.empty()) {
                signals.first.reserve(values.size());
                signals.second.reserve(values.size());
                orderedValues.reserve(values.size());

                for (const auto& pair : values) {
                    signals.first.emplace(pair.first, signals.second.size());
                    signals.second.emplace_back(pair.first);
                    orderedValues.emplace_back(&pair.second);
                }

            } else {
                orderedValues.resize(signals.second.size(), nullptr);
                for (const auto& pair : values) {
                    const auto it(signals.first.emplace(pair.first, signals.second.size()));
                    if (it.second) {
                        signals.second.emplace_back(pair.first);
                        orderedValues.emplace_back(&pair.second);
                    } else {
                        orderedValues[it.first->second] = &pair.second;
                    }
                }
            }

            if (orderedValues.empty()) {
                return;
            }

            Header.Buffer.clear();
            msgpack::packer<msgpack::sbuffer> packer(Header.Buffer);
            packer.pack_array(orderedValues.size());

            NZoom::NPython::TValueRefSerializer<msgpack::sbuffer, NZoom::NHgram::TUgramCompressor> valueSerializer(packer);
            for (const auto value : orderedValues) {
                if (value != nullptr) {
                    value->Update(valueSerializer);
                } else {
                    packer.pack_nil();
                }
            }

            WriteValues(timestamp, keyIndex, Header.Buffer);
        }

    private:
        TSomethingFormat& Header;
        TSnappyOutputStream& Stream;

        bool OffsetsContains(const TVector<ui64>& needle, const TVector<ui64>& offsets) {
            auto start(offsets.cbegin());
            for (const auto offset : needle) {
                start = LowerBound(start, offsets.cend(), offset);
                if (start == offsets.cend()) {
                    return false;
                } else if (*start == offset) {
                    return true;
                }
            }
            return false;
        }
    };

    class TSomethingFormat::TBulkWriter {
    public:
        TBulkWriter(TSomethingFormat& header, TSnappyOutputStream& stream)
            : Writer(header, stream)
        {
        }

        TSomethingBulkCursor CreateSeries(NTags::TInstanceKey key, NZoom::NSignal::TSignalName signalName) {
            const auto keyIndex(Writer.AllocateKey(key));
            const auto signalIndex(Writer.AllocateSignal(keyIndex, signalName));
            return TSomethingBulkCursor(keyIndex, signalIndex);
        }

        void InsertValue(const size_t keyIndex, const size_t signalIndex,
                         TSomethingFormat::TTimestamp timestamp, NZoom::NValue::TValueRef value) {
            auto& keys(GetOrCreate(InsertedValues, timestamp));
            auto& signals(GetOrCreate(keys, keyIndex));
            GetOrCreate(signals, signalIndex) = SerializeValue(timestamp, value);
        }

        void Flush(TSomethingFormat::TTimestamp upperBorder) {
            for (const auto timestamp : xrange(EarliestTimestamp, Min(upperBorder, InsertedValues.size()))) {
                WriteWholeTimestamp(timestamp);
                EarliestTimestamp = timestamp + 1;
            }
        }

        void Flush() {
            Flush(InsertedValues.size());
        }

    private:
        class TMovablePool {
        public:
            TMovablePool()
                : Pool(MakeHolder<TMemoryPool>(4096UL))
            {
            }

            inline TStringBuf Copy(const msgpack::sbuffer& buf) {
                return Pool->AppendString(TStringBuf(buf.data(), buf.size()));
            }

            inline void Clear() {
                Pool->Clear();
            }

        private:
            THolder<TMemoryPool> Pool;
        };

        TSomethingFormat::TWriter Writer;
        msgpack::sbuffer TempBuffer;
        TVector<TMovablePool> SerializedValues;
        // timestamp -> key -> signal -> value
        TVector<TVector<TVector<TStringBuf>>> InsertedValues;
        size_t EarliestTimestamp = 0;

        template <class T>
        static inline typename T::value_type& GetOrCreate(T& container, size_t pos) {
            container.resize(Max(container.size(), pos + 1));
            return container[pos];
        }

        inline TStringBuf SerializeValue(TSomethingFormat::TTimestamp timestamp, NZoom::NValue::TValueRef value) {
            EarliestTimestamp = Min(EarliestTimestamp, timestamp);
            TempBuffer.clear();
            msgpack::packer<msgpack::sbuffer> packer(TempBuffer);
            NZoom::NPython::TValueRefSerializer<msgpack::sbuffer, NZoom::NHgram::TUgramCompressor> serializer(packer);
            value.Update(serializer);
            return GetOrCreate(SerializedValues, timestamp).Copy(TempBuffer);
        }

        inline void WriteWholeTimestamp(TSomethingFormat::TTimestamp timestamp) {
            Writer.AllocateTimestamp(timestamp);

            auto& keys(GetOrCreate(InsertedValues, timestamp));
            for (const auto keyIndex : xrange(keys.size())) {
                const auto& signals(keys[keyIndex]);
                if (signals.empty()) {
                    continue;
                }

                TempBuffer.clear();
                msgpack::packer<msgpack::sbuffer> packer(TempBuffer);
                packer.pack_array(signals.size());

                for (const auto signalIndex : xrange(signals.size())) {
                    const auto value(signals[signalIndex]);
                    if (!value.empty()) {
                        TempBuffer.write(value.data(), value.size());
                    } else {
                        packer.pack_nil();
                    }
                }

                Writer.WriteValues(timestamp, keyIndex, TempBuffer);
            }

            keys.clear();
            GetOrCreate(SerializedValues, timestamp).Clear();
        }
    };
}
