#include "continuous.h"

using namespace NHistDb;
using namespace NZoom::NSignal;
using namespace NTags;

namespace {
    static constexpr size_t CHUNKS = 2;
}

TContinuousSeriesCursor::TContinuousSeriesCursor(TSomethingBulkWriter& writer, const TSomethingBulkCursor& cursor, const size_t startOffset)
    : Writer(writer)
    , Cursor(cursor)
    , Offset(startOffset)
{
}

void TContinuousSeriesCursor::Append(NZoom::NValue::TValueRef value) {
    Cursor.Insert(Writer, Offset, value);
    Offset++;
}

TEncodedSeriesCursor::TEncodedSeriesCursor(TCompactWriter writer, NTags::TInstanceKey key, const TRecordPeriod& period)
    : Writer(std::move(writer))
    , Period(period)
{
    Writer.Start(key);
}

void TEncodedSeriesCursor::Append(
    TSignalName signalName,
    TInstant startTime,
    size_t valuesCount,
    NYasmServer::ESeriesKind seriesKind,
    const TString& chunk
) {
    Writer.Append(
        signalName,
        Period.GetOffset(startTime),
        valuesCount,
        seriesKind,
        chunk
    );
}

void TEncodedSeriesCursor::Commit() {
    Writer.Commit();
}

TPlacementContinuousWriter::TPlacementContinuousWriter(const TString& root, const TString& hostName, const TRecordPeriod& period)
    : Root(root)
    , HostName(hostName)
    , Period(period)
{
}

TContinuousSeriesCursor TPlacementContinuousWriter::CreateContinuousSeriesCursor(TInstanceKey key, TSignalName signal, TInstant startTime) {
    auto& state(SelectState(startTime));
    return {state.GetSomethingWriter(), state.GetSomethingWriter().CreateSeries(key, signal), Period.GetOffset(startTime)};
}

TEncodedSeriesCursor TPlacementContinuousWriter::CreateEncodedSeriesCursor(TInstanceKey key, TInstant startTime) {
    auto& state(SelectState(startTime));
    return {state.GetCompactWriter(), key, Period};
}

bool TPlacementContinuousWriter::SwitchToCompact(TInstant startTime) {
    return SelectState(startTime).GetPlacement().SwitchToCompact();
}

void TPlacementContinuousWriter::Flush(TInstant finishedTime, bool dump) {
    if (States.empty() || States.front().GetStartTime() > Period.GetStartTime(finishedTime)) {
        return;
    }

    auto& state(SelectState(finishedTime));
    if (dump && States.front().GetStartTime() < Period.GetStartTime(finishedTime)) {
        States.front().Finish();
        States.pop_front();
    }

    const auto offset(Period.GetOffset(finishedTime));
    if (!offset) {
        return;
    }

    state.Flush(offset);
    if (dump) {
        state.Dump();
    }
}

void TPlacementContinuousWriter::Finish() {
    while (!States.empty()) {
        States.front().Finish();
        States.pop_front();
    }
}

TVector<TInstant> TPlacementContinuousWriter::ActiveChunks() const {
    TVector<TInstant> times;
    for (const auto& state : States) {
        times.emplace_back(state.GetStartTime());
    }
    return times;
}

TMaybe<TInstant> TPlacementContinuousWriter::LastRecordTime() {
    const auto lastStartTime(TSecondPlacementReader::LastStartTime(Root, HostName, Period));
    if (lastStartTime.Defined()) {
        return SelectState(*lastStartTime).GetPlacement().LastRecordTime();
    } else {
        return Nothing();
    }
}

TPlacementContinuousWriter::TChunkState::TChunkState(const TPlacementContinuousWriter& parent, TInstant startTime)
    : StartTime(parent.Period.GetStartTime(startTime))
    , Placement(parent.Root, parent.HostName, parent.Period, StartTime)
{
}

void TPlacementContinuousWriter::TChunkState::Flush(TSomethingFormat::TTimestamp upperBorder) {
    if (SomethingWriter.Defined()) {
        SomethingWriter->Flush(upperBorder);
    }
}

void TPlacementContinuousWriter::TChunkState::Finish() {
    if (SomethingWriter.Defined()) {
        SomethingWriter->Flush();
    }
    Dump();
}

void TPlacementContinuousWriter::TChunkState::Dump() {
    Placement.Dump();
}

TInstant TPlacementContinuousWriter::TChunkState::GetStartTime() const {
    return StartTime;
}

TSecondPlacementWriter& TPlacementContinuousWriter::TChunkState::GetPlacement() {
    return Placement;
}

TSomethingBulkWriter& TPlacementContinuousWriter::TChunkState::GetSomethingWriter() {
    Y_VERIFY(CompactWriter.Empty());
    if (SomethingWriter.Empty()) {
        SomethingWriter.ConstructInPlace(Placement.CreateSomethingWriter());
    }
    return *SomethingWriter;
}

TCompactWriter& TPlacementContinuousWriter::TChunkState::GetCompactWriter() {
    Y_VERIFY(SomethingWriter.Empty());
    if (CompactWriter.Empty()) {
        CompactWriter.ConstructInPlace(Placement.CreateCompactWriter());
    }
    return *CompactWriter;
}

bool TPlacementContinuousWriter::TChunkState::operator ==(const TChunkState& other) const {
    return StartTime == other.StartTime;
}

TPlacementContinuousWriter::TChunkState& TPlacementContinuousWriter::SelectState(TInstant startTime) {
    startTime = Period.GetStartTime(startTime);
    if (States.empty()) {
        // create one if no states defined
        States.emplace_back(*this, startTime);
        return States.back();
    } else if (startTime == States.front().GetStartTime()) {
        // try to check one end of deque
        return States.front();
    } else if (startTime == States.back().GetStartTime()) {
        // and another one
        return States.back();
    }

    Y_VERIFY(States.size() <= CHUNKS && States.size() > 0);

    // always store only adjacent chunks
    if (States.front().GetStartTime() > startTime) {
        if (States.size() == CHUNKS || States.front().GetStartTime() - startTime != Period.GetInterval()) {
            ythrow yexception() << "given start time is outdated";
        }

        States.emplace_front(*this, startTime);
        return States.front();
    } else {
        Y_VERIFY(States.front().GetStartTime() != startTime);

        if (States.size() == CHUNKS) {
            // store only two chunks, delete first one before emplace
            States.front().Finish();
            States.pop_front();
        }

        if (startTime - States.front().GetStartTime() != Period.GetInterval()) {
            States.front().Finish();
            States.pop_front();
        }

        States.emplace_back(*this, startTime);
        return States.back();
    }
}
