#include "continuous.h"
#include "periods.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/folder/tempdir.h>

using namespace NHistDb;
using namespace NTags;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;

namespace {
    struct TContinuousPlacementState {
        TContinuousPlacementState()
            : HostName("SAS.000")
            , Period(TRecordPeriod::Get("s5"))
            , Now(TInstant::Seconds(1530465000))
            , InstanceKey(TInstanceKey::FromNamed("testing||geo,ctype,prj,tier"))
            , SignalName(TStringBuf("disk_usage_summ"))
            , Value(42)
            , Writer(Root.Name(), HostName, Period)
        {
        }

        TContinuousSeriesCursor Create(TInstant startTime) {
            return Writer.CreateContinuousSeriesCursor(InstanceKey, SignalName, startTime);
        }

        void AssertActiveChunks(std::initializer_list<TInstant> reference) {
            UNIT_ASSERT_VALUES_EQUAL(Writer.ActiveChunks(), TVector<TInstant>(reference));
        }

        TTempDir Root;
        const TString HostName;
        const TRecordPeriod Period;
        const TInstant Now;

        const TInstanceKey InstanceKey;
        const TSignalName SignalName;
        const TValue Value;

        TPlacementContinuousWriter Writer;
    };
}

Y_UNIT_TEST_SUITE(TContinuousPlacementTest) {

    Y_UNIT_TEST(WriteToTheEndOfChunk) {
        TContinuousPlacementState state;

        auto cursor(state.Create(state.Now - state.Period.GetResolution()));
        cursor.Append(state.Value.GetValue());
        state.AssertActiveChunks({state.Now - state.Period.GetInterval()});

        state.Writer.Flush(state.Now);
        state.AssertActiveChunks({state.Now - state.Period.GetInterval(), state.Now});

        state.Writer.Flush(state.Now, true);
        state.AssertActiveChunks({state.Now});
    }

    Y_UNIT_TEST(WriteToChunkBeforeCurrent) {
        TContinuousPlacementState state;

        auto currentCursor(state.Create(state.Now));
        currentCursor.Append(state.Value.GetValue());
        state.AssertActiveChunks({state.Now});

        auto prevCursor(state.Create(state.Now - state.Period.GetResolution()));
        prevCursor.Append(state.Value.GetValue());
        state.AssertActiveChunks({state.Now - state.Period.GetInterval(), state.Now});

        state.Writer.Flush(state.Now + state.Period.GetResolution(), true);

        {
            TSecondPlacementReader reader(state.Root.Name(), state.HostName, state.Period, state.Now);
            const TVector<TSomethingFormat::TTimestamp> timestamps = {0};
            TSomethingFormat::TTagSignals tags;
            tags.emplace_back(TRequestKey::FromString("itype=testing"), TVector<TSignalName>{state.SignalName});
            UNIT_ASSERT_VALUES_EQUAL(reader.Read(timestamps, tags).size(), 1);
        }

        {
            TSecondPlacementReader reader(state.Root.Name(), state.HostName, state.Period, state.Now - state.Period.GetInterval());
            const TVector<TSomethingFormat::TTimestamp> timestamps = {299};
            TSomethingFormat::TTagSignals tags;
            tags.emplace_back(TRequestKey::FromString("itype=testing"), TVector<TSignalName>{state.SignalName});
            UNIT_ASSERT_VALUES_EQUAL(reader.Read(timestamps, tags).size(), 1);
        }
    }

    Y_UNIT_TEST(WriteToChunkAfterCurrent) {
        TContinuousPlacementState state;

        auto currentCursor(state.Create(state.Now));
        currentCursor.Append(state.Value.GetValue());
        state.AssertActiveChunks({state.Now});

        auto nextCursor(state.Create(state.Now + state.Period.GetInterval()));
        nextCursor.Append(state.Value.GetValue());
        state.AssertActiveChunks({state.Now, state.Now + state.Period.GetInterval()});
    }

    Y_UNIT_TEST(WriteToOutdatedChunk) {
        TContinuousPlacementState state;

        state.Create(state.Now);
        state.AssertActiveChunks({state.Now});
        UNIT_ASSERT_EXCEPTION(state.Create(state.Now - state.Period.GetInterval() * 2), yexception);

        state.Create(state.Now + state.Period.GetInterval());
        state.AssertActiveChunks({state.Now, state.Now + state.Period.GetInterval()});
        UNIT_ASSERT_EXCEPTION(state.Create(state.Now - state.Period.GetInterval()), yexception);
    }

    Y_UNIT_TEST(WriteToVeryRecentChunk) {
        TContinuousPlacementState state;

        state.Create(state.Now);
        state.Create(state.Now + state.Period.GetInterval() * 2);
        state.AssertActiveChunks({state.Now + state.Period.GetInterval() * 2});
    }

    Y_UNIT_TEST(Finish) {
        TContinuousPlacementState state;
        state.Create(state.Now);
        state.Writer.Finish();
        state.AssertActiveChunks({});
    }

    Y_UNIT_TEST(LastRecordTime) {
        TContinuousPlacementState state;

        TPlacementContinuousWriter emptyWriter(state.Root.Name(), state.HostName, state.Period);
        UNIT_ASSERT_VALUES_EQUAL(emptyWriter.LastRecordTime(), Nothing());

        auto cursor(state.Create(state.Now));
        cursor.Append(state.Value.GetValue());
        cursor.Append(state.Value.GetValue());
        state.Writer.Flush(state.Period.FromOffset(state.Now, 2), true);

        TPlacementContinuousWriter writer(state.Root.Name(), state.HostName, state.Period);
        UNIT_ASSERT_VALUES_EQUAL(*writer.LastRecordTime(), state.Now + state.Period.GetResolution());
    }
}
