#include "periods.h"

#include <util/generic/hash.h>
#include <util/generic/vector.h>

using namespace NHistDb;

namespace {
    static const TVector<TRecordPeriod> PERIODS = {
        TPeriodDescriptor{
            .Resolution = TDuration::Seconds(5),
            .RecordsPerFile = 300,
            .TargetPrefix = "s5",
        },
        TPeriodDescriptor{
            .Resolution = TDuration::Minutes(5),
            .RecordsPerFile = 300,
            .TargetPrefix = "m5",
        },
        TPeriodDescriptor{
            .Resolution = TDuration::Hours(1),
            .RecordsPerFile = 300,
            .TargetPrefix = "h",
        },
        TPeriodDescriptor{
            .Resolution = TDuration::Hours(3),
            .RecordsPerFile = 300,
            .TargetPrefix = "h3",
        },
        TPeriodDescriptor{
            .Resolution = TDuration::Hours(6),
            .RecordsPerFile = 300,
            .TargetPrefix = "h6",
        },
        TPeriodDescriptor{
            .Resolution = TDuration::Hours(12),
            .RecordsPerFile = 300,
            .TargetPrefix = "h12",
        },
        TPeriodDescriptor{
            .Resolution = TDuration::Days(1),
            .RecordsPerFile = 300,
            .TargetPrefix = "h24",
        },
    };

    THashMap<TString, TRecordPeriod> PeriodsByPrefix() {
        THashMap<TString, TRecordPeriod> result;
        result.reserve(PERIODS.size());
        for (const auto& period : PERIODS) {
            result.emplace(period.GetTargetPrefix(), period);
        }
        return result;
    }

    THashMap<TDuration, TRecordPeriod> PeriodsByResolution() {
        THashMap<TDuration, TRecordPeriod> result;
        result.reserve(PERIODS.size());
        for (const auto& period : PERIODS) {
            result.emplace(period.GetResolution(), period);
        }
        return result;
    }

    static const THashMap<TString, TRecordPeriod> PERIODS_BY_PREFIX = PeriodsByPrefix();
    static const THashMap<TDuration, TRecordPeriod> PERIODS_BY_RESOLUTION = PeriodsByResolution();
}

TRecordPeriod::TRecordPeriod(const TPeriodDescriptor& descriptor)
    : Descriptor(descriptor)
{
}

TRecordPeriod::TRecordPeriod(const TRecordPeriod& other)
    : TRecordPeriod(other.Descriptor)
{
}

const TRecordPeriod& TRecordPeriod::Get(const TString& targetPrefix) {
    return PERIODS_BY_PREFIX.at(targetPrefix);
}

const TRecordPeriod& TRecordPeriod::Get(const TDuration& resolution) {
    return PERIODS_BY_RESOLUTION.at(resolution);
}

const TVector<TRecordPeriod>& TRecordPeriod::GetAll() {
    return PERIODS;
}

TDuration TRecordPeriod::GetResolution() const {
    return Descriptor.Resolution;
}

TDuration TRecordPeriod::GetInterval() const {
    return Descriptor.Resolution * Descriptor.RecordsPerFile;
}

TInstant TRecordPeriod::GetStartTime(TInstant startTime) const {
    return startTime - TDuration::FromValue(startTime.GetValue() % GetInterval().GetValue());
}

size_t TRecordPeriod::GetOffset(TInstant timestamp) const {
    const auto startTime(GetStartTime(timestamp));
    return (timestamp - startTime).GetValue() / GetResolution().GetValue();
}

TStringBuf TRecordPeriod::GetTargetPrefix() const {
    return Descriptor.TargetPrefix;
}

size_t TRecordPeriod::GetRecordsPerFile() const {
    return Descriptor.RecordsPerFile;
}

TInstant TRecordPeriod::FromOffset(TInstant startTime, size_t offset) const {
    return GetStartTime(startTime) + Descriptor.Resolution * offset;
}

TInstant TRecordPeriod::GetPointStartTime(TInstant pointStartTime) const {
    return pointStartTime - TDuration::FromValue(pointStartTime.GetValue() % GetResolution().GetValue());
}
