#include "second.h"
#include "transpiler.h"

#include <util/system/file.h>
#include <util/stream/file.h>

using namespace NHistDb;

namespace {
    const static NBlockCodecs::ICodec* ZSTD_CODEC = NBlockCodecs::Codec("zstd08_1");

    static constexpr size_t COMPACT_CHUNK_SIZE = 1 << 18;
    static constexpr bool USE_DIRECT = false;

    static constexpr TFlags<EOpenModeFlag> HEADER_READ_FLAGS = EOpenModeFlag::OpenExisting | EOpenModeFlag::RdOnly | EOpenModeFlag::Seq;
    static constexpr TFlags<EOpenModeFlag> HEADER_WRITE_FLAGS = EOpenModeFlag::CreateAlways | EOpenModeFlag::WrOnly | EOpenModeFlag::Seq;

    static const TString HEADER_SUFFIX = ".header";
    static const TString DATA_SUFFIX = ".data";

    TFsPath GeneratePath(const TString& root, const TString& hostName) {
        return TFsPath(root) / TFsPath(hostName);
    }

    TFsPath GeneratePath(const TString& root, const TString& hostName, const TRecordPeriod& period, TInstant startTime, const TString& suffix) {
        TStringBuilder chunkName;
        chunkName << period.GetTargetPrefix() << "_" << period.GetStartTime(startTime).Seconds() << suffix;
        return GeneratePath(root, hostName) / TFsPath(chunkName);
    }

    TFsPath GenerateHeaderPath(const TString& root, const TString& hostName, const TRecordPeriod& period, TInstant startTime) {
        return GeneratePath(root, hostName, period, startTime, HEADER_SUFFIX);
    }

    TFsPath GenerateDataPath(const TString& root, const TString& hostName, const TRecordPeriod& period, TInstant startTime) {
        return GeneratePath(root, hostName, period, startTime, DATA_SUFFIX);
    }

    const TFsPath& EnsureExists(const TFsPath& path) {
        path.Parent().MkDirs();
        return path;
    }

    TVector<TString> GetHosts(const TString& root) {
        const TFsPath path(root);
        path.MkDirs();

        TVector<TString> hostNames;
        path.ListNames(hostNames);

        return hostNames;
    }

    TVector<TInstant> GetChunkTimes(const TString& root, const TString& hostName, const TRecordPeriod& period) {
        const auto path(GeneratePath(root, hostName));
        path.MkDirs();
        if (path.Exists() && !path.IsDirectory()) {
            return {};
        }

        TVector<TString> headers;
        path.ListNames(headers);
        EraseIf(headers, [&period](const TString& file) { return !(file.StartsWith(period.GetTargetPrefix()) && file.EndsWith(HEADER_SUFFIX)); });

        TVector<TInstant> chunkTimes;
        size_t prefix_size = period.GetTargetPrefix().size();
        for (const auto& header : headers) {
            try {
                chunkTimes.emplace_back(TInstant::Seconds(FromString<ui64>(TStringBuf(header).SubString(prefix_size + 1, 10))));
            } catch (...) {
            }
        }
        Sort(chunkTimes);
        return chunkTimes;
    }

    TMaybe<TInstant> FindLastStartTime(const TString& root, const TString& hostName, const TRecordPeriod& period) {
        TVector<TInstant> chunkTimes = GetChunkTimes(root, hostName, period);

        const auto it(MaxElement(chunkTimes.begin(), chunkTimes.end()));
        if (it != chunkTimes.end()) {
            return *it;
        } else {
            return Nothing();
        }
    }

    TVector<NZoom::NHost::THostName> FindLastHostNames(const TString& root, const TRecordPeriod& period) {
        auto hostNames = GetHosts(root);

        TVector<NZoom::NHost::THostName> actualHostNames;
        for (const auto& hostName : hostNames) {
            const auto lastStartTime(FindLastStartTime(root, hostName, period));
            if (lastStartTime.Defined() && *lastStartTime + period.GetInterval() * 2 > TInstant::Now()) {
                actualHostNames.emplace_back(hostName);
            }
        }

        return actualHostNames;
    }
}

THeaderStorage::THeaderStorage(TMaybe<THeaderDescriptor> descriptor) {
    if (descriptor.Defined()) {
        if (descriptor->Version == TSomethingFormat::GetSomethingVersion()) {
            Assign(MakeHolder<TSomethingFormat>(descriptor->Data));
        } else if (descriptor->Version == TCompactFormat::GetCompactVersion()) {
            Assign(MakeHolder<TCompactFormat>(descriptor->Data));
        } else {
            ythrow yexception() << "wrong header version given";
        }
    }
}

void THeaderStorage::Assign(THolder<TAbstractFormat> format) {
    Format = std::move(format);
}

bool THeaderStorage::Empty() const {
    return Format.Get() == nullptr;
}

bool THeaderStorage::IsSomething() const {
    return Format && Format->GetVersion() == TSomethingFormat::GetSomethingVersion();
}

bool THeaderStorage::IsCompact() const {
    return Format && Format->GetVersion() == TCompactFormat::GetCompactVersion();
}

TSomethingFormat& THeaderStorage::GetSomething() {
    Y_VERIFY(IsSomething());
    return *dynamic_cast<TSomethingFormat*>(Format.Get());
}

TCompactFormat& THeaderStorage::GetCompact() {
    Y_VERIFY(IsCompact());
    return *dynamic_cast<TCompactFormat*>(Format.Get());
}

TAbstractFormat& THeaderStorage::GetFormat() {
    Y_VERIFY(!Empty());
    return *Format;
}

const TAbstractFormat& THeaderStorage::GetFormat() const {
    Y_VERIFY(!Empty());
    return *Format;
}

void TCompactWriter::Start(NTags::TInstanceKey instanceKey) {
    Format.Start(instanceKey, Stream);
}

void TCompactWriter::Append(
    NZoom::NSignal::TSignalName signalName,
    TAbstractFormat::TTimestamp offset,
    size_t valuesCount,
    NYasmServer::ESeriesKind seriesKind,
    const TString& chunk
) {
    Format.Append(signalName, offset, valuesCount, seriesKind, chunk);
}

void TCompactWriter::Commit() {
    Format.Commit(Stream);
}

TSecondPlacementWriter::TSecondPlacementWriter(const TString& root, const TString& hostName,
                                               const TRecordPeriod& period, TInstant startTime)
    : Period(period)
    , StartTime(startTime)
    , HeaderPath(GenerateHeaderPath(root, hostName, Period, StartTime))
    , DataPath(GenerateDataPath(root, hostName, Period, StartTime))
    , HeaderStorage(TSecondPlacementReader::ReadHeaderBody(HeaderPath))
{
    Initialize();
}

TSomethingBulkWriter TSecondPlacementWriter::CreateSomethingWriter() {
    return HeaderStorage.GetSomething().BulkWriter(*DataStream);
}

TCompactWriter TSecondPlacementWriter::CreateCompactWriter() {
    return {HeaderStorage.GetCompact(), *DataStream};
}

bool TSecondPlacementWriter::SwitchToCompact() {
    if (HeaderStorage.IsCompact()) {
        return true;
    }
    if (HeaderStorage.GetFormat().GetBlocks().empty()) {
        HeaderStorage.Assign(MakeHolder<TCompactFormat>());
        Initialize();
        return true;
    }
    return false;
}

void TSecondPlacementWriter::Dump() {
    const auto temporaryPath(WriteHeader());
    temporaryPath.RenameTo(HeaderPath);
}

TFsPath TSecondPlacementWriter::WriteHeader() {
    const auto temporaryPath(HeaderPath.Parent().Child(TStringBuilder() << HeaderPath.Basename() << ".tmp"));
    TFile headerFile(temporaryPath.GetPath(), HEADER_WRITE_FLAGS);
    TFileOutput headerStream(headerFile);
    DataStream->Flush();
    HeaderStorage.GetFormat().SaveBlocks(DataStream->Blocks());
    const auto version(HeaderStorage.GetFormat().GetVersion());
    headerStream.Write(version.data(), version.size());
    headerStream.Write(HeaderStorage.GetFormat().Dump());
    return temporaryPath;
}

TMaybe<TInstant> TSecondPlacementWriter::LastRecordTime() {
    const auto lastOffset(HeaderStorage.GetFormat().LastRecordTime());
    if (!lastOffset.Empty()) {
        return Period.FromOffset(StartTime, *lastOffset);
    } else {
        return Nothing();
    }
}

void TSecondPlacementWriter::Compact() {
    if (!HeaderStorage.IsSomething()) {
        return;
    }

    TSnappyInputStream stream(DataPath, HeaderStorage.GetFormat().GetBlocks());
    auto iterator(HeaderStorage.GetSomething().IterateRecords(stream));

    const auto temporaryDataPath(DataPath.Parent().Child(TStringBuilder() << DataPath.Basename() << ".compacted"));
    THolder<TCompactFormat> compactFormat(MakeHolder<TCompactFormat>());
    TSnappyOutputStream compactDataStream(
        EnsureExists(temporaryDataPath).GetPath(),
        compactFormat->GetBlocks(),
        USE_DIRECT,
        COMPACT_CHUNK_SIZE,
        ZSTD_CODEC
    );

    TFormatTranspiler transpiler;
    TMaybe<NTags::TInstanceKey> prevInstanceKey;
    while (iterator.Next()) {
        const TSomethingFormat::TTimestamp timestamp(std::get<0>(iterator.Get()));
        const NTags::TInstanceKey instanceKey(std::get<1>(iterator.Get()));

        if (!prevInstanceKey.Defined() || instanceKey != *prevInstanceKey) {
            if (prevInstanceKey.Defined()) {
                transpiler.Finish(*compactFormat);
                compactFormat->Commit(compactDataStream);
            }
            prevInstanceKey = instanceKey;
            compactFormat->Start(instanceKey, compactDataStream);
        }

        transpiler.SetTimestamp(timestamp);
        const NZoom::NRecord::TRecord& record(std::get<2>(iterator.Get()));
        record.Process(transpiler);
    }

    if (prevInstanceKey.Defined()) {
        transpiler.Finish(*compactFormat);
        compactFormat->Commit(compactDataStream);
    }

    HeaderStorage.Assign(std::move(compactFormat));
    const auto temporaryHeaderPath(WriteHeader());

    // TODO: this isn't atomic
    HeaderPath.DeleteIfExists();
    temporaryDataPath.RenameTo(DataPath);
    temporaryHeaderPath.RenameTo(HeaderPath);
}

void TSecondPlacementWriter::Initialize() {
    if (HeaderStorage.IsSomething()) {
        DataStream.ConstructInPlace(
            EnsureExists(DataPath).GetPath(),
            HeaderStorage.GetFormat().GetBlocks(),
            USE_DIRECT
        );
    } else if (HeaderStorage.IsCompact()) {
        DataStream.ConstructInPlace(
            EnsureExists(DataPath).GetPath(),
            HeaderStorage.GetFormat().GetBlocks(),
            USE_DIRECT,
            COMPACT_CHUNK_SIZE,
            ZSTD_CODEC
        );
    } else {
        HeaderStorage.Assign(MakeHolder<TSomethingFormat>());
        DataStream.ConstructInPlace(
            EnsureExists(DataPath).GetPath(),
            HeaderStorage.GetFormat().GetBlocks(),
            USE_DIRECT
        );
    }
}

TSecondPlacementReader::TSecondPlacementReader(TAtomicSharedPtr<THeaderStorage> storage, const TFsPath& dataPath)
    : HeaderStorage(storage)
{
    CreateDataStream(dataPath);
}

TSecondPlacementReader::TSecondPlacementReader(const TString& root, const TString& hostName,
                                               const TRecordPeriod& period, TInstant startTime)
    : HeaderStorage(
        MakeAtomicShared<THeaderStorage>(ReadHeaderBody(GenerateHeaderPath(root, hostName, period, startTime)))
    )
{
    CreateDataStream(GenerateDataPath(root, hostName, period, startTime));
}

TVector<TSomethingFormat::TReadData> TSecondPlacementReader::Read(
        const TVector<TSomethingFormat::TTimestamp>& times,
        const TSomethingFormat::TTagSignals& tags
) {
    return HeaderStorage->GetFormat().Read(times, tags, *DataStream);
}

TVector<TSomethingFormat::TReadAggregatedData> TSecondPlacementReader::ReadAggregated(
        size_t offset, size_t limit,
        const NHistDb::TSomethingFormat::TTagSignals& tags,
        const NZoom::NYasmConf::TYasmConf& conf
) {
    return HeaderStorage->GetFormat().ReadAggregated(offset, limit, tags, conf, *DataStream);
}

TVector<TInstant> TSecondPlacementReader::GetChunkTimes(const TString& root, const TString& hostName, const TRecordPeriod& period) {
    return ::GetChunkTimes(root, hostName, period);
}

TFsPath TSecondPlacementReader::GetDataPath(const TString& root, const TString& hostName, const TRecordPeriod& period, TInstant startTime) {
    return ::GenerateDataPath(root, hostName, period, startTime);
}

TVector<TString> TSecondPlacementReader::GetHostNames(const TString& root) {
    return ::GetHosts(root);
}

TMaybe<TInstant> TSecondPlacementReader::LastStartTime(const TString& root, const TString& hostName, const TRecordPeriod& period) {
    return FindLastStartTime(root, hostName, period);
}

TVector<NZoom::NHost::THostName> TSecondPlacementReader::LastHostNames(const TString& root, const TRecordPeriod& period) {
    return FindLastHostNames(root, period);
}

TMaybe<THeaderDescriptor> TSecondPlacementReader::ReadHeaderBody(const TFile& headerFile) {
    try {
        const auto header(TBlob::FromFile(headerFile));
        const auto versionData(header.SubBlob(HEADER_VERSION_SIZE));
        THeaderVersionStorage version;
        std::memcpy(version.data(), versionData.AsCharPtr(), versionData.Size());
        return THeaderDescriptor{
            .Version = version,
            .Data = header.SubBlob(HEADER_VERSION_SIZE, header.Size())
        };
    } catch (const yexception&) {
        return Nothing();
    }
}

TFsPath TSecondPlacementReader::GenerateHeaderPath(const TString& root, const TString& hostName, const TRecordPeriod& period, TInstant startTime) {
    return ::GenerateHeaderPath(root, hostName, period, startTime);

}

TMaybe<THeaderDescriptor> TSecondPlacementReader::ReadHeaderBody(const TFsPath& headerPath) {
    TMaybe<TFileStat> stat;
    return ReadHeaderBody(headerPath, stat);
}

TMaybe<THeaderDescriptor> TSecondPlacementReader::ReadHeaderBody(const TFsPath& headerPath, TMaybe<TFileStat>& stat) {
    try {
        const TFile file(headerPath.GetPath(), HEADER_READ_FLAGS);
        stat.ConstructInPlace(file);
        return ReadHeaderBody(file);
    } catch (const TFileError& error) {
        if (error.Status() == ENOENT) {
            return Nothing();
        } else {
            throw;
        }
    }
}

void TSecondPlacementReader::IterateSomething(TSecondPlacementReader::TSomethingCallback callback) {
    if (!HeaderStorage->IsSomething()) {
        ythrow yexception() << "Storage is not Something";
    }
    auto iterator(HeaderStorage->GetSomething().IterateRecords(DataStream.GetRef()));
    while (iterator.Next()) {
        callback(iterator.Get());
    }
}

void TSecondPlacementReader::IterateSomethingKeys(TSecondPlacementReader::TSomethingKeyCallback callback) {
    if (!HeaderStorage->IsSomething()) {
        ythrow yexception() << "Storage is not Something";
    }

    auto iterator(HeaderStorage->GetSomething().IterateKeys());
    while (iterator.Next()) {
        callback(iterator.Get());
    }
}

TVector<TSomethingFormat::TTimestamp> TSecondPlacementReader::GetTimes() const {
    if (!HeaderStorage->IsSomething()) {
        ythrow yexception() << "Storage is not Something";
    }

    return HeaderStorage->GetSomething().GetTimes();
}

void TSecondPlacementReader::IterateSomething(const TVector<TSomethingFormat::TTimestamp>& timestamps, TSecondPlacementReader::TSomethingCallback callback) {
    if (!HeaderStorage->IsSomething()) {
        ythrow yexception() << "Storage is not Something";
    }
    auto iterator(HeaderStorage->GetSomething().IterateRecords(timestamps, DataStream.GetRef()));
    while (iterator.Next()) {
        callback(iterator.Get());
    }
}

void TSecondPlacementReader::CreateDataStream(const TFsPath& dataPath) {
    if (HeaderStorage->IsSomething()) {
        DataStream.ConstructInPlace(dataPath.GetPath(), HeaderStorage->GetFormat().GetBlocks());
    } else if (HeaderStorage->IsCompact()) {
        DataStream.ConstructInPlace(dataPath.GetPath(), HeaderStorage->GetFormat().GetBlocks(), COMPACT_CHUNK_SIZE, ZSTD_CODEC);
    } else {
        ythrow yexception() << "chunk not exists";
    }
}

TSecondHeaderCache::TEntry::TEntry(const TFsPath& headerPath)
    : HeaderPath(headerPath)
{
}

TAtomicSharedPtr<THeaderStorage> TSecondHeaderCache::TEntry::Get() {
    {
        TReadGuard readGuard(Lock);
        if (IsActual()) {
            return HeaderStorage;
        }
    }

    TWriteGuard writeGuard(Lock);

    if (IsActual()) {
        return HeaderStorage;
    }

    Stat.Clear();
    HeaderStorage = MakeAtomicShared<THeaderStorage>(TSecondPlacementReader::ReadHeaderBody(HeaderPath, Stat));

    return HeaderStorage;
}

bool TSecondHeaderCache::TEntry::IsActual() const {
    if (!Stat || !HeaderStorage) {
        return false;
    } else {
        TFileStat stat(HeaderPath);
        return Stat->MTime == stat.MTime && Stat->Size == stat.Size;
    }
}

TSecondHeaderCache::TSecondHeaderCache(size_t entries)
    : Entries(entries)
{
}

THolder<TSecondPlacementReader> TSecondHeaderCache::Create(const TString& root, const TString& hostName,
                                                           const TRecordPeriod& period, TInstant startTime) {
    return Create(
        GenerateHeaderPath(root, hostName, period, startTime),
        GenerateDataPath(root, hostName, period, startTime)
    );
}

THolder<TSecondPlacementReader> TSecondHeaderCache::Create(const TFsPath& headerPath, const TFsPath& dataPath) {
    TAtomicSharedPtr<TEntry> entry;
    {
        auto guard = Guard(Lock);
        const auto it(Entries.Find(headerPath.GetPath()));
        if (it != Entries.End()) {
            entry = it.Value();
        } else {
            entry = MakeAtomicShared<TEntry>(headerPath);
            Entries.Insert(headerPath.GetPath(), entry);
        }
    }

    const auto headerStorage(entry->Get());
    if (headerStorage->Empty()) {
        return {};
    } else {
        try {
            return MakeHolder<TSecondPlacementReader>(entry->Get(), dataPath);
        } catch (const TFileError& error) {
            if (error.Status() == ENOENT) {
                return {};
            } else {
                throw;
            }
        }
    }
}
