#pragma once

#include "periods.h"

#include <library/cpp/cache/cache.h>
#include <library/cpp/any/any.h>

#include <infra/yasm/histdb/components/formats/something.h>
#include <infra/yasm/histdb/components/formats/compact.h>
#include <infra/yasm/histdb/components/streams/snappy.h>
#include <infra/yasm/zoom/components/yasmconf/yasmconf.h>
#include <infra/yasm/common/labels/host/host.h>

#include <util/folder/path.h>
#include <util/system/mutex.h>
#include <util/system/fstat.h>
#include <util/system/rwlock.h>

namespace NHistDb {
    struct THeaderDescriptor {
        THeaderVersionStorage Version;
        TBlob Data;
    };

    class TCompactWriter {
    public:
        TCompactWriter(TCompactFormat& format, TSnappyOutputStream& stream)
            : Format(format)
            , Stream(stream)
        {
        }

        void Start(NTags::TInstanceKey instanceKey);

        void Append(
            NZoom::NSignal::TSignalName signalName,
            TAbstractFormat::TTimestamp offset,
            size_t valuesCount,
            NYasmServer::ESeriesKind seriesKind,
            const TString& chunk
        );

        void Commit();

    private:
        TCompactFormat& Format;
        TSnappyOutputStream& Stream;
    };

    class THeaderStorage {
    public:
        THeaderStorage(TMaybe<THeaderDescriptor> descriptor);

        void Assign(THolder<TAbstractFormat> format);

        bool Empty() const;
        bool IsSomething() const;
        bool IsCompact() const;

        TSomethingFormat& GetSomething();
        TCompactFormat& GetCompact();

        TAbstractFormat& GetFormat();
        const TAbstractFormat& GetFormat() const;

    private:
        THolder<TAbstractFormat> Format;
    };

    class TSecondPlacementWriter {
    public:
        TSecondPlacementWriter(const TString& root, const TString& hostName,
                               const TRecordPeriod& period, TInstant startTime);

        TSomethingBulkWriter CreateSomethingWriter();
        TCompactWriter CreateCompactWriter();

        bool SwitchToCompact();

        void Dump();

        TMaybe<TInstant> LastRecordTime();

        void Compact();

    private:
        void Initialize();
        TFsPath WriteHeader();

        const TRecordPeriod Period;
        const TInstant StartTime;

        const TFsPath HeaderPath;
        const TFsPath DataPath;

        THeaderStorage HeaderStorage;
        TMaybe<TSnappyOutputStream> DataStream;
    };

    class TSecondPlacementReader {
    public:
        TSecondPlacementReader(TAtomicSharedPtr<THeaderStorage> storage, const TFsPath& dataPath);
        TSecondPlacementReader(const TString& root, const TString& hostName,
                               const TRecordPeriod& period, TInstant startTime);

        TVector<TSomethingFormat::TReadData> Read(
            const TVector<TSomethingFormat::TTimestamp>& times,
            const TSomethingFormat::TTagSignals& tags
        );

        TVector<TSomethingFormat::TReadAggregatedData> ReadAggregated(
            size_t offset, size_t limit,
            const TSomethingFormat::TTagSignals& tags,
            const NZoom::NYasmConf::TYasmConf& conf
        );

        static TFsPath GetDataPath(const TString& root, const TString& hostName, const TRecordPeriod& period, TInstant startTime);
        static TVector<TInstant> GetChunkTimes(const TString& root, const TString& hostName, const TRecordPeriod& period);
        static TVector<TString> GetHostNames(const TString& root);
        static TMaybe<TInstant> LastStartTime(const TString& root, const TString& hostName, const TRecordPeriod& period);
        static TVector<NZoom::NHost::THostName> LastHostNames(const TString& root, const TRecordPeriod& period);
        static TFsPath GenerateHeaderPath(const TString& root, const TString& hostName, const TRecordPeriod& period, TInstant startTime);

        static TMaybe<THeaderDescriptor> ReadHeaderBody(const TFile& headerFile);
        static TMaybe<THeaderDescriptor> ReadHeaderBody(const TFsPath& headerPath);
        static TMaybe<THeaderDescriptor> ReadHeaderBody(const TFsPath& headerPath, TMaybe<TFileStat>& stat);

        using TSomethingKeyCallback = std::function<void(TSomethingFormat::TKeyIteratorRow&)>;
        using TSomethingCallback = std::function<void(TSomethingFormat::TIteratorRow&)>;
        TVector<TSomethingFormat::TTimestamp> GetTimes() const;
        void IterateSomethingKeys(TSomethingKeyCallback callback);
        void IterateSomething(TSomethingCallback callback);
        void IterateSomething(const TVector<TSomethingFormat::TTimestamp>& timestamps, TSomethingCallback callback);
    private:
        void CreateDataStream(const TFsPath& dataPath);

        TAtomicSharedPtr<THeaderStorage> HeaderStorage;
        TMaybe<TSnappyInputStream> DataStream;
    };

    class TSecondHeaderCache {
    public:
        TSecondHeaderCache(size_t entries);

        THolder<TSecondPlacementReader> Create(const TString& root, const TString& hostName,
                                               const TRecordPeriod& period, TInstant startTime);
        THolder<TSecondPlacementReader> Create(const TFsPath& headerPath, const TFsPath& dataPath);

    private:
        class TEntry {
        public:
            TEntry(const TFsPath& headerPath);

            TAtomicSharedPtr<THeaderStorage> Get();

        private:
            bool IsActual() const;

            const TFsPath HeaderPath;

            TAtomicSharedPtr<THeaderStorage> HeaderStorage;

            TMaybe<TFileStat> Stat;
            TRWMutex Lock;
        };

        TLRUCache<TString, TAtomicSharedPtr<TEntry>> Entries;

        TMutex Lock;
    };
}
