#include "second.h"
#include "periods.h"

#include <infra/yasm/common/labels/host/host.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/folder/tempdir.h>

using namespace NHistDb;
using namespace NTags;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;

namespace {
    struct TSecondPlacementState {
        TSecondPlacementState()
            : HostName("SAS.000")
            , Period(TRecordPeriod::Get("s5"))
            , StartTime(TInstant::Now())
            , InstanceKey(TInstanceKey::FromNamed("testing||geo,ctype,prj,tier"))
            , SignalName(TStringBuf("disk_usage_summ"))
            , Value(42)
        {
        }

        void Write() {
            TSecondPlacementWriter writer(Root.Name(), HostName, Period, StartTime);
            auto bulkWriter(writer.CreateSomethingWriter());
            auto cursor(bulkWriter.CreateSeries(InstanceKey, SignalName));
            cursor.Insert(bulkWriter, 5, Value.GetValue());
            bulkWriter.Flush();
            writer.Dump();
        }

        const TTempDir Root;

        const TString HostName;
        const TRecordPeriod Period;
        const TInstant StartTime;

        const TInstanceKey InstanceKey;
        const TSignalName SignalName;
        const TValue Value;
    };
}

Y_UNIT_TEST_SUITE(TSecondPlacementState) {

    Y_UNIT_TEST(DumpRead) {
        TSecondPlacementState state;
        state.Write();

        TSecondPlacementReader reader(state.Root.Name(), state.HostName, state.Period, state.StartTime);
        const TVector<TSomethingFormat::TTimestamp> timestamps = {5};
        TSomethingFormat::TTagSignals tags;
        tags.emplace_back(TRequestKey::FromString("itype=testing"), TVector<TSignalName>{state.SignalName});
        auto reply(reader.Read(timestamps, tags));
        UNIT_ASSERT_VALUES_EQUAL(reply.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(reply.front().Timestamp, 5);
        UNIT_ASSERT_VALUES_EQUAL(reply.front().InstanceKey, state.InstanceKey);
    }

    Y_UNIT_TEST(LastChunks) {
        TSecondPlacementState state;
        state.Write();

        UNIT_ASSERT_VALUES_EQUAL(
            TSecondPlacementReader::LastHostNames(state.Root.Name(), state.Period),
            TVector<NZoom::NHost::THostName>{state.HostName}
        );
        UNIT_ASSERT_VALUES_EQUAL(
            TSecondPlacementReader::LastStartTime(state.Root.Name(), state.HostName, state.Period),
            MakeMaybe<TInstant>(state.Period.GetStartTime(state.StartTime))
        );
    }

    Y_UNIT_TEST(TestChunkTimes) {
        TTempDir root;
        root.Path().MkDir();
        NZoom::NHost::THostName host(TStringBuf("SAS.000"));

        auto hostPath = root.Path() / host.GetName();
        hostPath.MkDir();

        for (const auto& period: TRecordPeriod::GetAll()) {
            TVector<TInstant> instances = {period.GetStartTime(TInstant::Now())};
            for (size_t i = 0; i < 10; ++i) {
                instances.emplace_back(instances.back() + period.GetInterval());
            }

            for (auto time : instances) {
                auto path = TSecondPlacementReader::GenerateHeaderPath(root.Name(), host.GetName(), period, time);
                TFile tempFile(path, EOpenModeFlag::CreateAlways | EOpenModeFlag::WrOnly);
                tempFile.Close();
            }

            auto writedInstances = TSecondPlacementReader::GetChunkTimes(root.Name(), host.GetName(), period);
            UNIT_ASSERT_VALUES_EQUAL(writedInstances, instances);
        }
    }
}
