#include "stockpile_cleaner.h"

using namespace NHistDb;
using TSignalName = NZoom::NSignal::TSignalName;
using TInstanceKey = NTags::TInstanceKey;
using THostName = NZoom::NHost::THostName;

namespace {
    const long SERIES_IN_REQUEST = 10000;

    class TDeleteState {
    public:
        TDeleteState(NStockpile::TStockpileState& stockpileState, const NStockpile::TSensorId& sensorId, TInstant begin, TInstant end)
            : StockpileState(&stockpileState)
            , Options{.Sensor = sensorId, .Start = begin, .End = end}
        {
        }

        void Schedule(NStockpile::TGrpcStateHandler& handler, TLog& logger) {
            State.Reset();
            try {
                auto stockpileShard = StockpileState->ResolveOneStockpileShardForWrite(Options.Sensor.ShardId);
                State = MakeHolder<NStockpile::TStockpileDeleteState>(Options, stockpileShard);
                State->ScheduleForExecute(handler.GetQueue());
            } catch (const NStockpile::TShardNotFoundError &) {
                logger << TLOG_ERR << "Stockpile shard " << Options.Sensor.ShardId << " was not found";
            }
        }

        bool IsSuccess() const {
            return State && State->IsSuccess();
        }

    private:
        NStockpile::TStockpileState* StockpileState;
        NStockpile::TStockpileDeleteOptions Options;
        THolder<NStockpile::TStockpileDeleteState> State;
    };
}

StockpileDataCleaner::StockpileDataCleaner(TLog& logger, NStockpile::TStockpileState& stockpileState)
    : Logger(logger)
    , Handler(Logger)
    , StockpileState(stockpileState)
    , Limit(SERIES_IN_REQUEST)
{
}

void StockpileDataCleaner::ClearStockpileData(const NZoom::NHost::THostName& host,
                                              const NStockpile::TMetabaseShardKey& shardKey,
                                              TInstant begin,
                                              TInstant end) {
    auto shardState = ResolveShardForReadUntilReady(shardKey);
    auto findState = NStockpile::TMetabaseFindSensor::Make(host, shardState.Shard, shardState.GrpcRemoteHost, Limit, Logger);
    TVector<TDeleteState> deleteStates;

    while (findState || !deleteStates.empty()) {
        if (findState) {
            findState->ScheduleForExecute(Handler.GetQueue());
        }
        for (auto& deleteState : deleteStates) {
            deleteState.Schedule(Handler, Logger);
        }
        Handler.Wait();

        EraseIf(deleteStates, [](const TDeleteState& state) { return state.IsSuccess(); });

        if (findState) {
            if (findState->IsSuccess()) {
                auto sensors = findState->GetResult();
                for (const auto& keyAndSensorId : sensors) {
                    deleteStates.emplace_back(StockpileState, keyAndSensorId.second, begin, end);
                }
                findState = findState->Next();
            } else if (findState->IsRetriable()) {
                // retry find sensors
                shardState = ResolveShardForReadUntilReady(shardKey);
                findState = findState->Repeat(shardState.Shard, shardState.GrpcRemoteHost);
            } else {
                ythrow yexception() << "Can't find sensors for " << shardKey << ". Find request failed and is not retriable.";
            }
        }
    }
}

NStockpile::TMetabaseShardState StockpileDataCleaner::ResolveShardForReadUntilReady(const NStockpile::TMetabaseShardKey& shardKey) {
    NStockpile::TMetabaseShardState shardState;
    do {
        try {
            shardState = StockpileState.ResolveOneMetabaseShardForRead(shardKey);
        } catch (NStockpile::TShardNotFoundError&) {
        }
        if (!shardState.IsReady()) {
            Sleep(TDuration::MilliSeconds(500));
        }
    } while (!shardState.IsReady());

    return shardState;
}
