#include "stockpile_reader.h"
#include "metrics.h"

#include <infra/yasm/common/interval.h>
#include <infra/yasm/common/labels/tags/dynamic_filter.h>
#include <infra/yasm/common/threaded_executor.h>
#include <infra/monitoring/common/perf.h>

using namespace NTags;
using namespace NHistDb;
using namespace NHistDb::NPrivate;
using namespace NHistDb::NStockpile;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;
using namespace NZoom::NHost;
using namespace NZoom::NSubscription;
using namespace NZoom::NAccumulators;
using namespace NZoom::NProtobuf;
using namespace NMonitoring;

namespace {
    // maximal count for request sensors in one request
    static constexpr i64 MAX_SENSORS_COUNT = 90000;
    // how often should we check for changes
    static constexpr TDuration POLL_INTERVAL = TDuration::MilliSeconds(50);
    // minimal resolution
    static constexpr TDuration MINIMAL_RESOLUTION = TDuration::Seconds(5);

    static constexpr TDuration LOG_METABASE_REQUESTS_LONGER_THAN = TDuration::Seconds(2);
    static constexpr TDuration LOG_STOCKPILE_REQUESTS_LONGER_THAN = TDuration::Seconds(5);

    bool IsDivisiblePoint(const TInstant point, const TDuration period) {
        return point.GetValue() % period.GetValue() == 0;
    }

    size_t GetPointsInPeriod(const TInstant start, const TInstant end, const TDuration period) {
        return (end - start).GetValue() / period.GetValue() + (IsDivisiblePoint(start, period) ? 1 : 0);
    }

    class TStockpileExecutor : public NYasm::NCommon::TBaseThreadedExecutor {
        Y_DECLARE_SINGLETON_FRIEND()
    public:
        static TStockpileExecutor& Get() {
            return *SingletonWithPriority<TStockpileExecutor, 100001>();
        }

    protected:
        TStockpileExecutor()
            : TBaseThreadedExecutor("TStockpileExecutor", 0)
        {
        }
    };

    struct TMetabaseQueryAccumulator {
        THashSet<TSignalName> SignalNames;
        TVector<TSeriesMerger*> Mergers;

        TVector<TSignalName> GetSignalNames() {
            TVector<TSignalName> result(Reserve(SignalNames.size()));
            std::move(std::begin(SignalNames), std::end(SignalNames), std::back_inserter(result));
            return result;
        }
    };

    static const THashMap<TDuration, TString> DURATION_TO_METRICS_NAME = {
        {TDuration::Hours(1), NHistDb::NMetrics::READER_STOCKPILE_UGRAM_1H_POINTS},
        {TDuration::Hours(3), NHistDb::NMetrics::READER_STOCKPILE_UGRAM_3H_POINTS},
        {TDuration::Hours(6), NHistDb::NMetrics::READER_STOCKPILE_UGRAM_6H_POINTS},
        {TDuration::Hours(12), NHistDb::NMetrics::READER_STOCKPILE_UGRAM_12H_POINTS},
        {TDuration::Hours(24), NHistDb::NMetrics::READER_STOCKPILE_UGRAM_24H_POINTS},
    };

    TStockpileReadOptions GetReadOptions(const TSeriesMerger& merger,
                                         const TVector<NStockpile::TSensorId>& sensors) {
        // request every series only once
        TVector<NStockpile::TSensorId> uniqueSensors(Reserve(sensors.size()));
        std::copy(std::begin(sensors), std::end(sensors), std::back_inserter(uniqueSensors));
        SortUniqueBy(uniqueSensors, [](const NStockpile::TSensorId& sensor) {
            return sensor.LocalId;
        });

        TStockpileReadOptions options;
        for (const auto& sensor : uniqueSensors) {
            options.Sensors.emplace_back(sensor);
            options.SensorTypes.emplace_back(merger.AccumulatorType);
        }
        options.Start = merger.Request.Start;
        // We need extra period, because End point is not always included
        options.End = merger.Request.End + merger.Request.Period;

        // Stockpile can response with extra points so we trim them
        options.ResponsePointsWindow = std::make_pair<TInstant, TInstant>(
            NYasm::NCommon::NInterval::NormalizeToIntervalDown(merger.Request.Start, merger.Request.Period),
            NYasm::NCommon::NInterval::NormalizeToIntervalDown(merger.Request.End, merger.Request.Period)
        );

        // check that all series have same kind
        SortUniqueBy(uniqueSensors, [](const NStockpile::TSensorId& sensor) {
            return sensor.Type;
        });

        if (!uniqueSensors.empty() && merger.Request.Period > MINIMAL_RESOLUTION) {
            options.Downsampling.ConstructInPlace(TStockpileDownsamplingOptions{
                .Grid = merger.Request.Period,
                .Aggregation = merger.AccumulatorType,
                .SensorType = uniqueSensors.back().Type
            });
        }

        /*
        if (uniqueSensors.size() == 1) {
            options.Combining.ConstructInPlace(TStockpileCombiningOptions{
                .Aggregation = merger.AccumulatorType,
                .SensorType = uniqueSensors.back().Type
            });
        }
        */

        return options;
    }

    void UpdateMergersStatus(TVector<TSeriesMerger*>& mergers, EStatusCode statusCode) {
        for (auto* merger : mergers) {
            TGuard<TAdaptiveLock> guard(merger->Lock);
            merger->StatusCode = statusCode;
        }
    }

    template <class Q, class F>
    void ProcessChangedQueries(TRequestLog& logger, TDeque<Q*>& changedQueries, TVector<TStockpileExecutor::TFuture>& futures, F&& cb) {
        while (!changedQueries.empty()) {
            auto* query = changedQueries.front();
            changedQueries.pop_front();
            if (!query->Request->IsSuccess()) {
                continue;
            }
            futures.emplace_back(TStockpileExecutor::Get().Add([cb_ = std::move(cb), query, &logger]() {
                try {
                    cb_(*query);
                } catch (...) {
                    logger << TLOG_ERR << "Query callback failed: " << CurrentExceptionMessage();
                }
            }));
        }
    }

    class TSensorValuesMerger final : public ISensorValuesVisitor {
    public:
        TSensorValuesMerger(TSeriesMerger& merger)
            : Merger(merger)
        {
        }

        void OnSensorId(const TSensorId&) override {
        }

        void OnValue(TInstant timestamp, NZoom::NValue::TValue value) override {
            size_t offset = (timestamp - Merger.Request.Start).GetValue() / Merger.Request.Period.GetValue();
            Merger.Accumulators.Mul(value, offset);
        }

    private:
        TSeriesMerger& Merger;
    };
}

TSeriesMerger::TSeriesMerger(const THistoryRequest& request)
    : Request(request)
    , AccumulatorType(GetAggregationType(Request.SignalName))
    , Accumulators(GetAggregationType(Request.SignalName, EAggregationMethod::MetaGroup), Request.GetPoints())
    , StatusCode(EStatusCode::THistoryAggregatedSeries_EStatusCode_OK)
{
}

TSeriesMerger::TSeriesMerger(const THistoryRequest& request, EStatusCode statusCode)
    : Request(request)
    , AccumulatorType(EAccumulatorType::Summ)
    , Accumulators(AccumulatorType, Request.GetPoints())
    , StatusCode(statusCode)
{
}

TSeriesMerger TSeriesMerger::Invalid(const THistoryRequest& request) {
    return TSeriesMerger(request, EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR);
}

TMergerKey TSeriesMerger::GetKey() const {
    return std::make_pair(Request.HostName, Request.SignalName);
}

TMetabaseQuery::TMetabaseQuery(
        const TAtomicSharedPtr<TMetabaseShard>& shard,
        const TAtomicSharedPtr<TGrpcRemoteHost>& metabaseHost,
        const THostName& hostName,
        const TRequestKey& requestKey,
        const TVector<TSignalName>& signalNames,
        NStockpile::TGrpcCompletionQueue& grpcQueue,
        TDeque<NPrivate::TMetabaseQuery*>& readyQueue,
        long singleRequestSizeLimit,
        TLog& logger)
    : Shard(shard)
    , MetabaseHost(metabaseHost)
    , RequestKey(requestKey)
    , GrpcQueue(grpcQueue)
    , ReadyQueue(readyQueue)
    , MaxRequestDuration()
    , TotalRequestDuration()
{
    Request = TMetabaseFindSensor::Make(
        hostName,
        requestKey,
        signalNames,
        shard,
        metabaseHost,
        singleRequestSizeLimit,
        logger);
}

void TMetabaseQuery::Schedule() {
    Request->SetCallback([this](const auto&) { this->Callback(); });
    Request->ScheduleForExecute(GrpcQueue);
}

void TMetabaseQuery::Callback() {
    auto requestResult = Request->GetResult();
    if (Result.empty()) {
        Result = std::move(requestResult);
    } else {
        std::move(requestResult.begin(), requestResult.end(), std::back_inserter(Result));
    }

    MaxRequestDuration = Max(MaxRequestDuration, Request->GetDuration());
    TotalRequestDuration += Request->GetDuration();

    auto nextRequest = Request->Next();
    if (!nextRequest || !Request->IsSuccess()) {
        PreviousRequest.Reset();
        ReadyQueue.emplace_back(this);
    } else {
        // we cannot just reset the current request in callback as it will get deleted immediately,
        // so we store it in PreviousRequest
        PreviousRequest = std::move(Request);
        Request = std::move(nextRequest);
        Schedule();
    }
}

TStockpileQuery::TStockpileQuery(
        TSeriesMerger& merger,
        const TAtomicSharedPtr<NStockpile::TStockpileShard>& shard,
        const TVector<NStockpile::TSensorId>& sensors,
        TLog& logger)
    : Merger(merger)
    , Shard(shard)
    , RequestDuration()
{
    TStockpileReadOptions options = GetReadOptions(merger, sensors);
    Request.ConstructInPlace(options, Shard, logger);
}

void TStockpileQuery::Schedule(TGrpcCompletionQueue& grpcQueue, TDeque<TStockpileQuery*>& readyQueue) {
    Request->SetCallback([this, &readyQueue](const auto&) {
        RequestDuration = Request->GetDuration();
        readyQueue.emplace_back(this);
    });
    Request->ScheduleForExecute(grpcQueue);
}

TStockpileReader::TStockpileReader(
        NStockpile::TStockpileState& stockpileState,
        TDuration metabaseTimeout,
        TDuration stockpileTimeout,
        long metabaseFindSingleRequestLimit,
        const TVector<THistoryRequest>& requests,
        const NMonitoring::TRequestLog& logger)
    : StockpileState(stockpileState)
    , MetabaseTimeout(metabaseTimeout)
    , StockpileTimeout(stockpileTimeout)
    , MetabaseFindSingleRequestLimit(metabaseFindSingleRequestLimit)
    , Requests(requests)
    , SeriesMergers(Reserve(Requests.size()))
    , Logger(logger)
    , GrpcHandler(Logger.GetLogger())
{
}

TStockpileReader::TStockpileReader(
        NStockpile::TStockpileState& stockpileState,
        TDuration metabaseTimeout,
        TDuration stockpileTimeout,
        const TVector<THistoryRequest>& requests,
        const NMonitoring::TRequestLog& logger)
    : TStockpileReader(stockpileState, metabaseTimeout, stockpileTimeout, NStockpile::TMetabaseFindSensor::DEFAULT_LIMIT,
        requests, logger)
{
}

TStockpileReader::TStockpileReader(
        NStockpile::TStockpileState& stockpileState,
        TDuration metabaseTimeout,
        TDuration stockpileTimeout,
        const TCompatRequest& compatRequest,
        const NMonitoring::TRequestLog& logger)
    : TStockpileReader(stockpileState, metabaseTimeout, stockpileTimeout, GetRequestsFromCompat(compatRequest), logger)
{
}

TMetabaseShardKey TStockpileReader::GetMetabaseShardKey(const THistoryRequest& request) const {
    return TMetabaseShardKey::Make(
        request.HostName,
        request.RequestKey.GetRequestKey(),
        request.SignalName,
        StockpileState.GetShardsCount(request.RequestKey.GetRequestKey().GetItype())
    );
}

void TStockpileReader::PlanMetabaseQueries() {
    THashMap<TMetabaseShardKey, THashMap<std::pair<THostName, TInternedRequestKey>, TMetabaseQueryAccumulator>> requestsPerShard;
    requestsPerShard.reserve(Requests.size());

    for (const auto& request : Requests) {
        if (request.SignalName.IsOld()) {
            SeriesMergers.emplace_back(request, EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND);
        } else if (request.Start > request.End
                || request.Period.GetValue() % MINIMAL_RESOLUTION.GetValue()
                || request.Period < MINIMAL_RESOLUTION) {
            SeriesMergers.emplace_back(TSeriesMerger::Invalid(request));
        } else {
            // group history requests into metabase queries
            auto& accumulator =
                requestsPerShard[GetMetabaseShardKey(request)][std::make_pair(request.HostName, request.RequestKey)];
            accumulator.SignalNames.emplace(request.SignalName);
            accumulator.Mergers.emplace_back(&SeriesMergers.emplace_back(request));
        }
    }

    // and create requests to metabase shards
    for (auto& [shardKey, queryMap] : requestsPerShard) {
        try {
            auto shardState = StockpileState.ResolveOneMetabaseShardForRead(shardKey);
            for (auto& [queryKey, accumulator]: queryMap) {
                auto& [hostName, requestKey] = queryKey;
                if (!shardState.IsReady()) {
                    UpdateMergersStatus(accumulator.Mergers, EStatusCode::THistoryAggregatedSeries_EStatusCode_PARTIAL);
                    Logger << TLOG_ERR << "Metabase's shard " << shardKey << " for " << requestKey
                           << ((shardState.GrpcRemoteHost) ? " not found on any host" : " not ready");
                    continue;
                }
                auto& query = MetabaseQueries.emplace_back(
                    shardState.Shard,
                    shardState.GrpcRemoteHost,
                    hostName,
                    requestKey.GetRequestKey(),
                    accumulator.GetSignalNames(),
                    GrpcHandler.GetQueue(),
                    ChangedMetabaseQueries,
                    MetabaseFindSingleRequestLimit,
                    Logger.GetLogger());
                for (auto* merger : accumulator.Mergers) {
                    query.Mergers[merger->GetKey()].emplace_back(merger);
                }
                query.Schedule();
            }
        } catch (const TShardNotFoundError&) {
            for (auto& [queryKey, accumulator]: queryMap) {
                UpdateMergersStatus(accumulator.Mergers, EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND);
                Logger << TLOG_WARNING << "Metabase's shard " << shardKey << " was not found for " << queryKey.second;
            }
        }
    }

    Logger << TLOG_INFO << "Request consists of " << Requests.size() << " queries grouped into "
           << MetabaseQueries.size() << " metabase calls";
}

namespace {
    class TMetabaseQueryPostProcessor {
    public:
        using TStockpileRequestMap = THashMap<std::pair<TSeriesMerger*, TStockpileShardId>, TVector<TSensorId>>;

        TMetabaseQueryPostProcessor()
            : TotalSensorCount(0)
            , FoundSensorCount(0)
            , Lock()
            , RequestsPerShard() {
        }

        size_t GetTotalSensorCount() const {
            return AtomicGet(TotalSensorCount);
        }

        size_t GetFoundSensorCount() const {
            return AtomicGet(FoundSensorCount);
        }

        // unsafe as it does not lock the internal lock
        TStockpileRequestMap& GetRequestsPerShardUnsafe() {
            return RequestsPerShard;
        }

        void PostProcessQuery(TMetabaseQuery& query) {
            THashMap<TMergerKey, THashMap<TInstanceKey, TSensorId>> mergerInstances;
            {
                if (!query.Request->IsSuccess()) {
                    ythrow yexception() << "metabase call failed";
                }
                const auto& keysWithSensors = query.Result;
                AtomicAdd(FoundSensorCount, keysWithSensors.size());

                for (const auto& [seriesKey, sensorId]: keysWithSensors) {
                    auto mergerKey = TMergerKey(seriesKey.InstanceKey.GetHostName(), seriesKey.SignalName);
                    mergerInstances[mergerKey].emplace(seriesKey.InstanceKey, sensorId);
                }
                query.Clear(); // clear successful request
            }

            for (auto& [mergerKey, mergers] : query.Mergers) {
                auto it = mergerInstances.find(mergerKey);
                if (it == mergerInstances.end()) {
                    UpdateMergersStatus(mergers, EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND);
                    continue;
                }
                auto& instanceSensors = it->second;

                TDynamicFilter dynamicFilter(query.RequestKey);
                for (const auto& [instanceKey, sensorId] : instanceSensors) {
                    dynamicFilter.Feed(instanceKey);
                }

                auto instanceKeys = dynamicFilter.Resolve();
                if (instanceKeys.empty()) {
                    UpdateMergersStatus(mergers, EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND);
                } else if (AtomicGet(TotalSensorCount) > MAX_SENSORS_COUNT) {
                    UpdateMergersStatus(mergers, EStatusCode::THistoryAggregatedSeries_EStatusCode_LIMIT_EXCEEDED);
                } else {
                    TGuard<TAdaptiveLock> guard(Lock);
                    for (const auto& instanceKey : instanceKeys) {
                        const auto& sensorId = instanceSensors.at(instanceKey);
                        if (sensorId.Type == yandex::solomon::model::MetricType::METRIC_TYPE_UNSPECIFIED) {
                            continue;
                        }

                        // let's find which requests should go into which shards
                        for (auto* merger: mergers) {
                            RequestsPerShard[std::make_pair(merger, sensorId.ShardId)].emplace_back(sensorId);
                        }

                        AtomicIncrement(TotalSensorCount);
                    }
                }
            }
        }
    private:
        TAtomic TotalSensorCount;
        TAtomic FoundSensorCount;

        TAdaptiveLock Lock;
        TStockpileRequestMap RequestsPerShard;
    };
}

void TStockpileReader::ConsumeMetabaseQueries(TInstant overallDeadline) {
    NMonitoring::TMeasuredMethod timer(Logger, NMetrics::READER_METABASE_TOTAL_TIME, NMetrics::READER_METABASE_TOTAL_TIME);

    TInstant deadline = MetabaseTimeout.ToDeadLine();
    if (overallDeadline && overallDeadline < deadline) {
        deadline = overallDeadline;
    }

    TMetabaseQueryPostProcessor postProcessor;
    TVector<TStockpileExecutor::TFuture> futures(Reserve(MetabaseQueries.size()));
    while (MetabaseTimeout && GrpcHandler.WaitAsync(POLL_INTERVAL) && TInstant::Now() < deadline) {
        ProcessChangedQueries(Logger, ChangedMetabaseQueries, futures, [&postProcessor](TMetabaseQuery& query) {
            postProcessor.PostProcessQuery(query);
        });
    }
    NThreading::WaitExceptionOrAll(futures).GetValueSync();

    for (auto& query : MetabaseQueries) {
        if (query.TotalRequestDuration > LOG_METABASE_REQUESTS_LONGER_THAN) {
            Logger << TLOG_INFO << "Slow metabase request to " << *query.MetabaseHost << " took: "
                   << query.TotalRequestDuration.MilliSeconds() << " ms total, "
                   << query.MaxRequestDuration.MilliSeconds() << " ms max";
        }
        if (!query.Request) {
            // successful requests are cleared
            continue;
        } else if (!query.Request->IsFinished()) {
            query.VisitMergers([](TSeriesMerger& merger) {
                merger.StatusCode = EStatusCode::THistoryAggregatedSeries_EStatusCode_TIME_EXCEEDED;
            });
            TUnistat::Instance().PushSignalUnsafe(NMetrics::READER_METABASE_TIMEOUTS, 1);
            Logger << TLOG_ERR << "Metabase shard " << query.Shard->GetShardKey() << " not answered for " << query.RequestKey << " before deadline";
        } else if (!query.Request->IsSuccess()) {
            query.VisitMergers([](TSeriesMerger& merger) {
                merger.StatusCode = EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR;
            });
            TUnistat::Instance().PushSignalUnsafe(NMetrics::READER_METABASE_ERRORS, 1);
            Logger << TLOG_ERR << "Query to metabase shard " << query.Shard->GetShardKey() << " failed for " << query.RequestKey;
        }
        query.Request->Cancel();
    }

    // and create requests to stockpile shards
    for (auto& [key, sensors] : postProcessor.GetRequestsPerShardUnsafe()) {
        const auto& [merger, shardId] = key;
        try {
            auto& query = StockpileQueries.emplace_back(
                *merger,
                StockpileState.ResolveOneStockpileShardForRead(shardId),
                sensors,
                Logger.GetLogger());
            if (merger->Request.Period > TDuration::Minutes(5)) {
                for (auto& sensor: sensors) {
                    if (sensor.Type == yandex::solomon::model::MetricType::HIST) {
                        TUnistat::Instance().PushSignalUnsafe(
                            NHistDb::NMetrics::READER_STOCKPILE_UGRAM_DOWNSAMPLED_5M_POINTS,
                            GetPointsInPeriod(
                                merger->Request.Start,
                                merger->Request.End,
                                TDuration::Minutes(5)
                            ));
                        TUnistat::Instance().PushSignalUnsafe(
                            DURATION_TO_METRICS_NAME.at(merger->Request.Period),
                            merger->Request.GetPoints());
                    }
                }
            }
            query.Schedule(GrpcHandler.GetQueue(), ChangedStockpileQueries);
        } catch (const TShardNotFoundError&) {
            merger->StatusCode = EStatusCode::THistoryAggregatedSeries_EStatusCode_PARTIAL;
            Logger << TLOG_ERR << "Stockpile shard " << shardId << " was not found for " << merger->Request.RequestKey;
        }
    }

    Logger << TLOG_INFO << "Request needs " << postProcessor.GetTotalSensorCount() << " sensors (from "
        << postProcessor.GetFoundSensorCount() << ") grouped into " << StockpileQueries.size() << " stockpile calls";
    TUnistat::Instance().PushSignalUnsafe(NHistDb::NMetrics::READER_METABASE_SENSORS_PER_REQUEST,
        postProcessor.GetFoundSensorCount());
    TUnistat::Instance().PushSignalUnsafe(NHistDb::NMetrics::READER_STOCKPILE_SENSORS_PER_REQUEST,
        postProcessor.GetTotalSensorCount());
}

void TStockpileReader::ConsumeStockpileQueries(TInstant overallDeadline) {
    NMonitoring::TMeasuredMethod timer(Logger, NMetrics::READER_STOCKPILE_TOTAL_TIME, NMetrics::READER_STOCKPILE_TOTAL_TIME);

    TInstant deadline = StockpileTimeout.ToDeadLine();
    if (overallDeadline && overallDeadline < deadline) {
        deadline = overallDeadline;
    }

    TAtomic totalPointCount = 0;
    TVector<TStockpileExecutor::TFuture> futures(Reserve(MetabaseQueries.size()));
    while (StockpileTimeout && GrpcHandler.WaitAsync(POLL_INTERVAL) && TInstant::Now() < deadline) {
        ProcessChangedQueries(Logger, ChangedStockpileQueries, futures, [&totalPointCount](TStockpileQuery& query) {
            TGuard<TAdaptiveLock> guard(query.Merger.Lock);
            TSensorValuesMerger visitor(query.Merger);
            query.Request->VisitResult(visitor);
            query.Request.Clear();
            AtomicAdd(totalPointCount, query.Merger.Accumulators.Len());
        });
    }
    NThreading::WaitExceptionOrAll(futures).GetValueSync();

    for (auto& query : StockpileQueries) {
        if (query.RequestDuration > LOG_STOCKPILE_REQUESTS_LONGER_THAN) {
            Logger << TLOG_INFO << "Slow stockpile request to " << *query.Shard->GetHost() << " took: "
                   << query.RequestDuration.MilliSeconds() << " ms total";
        }
        if (query.Request.Empty()) {
            continue;
        } else if (!query.Request->IsFinished()) {
            query.Merger.StatusCode = EStatusCode::THistoryAggregatedSeries_EStatusCode_TIME_EXCEEDED;
            TUnistat::Instance().PushSignalUnsafe(NMetrics::READER_STOCKPILE_TIMEOUTS, 1);
            Logger << TLOG_ERR << "Stockpile shard " << query.Shard->GetShardId() << " not answered for "
                   << query.Merger.Request.RequestKey << " before deadline";
        } else if (!query.Request->IsSuccess()) {
            query.Merger.StatusCode = EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR;
            TUnistat::Instance().PushSignalUnsafe(NMetrics::READER_STOCKPILE_ERRORS, 1);
            Logger << TLOG_ERR << "Query to stockpile shard " << query.Shard->GetShardId() << " failed for "
                   << query.Merger.Request.RequestKey;
        }
        query.Request->Cancel();
    }

    Logger << TLOG_INFO << "Request consists of " << AtomicGet(totalPointCount) << " points";
}

void TStockpileReader::Execute(TInstant deadline) {
    PlanMetabaseQueries();

    TUnistat::Instance().PushSignalUnsafe(NMetrics::READER_METABASE_QUERIES, MetabaseQueries.size());

    ConsumeMetabaseQueries(deadline);

    TUnistat::Instance().PushSignalUnsafe(NMetrics::READER_STOCKPILE_QUERIES, StockpileQueries.size());

    ConsumeStockpileQueries(deadline);
}

void TStockpileReader::Execute() {
    Execute(TInstant::Zero());
}

THistoryResponse TStockpileReader::GetResponse(size_t offset) {
    auto& merger(SeriesMergers.at(offset));
    TVector<NZoom::NValue::TValue> values(Reserve(merger.Accumulators.Len()));
    for (const auto idx : xrange(merger.Accumulators.Len())) {
        values.emplace_back(merger.Accumulators.GetValue(idx));
    }
    merger.Accumulators.Clean();
    return THistoryResponse(merger.Request, merger.Request.Start, std::move(values), merger.StatusCode);
}

size_t TStockpileReader::ResponseCount() const {
    return SeriesMergers.size();
}

TVector<THistoryResponse> TStockpileReader::CreateResult() {
    TVector<THistoryResponse> result(Reserve(ResponseCount()));
    for (const auto idx : xrange(ResponseCount())) {
        result.emplace_back(GetResponse(idx));
    }
    return result;
}

TVector<TCompatResponse> TStockpileReader::CreateCompatResult() {
    const THistoryRequest* request = nullptr;
    THashMap<TInternedRequestKey, TVector<TSeriesMerger*>> mergersByRequestKey;
    for (auto& merger : SeriesMergers) {
        mergersByRequestKey[merger.Request.RequestKey].emplace_back(&merger);
        request = &merger.Request;
    }

    TVector<TCompatResponse> result(Reserve(mergersByRequestKey.size() * request->GetPoints()));
    for (auto& [requestKey, mergers] : mergersByRequestKey) {
        for (const auto idx : xrange(request->GetPoints())) {
            TVector<std::pair<TSignalName, TValue>> values(Reserve(mergers.size()));
            for (auto* merger : mergers) {
                if (idx < merger->Accumulators.Len()) {
                    values.emplace_back(merger->Request.SignalName, merger->Accumulators.GetValue(idx));
                }
            }
            result.emplace_back(
                requestKey.GetRequestKey(),
                request->Start + request->Period * idx,
                std::move(values)
            );
        }
        for (auto* merger : mergers) {
            merger->Accumulators.Clean();
        }
    }

    return result;
}

TVector<THistoryResponse> TStockpileReader::Read() {
    Execute();
    return CreateResult();
}

TVector<TCompatResponse> TStockpileReader::ReadCompat() {
    Execute();
    return CreateCompatResult();
}

TVector<NZoom::NProtobuf::THistoryRequest> NHistDb::GetRequestsFromCompat(const TCompatRequest& request) {
    TVector<THistoryRequest> result;
    for (const auto& [requestKey, signals] : request.TagSignals) {
        TInternedRequestKey internedRequestKey(requestKey.ToNamed());
        for (const auto& signal : signals) {
            result.emplace_back(THistoryRequest{
                .Start = request.Start,
                .End = request.End,
                .Period = request.Period,
                .HostName = request.HostName,
                .RequestKey = internedRequestKey,
                .SignalName = signal
            });
        }
    }
    return result;
}
