#pragma once

#include <infra/yasm/histdb/components/formats/something.h>
#include <infra/yasm/common/points/accumulators/accumulators.h>
#include <infra/yasm/zoom/components/serialization/history/history.h>
#include <infra/yasm/stockpile_client/state.h>
#include <infra/monitoring/common/logging.h>

#include <util/generic/list.h>
#include <util/generic/deque.h>

namespace NHistDb {
    namespace NPrivate {
        struct TSeriesMerger;
        struct TMetabaseQuery;
        struct TStockpileQuery;
    }

    struct TCompatRequest {
        TInstant Start;
        TInstant End;
        TDuration Period;
        NZoom::NHost::THostName HostName;
        TSomethingFormat::TTagSignals TagSignals;
    };

    struct TCompatResponse {
        TCompatResponse(const NTags::TRequestKey& requestKey, TInstant timestamp, NZoom::NRecord::TRecord record)
            : RequestKey(requestKey)
            , Timestamp(timestamp)
            , Record(std::move(record))
        {
        }

        const NTags::TRequestKey& RequestKey;
        TInstant Timestamp;
        NZoom::NRecord::TRecord Record;
    };

    TVector<NZoom::NProtobuf::THistoryRequest> GetRequestsFromCompat(const TCompatRequest& request);

    class TStockpileReader {
    public:
        TStockpileReader(NStockpile::TStockpileState& stockpileState,
                         TDuration metabaseTimeout,
                         TDuration stockpileTimeout,
                         const TVector<NZoom::NProtobuf::THistoryRequest>& requests,
                         const NMonitoring::TRequestLog& logger);

        TStockpileReader(NStockpile::TStockpileState& stockpileState,
                         TDuration metabaseTimeout,
                         TDuration stockpileTimeout,
                         const TCompatRequest& compatRequest,
                         const NMonitoring::TRequestLog& logger);

        TStockpileReader(NStockpile::TStockpileState& stockpileState,
                         TDuration metabaseTimeout,
                         TDuration stockpileTimeout,
                         long metabaseFindSingleRequestLimit,
                         const TVector<NZoom::NProtobuf::THistoryRequest>& requests,
                         const NMonitoring::TRequestLog& logger);

        void Execute(TInstant deadline);
        void Execute();

        NZoom::NProtobuf::THistoryResponse GetResponse(size_t offset);
        size_t ResponseCount() const;

        TVector<NZoom::NProtobuf::THistoryResponse> CreateResult();
        TVector<TCompatResponse> CreateCompatResult();

        TVector<NZoom::NProtobuf::THistoryResponse> Read();
        TVector<TCompatResponse> ReadCompat();

    private:
        NStockpile::TMetabaseShardKey GetMetabaseShardKey(const NZoom::NProtobuf::THistoryRequest& request) const;

        void PlanMetabaseQueries();
        void ConsumeMetabaseQueries(TInstant deadline);
        void ConsumeStockpileQueries(TInstant deadline);

        NStockpile::TStockpileState& StockpileState;
        const TDuration MetabaseTimeout;
        const TDuration StockpileTimeout;
        const long MetabaseFindSingleRequestLimit;
        TVector<NZoom::NProtobuf::THistoryRequest> Requests;
        TVector<NPrivate::TSeriesMerger> SeriesMergers;
        NMonitoring::TRequestLog Logger;

        TList<NPrivate::TMetabaseQuery> MetabaseQueries;
        TDeque<NPrivate::TMetabaseQuery*> ChangedMetabaseQueries;

        TList<NPrivate::TStockpileQuery> StockpileQueries;
        TDeque<NPrivate::TStockpileQuery*> ChangedStockpileQueries;

        NStockpile::TGrpcStateHandler GrpcHandler;
    };

    namespace NPrivate {
        using TMergerKey = std::pair<NZoom::NHost::THostName, NZoom::NSignal::TSignalName>;
        using EStatusCode = NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode;

        struct TSeriesMerger {
            TSeriesMerger(const NZoom::NProtobuf::THistoryRequest& request);
            TSeriesMerger(const NZoom::NProtobuf::THistoryRequest& request, EStatusCode statusCode);

            static TSeriesMerger Invalid(const NZoom::NProtobuf::THistoryRequest& request);

            const NZoom::NProtobuf::THistoryRequest& Request;
            NZoom::NAccumulators::EAccumulatorType AccumulatorType;
            NZoom::NAccumulators::TCompactAccumulatorsArray Accumulators;
            // last error should win
            EStatusCode StatusCode;
            TAdaptiveLock Lock;

            TMergerKey GetKey() const;
        };

        struct TMetabaseQuery {
            TAtomicSharedPtr<NStockpile::TMetabaseShard> Shard;
            TAtomicSharedPtr<NStockpile::TGrpcRemoteHost> MetabaseHost;
            const NTags::TRequestKey& RequestKey;

            THolder<NStockpile::TMetabaseFindSensor> Request;
            THolder<NStockpile::TMetabaseFindSensor> PreviousRequest;
            NStockpile::TGrpcCompletionQueue& GrpcQueue;
            TDeque<NPrivate::TMetabaseQuery*>& ReadyQueue;

            THashMap<std::pair<NZoom::NHost::THostName, NZoom::NSignal::TSignalName>, TVector<TSeriesMerger*>> Mergers;
            TVector<std::pair<NStockpile::TSeriesKey, NStockpile::TSensorId>> Result;

            TDuration MaxRequestDuration;
            TDuration TotalRequestDuration;

            TMetabaseQuery(
                const TAtomicSharedPtr<NStockpile::TMetabaseShard>& shard,
                const TAtomicSharedPtr<NStockpile::TGrpcRemoteHost>& metabaseHost,
                const NZoom::NHost::THostName& hostName,
                const NTags::TRequestKey& requestKey,
                const TVector<NZoom::NSignal::TSignalName>& signalNames,
                NStockpile::TGrpcCompletionQueue& grpcQueue,
                TDeque<NPrivate::TMetabaseQuery*>& readyQueue,
                long singleRequestSizeLimit,
                TLog& logger);

            void Schedule();
            void Callback();
            void Clear() {
                Request.Reset();
                PreviousRequest.Reset();
                Result.clear();
            }

            template <typename TVisitor>
            void VisitMergers(TVisitor cb) {
                for (auto& pair : Mergers) {
                    for (auto* merger : pair.second) {
                        cb(*merger);
                    }
                }
            }
        };

        struct TStockpileQuery {
            TSeriesMerger& Merger;
            TAtomicSharedPtr<NStockpile::TStockpileShard> Shard;
            TMaybe<NStockpile::TStockpileReadManyState> Request;
            TDuration RequestDuration;

            TStockpileQuery(TSeriesMerger& merger,
                            const TAtomicSharedPtr<NStockpile::TStockpileShard>& shard,
                            const TVector<NStockpile::TSensorId>& sensors,
                            TLog& logger);
            void Schedule(NStockpile::TGrpcCompletionQueue& grpcQueue, TDeque<TStockpileQuery*>& readyQueue);
        };
    }
}
