#include "stockpile_cleaner.h"
#include "stockpile_reader.h"
#include "stockpile_writer.h"

#include <infra/yasm/histdb/components/dumper/common_ut.h>
#include <infra/yasm/stockpile_client/state_ut.h>

#include <library/cpp/testing/unittest/gtest.h>

using namespace NHistDb;
using namespace NHistDb::NStockpile;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;
using namespace NZoom::NYasmConf;
using namespace NZoom::NHost;
using namespace NZoom::NHgram;
using namespace NZoom::NSubscription;
using namespace NZoom::NProtobuf;
using EHistoryStatusCode = NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode;

namespace {
    constexpr TStringBuf UNIT_TEST_ITYPE = "yasm_unittest";
    const TString UNIT_TEST_PROJECT_ID = TString::Join(STOCKPILE_YASM_PROJECTS_PREFIX, UNIT_TEST_ITYPE);
    THostName GROUP(TStringBuf("SOME_GROUP"));
    THostName HOST(TStringBuf("some_host"));
    constexpr size_t DEFAULT_METABASE_TIMEOUT = 30;
    constexpr size_t DEFAULT_STOCKPILE_TIMEOUT = 30;

    class THelper {
    public:
        THelper()
            : Client(NStockpile::NTest::TClient::Get())
            , Period(TRecordPeriod::Get("s5"))
            , Stats(1)
        {
        }

        template <class TRequests>
        TStockpileReader CreateReader(const TRequests& requests, TDuration metabaseTimeout, TDuration stockpileTimeout) {
            return TStockpileReader(
                Client.StockpileState,
                metabaseTimeout,
                stockpileTimeout,
                requests,
                Client.Log
            );
        }

        template <class TRequests>
        TStockpileReader CreateReader(const TRequests& requests, TDuration metabaseTimeout, TDuration stockpileTimeout,
            long metabaseFindSizeLimit) {
            return TStockpileReader(
                Client.StockpileState,
                metabaseTimeout,
                stockpileTimeout,
                metabaseFindSizeLimit,
                requests,
                Client.Log
            );
        }

        TVector<THistoryResponse> Reply(const THistoryRequest& request, size_t metabaseTimeout = DEFAULT_METABASE_TIMEOUT,
            size_t stockpileTimeout = DEFAULT_STOCKPILE_TIMEOUT)
        {
            auto reader = CreateReader(
                TVector<THistoryRequest>{request},
                TDuration::Seconds(metabaseTimeout),
                TDuration::Seconds(stockpileTimeout)
            );
            return reader.Read();
        }

        THistoryResponse ReplyOne(const THistoryRequest& request, size_t metabaseTimeout = DEFAULT_METABASE_TIMEOUT,
            size_t stockpileTimeout = DEFAULT_STOCKPILE_TIMEOUT)
        {
            auto responses = Reply(request, metabaseTimeout, stockpileTimeout);
            UNIT_ASSERT_VALUES_EQUAL(responses.size(), 1);
            return std::move(responses.back());
        }

        THistoryResponse ReplyOneWithLimit(const THistoryRequest& request, size_t metabaseTimeout, size_t stockpileTimeout,
            long metabaseFindSizeLimit) {

            auto reader = CreateReader(
                TVector<THistoryRequest>{request},
                TDuration::Seconds(metabaseTimeout),
                TDuration::Seconds(stockpileTimeout),
                metabaseFindSizeLimit
            );
            auto responses = reader.Read();
            UNIT_ASSERT_VALUES_EQUAL(responses.size(), 1);
            return std::move(responses.back());
        }

        TVector<TCompatResponse> ReplyCompat(const TCompatRequest& request, size_t metabaseTimeout = DEFAULT_METABASE_TIMEOUT,
            size_t stockpileTimeout = DEFAULT_STOCKPILE_TIMEOUT)
        {
            auto reader(CreateReader(
                request,
                TDuration::Seconds(metabaseTimeout),
                TDuration::Seconds(stockpileTimeout)
            ));
            return reader.ReadCompat();
        }

        void Dump(const IRecordDescriptor& record) {
            TStockpileWriter writer(Client.Log, Client.StockpileState, TStockpileWriterSettings(), 0, Stats);
            writer.OnRecord(record);
            writer.Finish();
        }

        void ClearHistory(const THostName& host, TInstant end, TString projectId = UNIT_TEST_PROJECT_ID) {

            StockpileDataCleaner cleaner(Client.Log, Client.StockpileStateForClear);
            cleaner.SetLimit(10);
            for (auto shardKey : Client.StockpileStateForClear.GetMetabaseShardKeys(true)) {
                if (shardKey.GetProjectId() == projectId) {
                    cleaner.ClearStockpileData(host, shardKey, end);
                }
            }
        }

        TMockedRecordDescriptor CreateRecord(TSignalName signalName, const TVector<TValue>& values, const TRecordPeriod& period) const {
            TInstant now(period.GetStartTime(TInstant::Now()));
            return CreateRecord(signalName, values, period, now);
        }

        TMockedRecordDescriptor CreateRecord(TSignalName signalName, const TVector<TValue>& values, const TRecordPeriod& period,
            TInstant now) const {

            auto key = NTags::TInstanceKey::FromNamed("yasm_unittest|geo=sas|prj").AddGroupAndHost(GROUP, HOST);
            TMockedRecordDescriptor record;
            record.SetHostName(key.GetHostName().GetName());
            record.SetInstanceKey(key);
            record.SetSignalName(signalName);
            record.SetValues(values);
            record.SetStartTime(now - period.GetResolution() * values.size());
            record.SetEndTime(now - period.GetResolution());
            return record;
        }

        TMockedRecordDescriptor CreateRecord(TSignalName signalName, const TVector<TValue>& values) const {
            return CreateRecord(signalName, values, Period);
        }

        TMockedRecordDescriptor CreateDummyRecord() const {
            TVector<TValue> values;
            for (auto val: xrange(0, 15)) {
                values.emplace_back(val);
            }
            return CreateRecord(TStringBuf("some_test_signal_summ"), values);
        }

        THistoryRequest CreateDummyRequest(const TMockedRecordDescriptor& record) const {
            return THistoryRequest{
                .Start = record.GetStartTime(),
                .End = record.GetEndTime(),
                .Period = GetPeriod().GetResolution(),
                .HostName = record.GetHostName(),
                .RequestKey = TInternedRequestKey(CreateRequestKeyString(record)),
                .SignalName = record.GetSignalName()
            };
        }

        THistoryRequest CreateDummyRequest(TStringBuf itype, THostName hostName, TSignalName signalName, size_t valueCount,
            const TRecordPeriod& period, TInstant now) const {

            return THistoryRequest{
                .Start = now - Period.GetResolution() * valueCount,
                .End = now - Period.GetResolution(),
                .Period = period.GetResolution(),
                .HostName = hostName,
                .RequestKey = TInternedRequestKey(TString::Join("itype=", itype)),
                .SignalName = signalName
            };
        }

        TCompatRequest CreateCompatRequest(const TMockedRecordDescriptor& record, TDuration resolution = TDuration::Zero()) const {
            TCompatRequest request{
                record.GetStartTime(),
                record.GetEndTime(),
                resolution ? resolution : GetPeriod().GetResolution(),
                record.GetHostName(),
                {}
            };
            request.TagSignals.emplace_back(
                NTags::TRequestKey::FromString(CreateRequestKeyString(record)),
                TVector<TSignalName>{record.GetSignalName()}
            );
            return request;
        }

        TInternedRequestKey GetRequestKey() const {
            return TInternedRequestKey(TStringBuf("itype=stockpile_reader"));
        }

        const TRecordPeriod& GetPeriod() const {
            return Period;
        }

    private:
        TString CreateRequestKeyString(const TMockedRecordDescriptor& record) const {
            return TString::Join("itype=", record.GetInstanceKey().GetItype());
        }

        NStockpile::NTest::TClient& Client;
        TRecordPeriod Period;
        TStockpileDumperStats Stats;
    };
}

Y_UNIT_TEST_SUITE(TStockpileReaderTest) {
    Y_UNIT_TEST(NoTimeToWaitMetabase) {
        THelper helper;
        auto record(helper.CreateDummyRecord());
        helper.Dump(record);

        auto request(helper.CreateDummyRequest(record));
        auto response(helper.ReplyOne(request, 0, DEFAULT_STOCKPILE_TIMEOUT));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EHistoryStatusCode::THistoryAggregatedSeries_EStatusCode_TIME_EXCEEDED);
    }

    Y_UNIT_TEST(NoTimeToWaitStockpile) {
        THelper helper;
        auto record(helper.CreateDummyRecord());
        helper.Dump(record);

        auto request(helper.CreateDummyRequest(record));
        auto response(helper.ReplyOne(request, DEFAULT_METABASE_TIMEOUT, 0));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EHistoryStatusCode::THistoryAggregatedSeries_EStatusCode_TIME_EXCEEDED);
    }

    Y_UNIT_TEST(TestFloatValues) {
        THelper helper;
        THashMap<char, double> expectValues5m = {
                {'v', 29.5},
                {'x', 59},
                {'n', 0},
                {'m', 1770 },
                {'e', 1770 },
        };

        TVector<TSignalName> signals;
        TString accLetters1("vtxnme");
        TString accLetters2("vxnme");
        TString base("test_float_t");
        for (char a1 : accLetters1) {
            for (char a2 : accLetters2) {
                signals.emplace_back(TStringBuilder() << base << a1 << a1 << a2);
            }
        }

        TVector<TValue> values;
        TVector<TValue> averValues;
        for (auto id : xrange(60)) {
            values.emplace_back(id);
            averValues.emplace_back(id, 1);
        }

        TVector<TMockedRecordDescriptor> dumpedRecords;
        for (const auto& signal: signals) {
            dumpedRecords.push_back(helper.CreateRecord(signal, values));
            helper.Dump(dumpedRecords.back());
        }

        for (size_t signalIdx = 0; signalIdx < signals.size(); ++signalIdx) {
            auto& signal = signals[signalIdx];
            auto& record = dumpedRecords[signalIdx];
            const auto& signalName = signal.GetName();

            auto request = helper.CreateDummyRequest(record);
            auto response = helper.ReplyOne(request);
            char metaAggr = *(++signalName.rbegin());
            char rollapAggr = signalName.back();
            if (metaAggr == 'v') {
                UNIT_ASSERT_VALUES_EQUAL_C(response.Series.GetValues(), averValues, signalName);
            } else {
                UNIT_ASSERT_VALUES_EQUAL_C(response.Series.GetValues(), values, signalName);
            }

            request.Period = TDuration::Minutes(5);
            auto response5m = helper.ReplyOne(request);
            const auto& values5m = response5m.Series.GetValues();
            UNIT_ASSERT_VALUES_EQUAL_C(values5m.size(), 1, signalName);

            if (metaAggr == 'v') {
                if (rollapAggr == 'v') {
                    UNIT_ASSERT_VALUES_EQUAL_C(values5m[0], TValue(1770, 60), signalName);
                } else {
                    auto expect = expectValues5m.at(rollapAggr);
                    UNIT_ASSERT_VALUES_EQUAL_C(values5m[0], TValue(expect, 1), signalName);
                }
            } else {
                auto expect = expectValues5m.at(rollapAggr);
                UNIT_ASSERT_VALUES_EQUAL_C(values5m[0], TValue(expect), signalName);
            }
        }
    }

    Y_UNIT_TEST(CompatRequest) {
        THelper helper;

        TVector<TValue> values;
        for (auto val: xrange(0, 15)) {
            values.emplace_back(val);
        }
        auto record(helper.CreateRecord(TStringBuf("five_seconds_signal_summ"), values));
        helper.Dump(record);

        auto request = helper.CreateCompatRequest(record);
        auto reply = helper.ReplyCompat(request);

        UNIT_ASSERT_VALUES_EQUAL(record.GetValuesCount(), reply.size());

        for (const auto& replyRecord : reply) {
            UNIT_ASSERT(record.GetStartTime() <= replyRecord.Timestamp);
            UNIT_ASSERT(replyRecord.Timestamp <= record.GetEndTime());
            size_t id = (replyRecord.Timestamp - record.GetStartTime()).Seconds() / helper.GetPeriod().GetResolution().Seconds();
            const auto& replyValues = replyRecord.Record.GetValues();
            UNIT_ASSERT(id < values.size());
            UNIT_ASSERT_VALUES_EQUAL(replyRecord.Record.Len(), 1);
            UNIT_ASSERT_VALUES_EQUAL(record.GetSignalName(), replyValues[0].first);
            UNIT_ASSERT_VALUES_EQUAL(values[id], replyValues[0].second);
        }
    }

    Y_UNIT_TEST(HgramReadWrite) {
        THelper helper;
        TVector<TValue> values;

        for (auto i: xrange(0, 3)) {
            TUgramBuckets buckets;
            for (auto val: xrange(0, 10)) {
                buckets.emplace_back(3 * val + i, 3 * val + 1 + i, 1);
            }
            values.emplace_back(THgram::Ugram(std::move(buckets)));
        }

        auto record(helper.CreateRecord(TStringBuf("five_seconds_signal_hgram"), values));
        helper.Dump(record);

        auto request = helper.CreateCompatRequest(record);
        auto reply = helper.ReplyCompat(request);

        UNIT_ASSERT_VALUES_EQUAL(record.GetValuesCount(), reply.size());

        for (const auto& replyRecord : reply) {
            UNIT_ASSERT(record.GetStartTime() <= replyRecord.Timestamp);
            UNIT_ASSERT(replyRecord.Timestamp <= record.GetEndTime());
            size_t id = (replyRecord.Timestamp - record.GetStartTime()).Seconds() / helper.GetPeriod().GetResolution().Seconds();
            const auto& replyValues = replyRecord.Record.GetValues();
            UNIT_ASSERT(id < values.size());
            UNIT_ASSERT_VALUES_EQUAL(replyRecord.Record.Len(), 1);
            UNIT_ASSERT_VALUES_EQUAL(record.GetSignalName(), replyValues[0].first);
            UNIT_ASSERT_VALUES_EQUAL(values[id], replyValues[0].second);
        }
    }

    Y_UNIT_TEST(HgramHourReadWrite) {
        THelper helper;
        TVector<TValue> values;

        for (auto i: xrange(0, 3600 / 5)) {
            Y_UNUSED(i);
            TUgramBuckets buckets;
            for (auto val: xrange(0, 10)) {
                buckets.emplace_back(3 * val, 3 * val + 1, 1);
            }
            values.emplace_back(THgram::Ugram(std::move(buckets)));
        }

        auto record(helper.CreateRecord(TStringBuf("hours_signal_hgram"), values));
        helper.Dump(record);

        auto request = helper.CreateCompatRequest(record, TDuration::Hours(1));
        auto reply = helper.ReplyCompat(request);

        UNIT_ASSERT_VALUES_EQUAL(reply.size(), 1);

        TUgramBuckets hourBuckets;
        for (auto val: xrange(0, 10)) {
            hourBuckets.emplace_back(3 * val, 3 * val + 1, 3600 / 5);
        }
        TValue hourValue(THgram::Ugram(std::move(hourBuckets)));

        const auto& replyRecord = reply[0];
        UNIT_ASSERT(record.GetStartTime() <= replyRecord.Timestamp);
        UNIT_ASSERT(replyRecord.Timestamp <= record.GetEndTime());
        size_t id = (replyRecord.Timestamp - record.GetStartTime()).Seconds() / helper.GetPeriod().GetResolution().Seconds();
        const auto& replyValues = replyRecord.Record.GetValues();
        UNIT_ASSERT(id < values.size());
        UNIT_ASSERT_VALUES_EQUAL(replyRecord.Record.Len(), 1);
        UNIT_ASSERT_VALUES_EQUAL(record.GetSignalName(), replyValues[0].first);
        UNIT_ASSERT_VALUES_EQUAL(hourValue, replyValues[0].second);
    }

    Y_UNIT_TEST(WriteInteravalAndPoint) {
        THelper helper;

        TVector<TValue> values;
        values.emplace_back(THgram::Ugram(TUgramBuckets{TUgramBucket{3.0, 3.0, 10},}));
        values.emplace_back(THgram::Ugram(TUgramBuckets{TUgramBucket{1.0, 2.0, 10},}));

        TVector<TValue> expValues;
        expValues.emplace_back(THgram::Ugram(TUgramBuckets{TUgramBucket{3.0, StockpilePoint(3.0), 10},}));
        expValues.emplace_back(THgram::Ugram(TUgramBuckets{TUgramBucket{1.0, 2.0, 10},}));


        auto record(helper.CreateRecord(TStringBuf("point_and_interval_signal_hgram"), values));
        helper.Dump(record);

        auto request = helper.CreateCompatRequest(record);
        auto reply = helper.ReplyCompat(request);

        UNIT_ASSERT_VALUES_EQUAL(record.GetValuesCount(), reply.size());

        for (const auto& replyRecord : reply) {
            UNIT_ASSERT(record.GetStartTime() <= replyRecord.Timestamp);
            UNIT_ASSERT(replyRecord.Timestamp <= record.GetEndTime());
            size_t id = (replyRecord.Timestamp - record.GetStartTime()).Seconds() / helper.GetPeriod().GetResolution().Seconds();
            const auto& replyValues = replyRecord.Record.GetValues();
            UNIT_ASSERT(id < values.size());
            UNIT_ASSERT_VALUES_EQUAL(replyRecord.Record.Len(), 1);
            UNIT_ASSERT_VALUES_EQUAL(record.GetSignalName(), replyValues[0].first);
            UNIT_ASSERT_VALUES_EQUAL(expValues[id], replyValues[0].second);
        }
    }

    Y_UNIT_TEST(RollupHgram) {
        THelper helper;

        TVector<TValue> values;
        values.emplace_back(THgram::Ugram(TUgramBuckets{TUgramBucket{3.0, 3.0, 10},}));
        values.emplace_back(THgram::Ugram(TUgramBuckets{TUgramBucket{},}));
        values.emplace_back(THgram::Ugram(TUgramBuckets{TUgramBucket{1.0, 2.0, 10},}));

        TVector<TValue> expValues;
        expValues.emplace_back(THgram::Ugram(TUgramBuckets{TUgramBucket{1.0, 2.0, 10}, TUgramBucket{3.0, StockpilePoint(3.0), 10},}));


        auto record(helper.CreateRecord(TStringBuf("rollup_signal_hgram"), values));
        helper.Dump(record);

        auto request = helper.CreateCompatRequest(record, TDuration::Hours(1));
        auto reply = helper.ReplyCompat(request);

        UNIT_ASSERT_VALUES_EQUAL(expValues.size(), reply.size());

        for (const auto& replyRecord : reply) {
            UNIT_ASSERT(record.GetStartTime() <= replyRecord.Timestamp);
            UNIT_ASSERT(replyRecord.Timestamp <= record.GetEndTime());
            size_t id = (replyRecord.Timestamp - record.GetStartTime()).Seconds() / helper.GetPeriod().GetResolution().Seconds();
            const auto& replyValues = replyRecord.Record.GetValues();
            UNIT_ASSERT(id < values.size());
            UNIT_ASSERT_VALUES_EQUAL(replyRecord.Record.Len(), 1);
            UNIT_ASSERT_VALUES_EQUAL(record.GetSignalName(), replyValues[0].first);
            UNIT_ASSERT_VALUES_EQUAL(expValues[id], replyValues[0].second);
        }
    }

    Y_UNIT_TEST(CompatRollupRequest) {
        static const size_t VALUES_COUNT = 117;
        static const size_t VALUE = 3;

        TRecordPeriod period5s(TRecordPeriod::Get("s5"));
        TRecordPeriod period5m(TRecordPeriod::Get("m5"));
        THelper helper;

        TVector<TValue> values;
        for (auto idx : xrange(VALUES_COUNT)) {
            Y_UNUSED(idx);
            values.emplace_back(VALUE);
        }
        auto record(helper.CreateRecord(TStringBuf("five_minutes_signal_summ"), values));
        auto now(period5m.GetStartTime(TInstant::Now()));
        record.SetStartTime(now - period5s.GetResolution() * values.size());
        record.SetEndTime(now - period5s.GetResolution());
        helper.Dump(record);

        auto request = helper.CreateCompatRequest(record, period5m.GetResolution());
        request.Start -= TDuration::FromValue(request.Start.GetValue() % period5m.GetResolution().GetValue());

        auto reply = helper.ReplyCompat(request);
        UNIT_ASSERT_VALUES_EQUAL((VALUES_COUNT + 59) / 60, reply.size());
        for (const auto& replyRecord : reply) {
            UNIT_ASSERT(request.Start <= replyRecord.Timestamp);
            UNIT_ASSERT(replyRecord.Timestamp <= request.End);
            const auto& replyValues = replyRecord.Record.GetValues();
            UNIT_ASSERT_VALUES_EQUAL(replyRecord.Record.Len(), 1);
            UNIT_ASSERT_VALUES_EQUAL(record.GetSignalName(), replyValues[0].first);
            UNIT_ASSERT_VALUES_EQUAL(values[0].GetType(), replyValues[0].second.GetType());
            size_t count = replyRecord.Timestamp == request.Start ? VALUES_COUNT % 60 : 60;
            UNIT_ASSERT_VALUES_EQUAL(TValue(count * VALUE), replyValues[0].second);
        }
    }

    Y_UNIT_TEST(GroupRequest) {
        THelper helper;

        TVector<TValue> values;
        for (auto val: xrange(0, 30)) {
            values.emplace_back(val);
        }
        auto record(helper.CreateRecord(TStringBuf("group_signal_summ"), values));
        record.SetInstanceKey(record.GetInstanceKey().AggregateBy(TVector<TStringBuf>{TStringBuf("host")}));
        record.SetHostName(record.GetInstanceKey().GetHostName().GetName());
        helper.Dump(record);

        auto request(helper.CreateDummyRequest(record));
        auto response(helper.ReplyOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EHistoryStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), values);
    }

    Y_UNIT_TEST(DownsampleRequest) {
        TRecordPeriod period5s(TRecordPeriod::Get("s5"));
        TRecordPeriod period5m(TRecordPeriod::Get("m5"));
        THelper helper;

        const auto valuesInFiveMinutes = period5m.GetResolution().GetValue() / period5s.GetResolution().GetValue();
        TVector<TValue> values(Reserve(valuesInFiveMinutes));
        for (auto idx: xrange(0UL, valuesInFiveMinutes)) {
            values.emplace_back(TValue(1));
            Y_UNUSED(idx);
        }
        auto record(helper.CreateRecord(TStringBuf("downsample_signal_summ"), values));
        auto now(period5m.GetStartTime(TInstant::Now()));
        record.SetStartTime(now);
        record.SetEndTime(now);
        helper.Dump(record);

        auto request(helper.CreateDummyRequest(record));
        request.Period = period5m.GetResolution();
        auto response(helper.ReplyOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EHistoryStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
        TVector<TValue> expectedValues;
        expectedValues.emplace_back(TValue(valuesInFiveMinutes));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
    }

    Y_UNIT_TEST(TestWriteDeleteRead) {
        THelper helper;

        size_t valuesCount = 10;
        size_t deleteTo = 5;
        TVector<TValue> values(Reserve(valuesCount));
        for (auto idx: xrange(valuesCount)) {
            values.emplace_back(TValue(idx));
        }

        TRecordPeriod period(TRecordPeriod::Get("s5"));
        TSignalName signalName(TStringBuf("delete_signal_summ"));
        auto record(helper.CreateRecord(signalName, values, period));
        helper.Dump(record);

        auto end = record.GetStartTime()  + period.GetResolution() * deleteTo;
        helper.ClearHistory(record.GetInstanceKey().GetHostName(), end);


        auto request(helper.CreateDummyRequest(record));
        auto response(helper.ReplyOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EHistoryStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
        for (auto idx: xrange(deleteTo)) {
            values[idx] = TValue(0);
        }

        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), values);
    }

    Y_UNIT_TEST(TestWriteDeleteManyRead) {
        THelper helper;

        size_t valuesCount = 10;
        size_t deleteTo = 5;
        TVector<TValue> values(Reserve(valuesCount));
        for (auto idx: xrange(valuesCount)) {
            values.emplace_back(TValue(idx));
        }

        TRecordPeriod period(TRecordPeriod::Get("s5"));
        TVector<TMockedRecordDescriptor> records;

        for (size_t id : xrange(20)) {
            TString signalName = TStringBuilder() << "delete_signal" << id << "_summ";
            auto& record = records.emplace_back(helper.CreateRecord(TSignalName(signalName), values, period));
            helper.Dump(record);
        }

        auto end = records[0].GetStartTime()  + period.GetResolution() * deleteTo;
        helper.ClearHistory(records[0].GetInstanceKey().GetHostName(), end);


        for (const auto& record: records) {
            auto request(helper.CreateDummyRequest(record));
            auto response(helper.ReplyOne(request));
            UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EHistoryStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
            for (auto idx: xrange(deleteTo)) {
                values[idx] = TValue(0);
            }

            UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), values);
        }
    }

    Y_UNIT_TEST(TestWriteDeleteReadHgram) {
        THelper helper;

        size_t valuesCount = 10;
        size_t deleteTo = 5;


        TVector<TValue> values(Reserve(valuesCount));
        for (auto idx: xrange(valuesCount)) {
            TUgramBuckets buckets;
            buckets.emplace_back(idx, idx + 0.5, 1);
            values.emplace_back(THgram::Ugram(std::move(buckets)));
        }

        TRecordPeriod period(TRecordPeriod::Get("s5"));
        TSignalName signalName(TStringBuf("delete_signal_hgram"));
        auto record(helper.CreateRecord(signalName, values, period));
        helper.Dump(record);

        auto end = record.GetStartTime()  + period.GetResolution() * deleteTo;
        helper.ClearHistory(record.GetInstanceKey().GetHostName(), end);


        auto request(helper.CreateDummyRequest(record));
        auto response(helper.ReplyOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EHistoryStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
        for (auto idx: xrange(deleteTo)) {
            values[idx] = TValue(THgram::Small({}, 0));
        }

        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), values);
    }

    Y_UNIT_TEST(InvalidPeriod) {
        THelper helper;
        THistoryRequest request{
            .Start = TInstant::Seconds(500),
            .End = TInstant::Seconds(1000),
            .Period = TDuration::Seconds(1),
            .HostName = TStringBuf("some_host"),
            .RequestKey = TStringBuf("itype=newsd"),
            .SignalName = TStringBuf("signal_summ")
        };
        auto response(helper.ReplyOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EHistoryStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR);
    }

    Y_UNIT_TEST(TestInvalidSignal) {
        THelper helper;
        THistoryRequest request{
            .Start = TInstant::Seconds(500),
            .End = TInstant::Seconds(1000),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("some_host"),
            .RequestKey = TStringBuf("itype=newsd;ctype=prod"),
            .SignalName = TStringBuf("wrong")
        };
        auto response(helper.ReplyOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EHistoryStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND);
    }

    Y_UNIT_TEST(TestInvalidRange) {
        THelper helper;
        THistoryRequest request{
            .Start = TInstant::Seconds(2000),
            .End = TInstant::Seconds(1000),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("some_host"),
            .RequestKey = TStringBuf("itype=newsd"),
            .SignalName = TStringBuf("signal_summ")
        };
        auto response(helper.ReplyOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EHistoryStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR);
    }

    Y_UNIT_TEST(MultiKeysRequest) {
        THelper helper;

        const long metabaseRequestPageSize = 4;
        const int keys = 10;
        const TStringBuf groupName = "TEST_GROUP";
        const TStringBuf signalName = "multi_keys_test_signal_summ";
        const auto period = helper.GetPeriod();

        TVector<TValue> values;
        TVector<TValue> expectedValues;
        for (auto val: xrange(0, 30)) {
            values.emplace_back(val);
            expectedValues.emplace_back(val * keys);
        }

        const auto now = period.GetStartTime(TInstant::Now());

        for (auto val: xrange(0, keys)) {
            auto record = helper.CreateRecord(signalName, values, period, now);
            TStringStream s;
            s << "val_" << val;
            auto instanceKey = NTags::TInstanceKey::FromParts(
                UNIT_TEST_ITYPE,
                TVector<std::pair<TStringBuf, TStringBuf>>{
                    std::make_pair(TStringBuf("tag"), TStringBuf(s.Str())),
                    std::make_pair(TStringBuf("group"), groupName)
                },
                TVector<TStringBuf>{ TStringBuf("host") }
            );
            record.SetInstanceKey(instanceKey);
            record.SetHostName(groupName);
            helper.Dump(record);
        }

        auto request = helper.CreateDummyRequest(
            UNIT_TEST_ITYPE,
            THostName(groupName),
            TSignalName(signalName),
            expectedValues.size(),
            period,
            now
        );
        auto response = helper.ReplyOneWithLimit(request,
            DEFAULT_METABASE_TIMEOUT,
            DEFAULT_STOCKPILE_TIMEOUT,
            metabaseRequestPageSize);
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EHistoryStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
    }

    Y_UNIT_TEST(MultiGroupMultiSignalRequest) {
        THelper helper;

        const auto notFoundGroup = THostName(TStringBuf("NOT_FOUND_GROUP"));
        const auto groupsAndSignals = TVector<std::pair<TString, TString>>{
            {"TEST_GROUP1", "multi_group_multi_signal_1_summ"},
            {"TEST_GROUP2", "multi_group_multi_signal_2_summ"}
        };
        const auto& period = helper.GetPeriod();
        const auto requestKey = TInternedRequestKey(TString::Join("itype=", UNIT_TEST_ITYPE));

        TVector<TValue> values;
        for (auto val: xrange(0, 30)) {
            values.emplace_back(val);
        }

        const auto now = period.GetStartTime(TInstant::Now());

        for (auto& [group, signal]: groupsAndSignals) {
            auto record = helper.CreateRecord(signal, values, period, now);
            auto instanceKey = NTags::TInstanceKey::FromParts(
                UNIT_TEST_ITYPE,
                TVector<std::pair<TStringBuf, TStringBuf>>{std::make_pair(TStringBuf("group"), group)},
                TVector<TStringBuf>{TStringBuf("host")}
            );
            record.SetInstanceKey(instanceKey);
            record.SetHostName(group);
            helper.Dump(record);
        }

        TVector<THistoryRequest> requests;
        for (auto& [group, signal]: groupsAndSignals) {
            requests.push_back(THistoryRequest{
                .Start = now - period.GetResolution() * values.size(),
                .End = now - period.GetResolution(),
                .Period = period.GetResolution(),
                .HostName = group,
                .RequestKey = requestKey,
                .SignalName = signal
            });
        }
        requests.push_back(THistoryRequest{
            .Start = now - period.GetResolution() * values.size(),
            .End = now - period.GetResolution(),
            .Period = period.GetResolution(),
            .HostName = notFoundGroup,
            .RequestKey = requestKey,
            .SignalName = TStringBuf("not_found_signal_summ")
        });

        auto reader = helper.CreateReader(
            requests,
            TDuration::Seconds(DEFAULT_METABASE_TIMEOUT),
            TDuration::Seconds(DEFAULT_STOCKPILE_TIMEOUT)
        );
        auto responses = reader.Read();
        UNIT_ASSERT_VALUES_EQUAL(requests.size(), responses.size());
        for (auto& response: responses) {
            if (response.Request.HostName != notFoundGroup) {
                UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EHistoryStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
                UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), values);
            } else {
                UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EHistoryStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND);
                UNIT_ASSERT(response.Series.GetValues().empty());
            }
        }
    }

    Y_UNIT_TEST(ConvertUgram) {
        THelper helper;

        TVector<TValue> ugramValues;
        {
            TUgramBuckets buckets = {{1, 2, 5}};
            ugramValues.emplace_back(THgram::Ugram(std::move(buckets)));
        }
        auto record(helper.CreateRecord(TStringBuf("convert_ugram_hgram"), ugramValues));
        {
            helper.Dump(record);
            auto reply = helper.ReplyOne(helper.CreateDummyRequest(record));
            UNIT_ASSERT_VALUES_EQUAL(reply.Series.GetValues(), ugramValues);
        }

        TVector<TValue> normalValues;
        normalValues.emplace_back(THgram::Normal({1, 0, 3}, 0, 1));
        {
            record.SetValues(normalValues);
            record.SetStartTime(record.GetStartTime() - TDuration::Minutes(5));
            record.SetEndTime(record.GetEndTime() - TDuration::Minutes(5));
            helper.Dump(record);

            TVector<TValue> expectedValues;
            TUgramBuckets buckets;
            buckets.emplace_back(pow(1.5, 1), pow(1.5, 2), 1);
            buckets.emplace_back(pow(1.5, 3), pow(1.5, 4), 3);
            expectedValues.emplace_back(THgram::Ugram(std::move(buckets)));

            auto reply = helper.ReplyOne(helper.CreateDummyRequest(record));
            UNIT_ASSERT_VALUES_EQUAL(reply.Series.GetValues(), expectedValues);
        }

    }

    Y_UNIT_TEST(TestMetaAggregate) {
        THelper helper;
        size_t valuesCount = 10;
        TVector<TValue> values1;
        TVector<TValue> values2;
        TVector<TValue> expectedValues;
        for (size_t t : xrange(valuesCount)) {
            Y_UNUSED(t);
            values1.emplace_back(10, 1);
            values2.emplace_back(30, 1);
            expectedValues.emplace_back(40);
        }

        auto record = helper.CreateRecord(TStringBuf("test_meta_tmmv"), values1);
        {
            record.SetInstanceKey(NTags::TInstanceKey::FromNamed("yasm_unittest|geo=sas").AddGroupAndHost(GROUP, HOST));
            helper.Dump(record);
        }
        {
            record.SetValues(values2);
            record.SetInstanceKey(NTags::TInstanceKey::FromNamed("yasm_unittest|geo=man").AddGroupAndHost(GROUP, HOST));
            helper.Dump(record);
        }

        auto request = helper.CreateDummyRequest(record);
        auto response = helper.ReplyOne(request);
        const auto& values = response.Series.GetValues();
        UNIT_ASSERT_VALUES_EQUAL(values, expectedValues);
    }
}
