#include "stockpile_writer.h"
#include "metrics.h"

#include <infra/yasm/stockpile_client/points.h>
#include <infra/yasm/stockpile_client/metrics.h>
#include <infra/monitoring/common/perf.h>

#include <util/generic/list.h>
#include <util/random/random.h>

using namespace NHistDb;
using namespace NStockpile;
using namespace NMonitoring;
using NZoom::NAccumulators::EAccumulatorType;
using NZoom::NValue::EValueType;
using yandex::solomon::model::MetricType;

namespace {
    static constexpr size_t DUMP_EACH_NTH_RECORD = 15000;
    static constexpr TDuration COOLDOWN_DELAY = TDuration::MilliSeconds(50);
    static constexpr TDuration EXCEPTION_COOLDOWN_DELAY = TDuration::Minutes(2);
    static constexpr TDuration RETRY_COOLDOWN_DELAY = TDuration::Seconds(30);
    static constexpr TDuration LOG_INTERVAL_WHEN_WRITING_IS_DISABLED = TDuration::Seconds(10);

    static const TSensorTypeMap ALLOWED_SENSOR_TYPE_CHANGES({
        {yandex::solomon::model::MetricType::LOG_HISTOGRAM, yandex::solomon::model::MetricType::HIST}
    });

    TDuration ApplyJitter(TDuration duration) {
        auto halfDuration = TDuration::FromValue(duration.GetValue() / 2);
        auto jitter = TDuration::FromValue(RandomNumber(halfDuration.GetValue()));
        return halfDuration + jitter;
    }

    TInstant GetDeadlineWithJitter(TDuration duration) {
        return ApplyJitter(duration).ToDeadLine();
    }

    class TStockpileRecordVisitor final: public IRecordVisitor {
    public:
        TStockpileRecordVisitor(
            const TRecordPeriod& fiveSecondsPeriod,
            NStockpile::TStockpileWriteManyState& writeState,
            TInstant startTime)
            : FiveSecondsPeriod(fiveSecondsPeriod)
            , WriteState(writeState)
            , Timestamp(startTime)
        {
        }

        void OnValue(NZoom::NValue::TValueRef value) override {
            WriteState.AddPoint(Timestamp, value);
            Timestamp += FiveSecondsPeriod.GetResolution();
        }

    private:
        const TRecordPeriod& FiveSecondsPeriod;
        NStockpile::TStockpileWriteManyState& WriteState;
        TInstant Timestamp;
    };

    class TStockpileTypeVisitor final: public IRecordVisitor {
    public:
        TStockpileTypeVisitor(EAccumulatorType aggregationType)
            : Detector(aggregationType)
        {
        }

        void OnValue(NZoom::NValue::TValueRef value) override {
            Detector.OnValue(value);
        }

        MetricType GetType() {
            return Detector.GetType();
        }

    private:
        TValueTypeDetector Detector;
    };
    class TWritingDisabled : public yexception {};
}

void TStockpileDumperStats::OnRecordsError(ui64 count) {
    TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_STOCKPILE_FAILED_RECORDS, count);
}

void TStockpileDumperStats::OnRecordsRejected(ui64 count) {
    TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_STOCKPILE_REJECTED_RECORDS, count);
}

void TStockpileDumperStats::OnRecordsEmpty(ui64 count) {
    TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_STOCKPILE_EMPTY_RECORDS, count);
}

void TStockpileDumperStats::UpdateRemainingCapacity(size_t workerIndex, size_t remainingCapacity) {
    AtomicSet(RemainingWorkerCapacities[workerIndex], remainingCapacity);
    size_t minSize = remainingCapacity;
    for (size_t i = 0; i < WorkerCount; ++i) {
        minSize = Min(minSize, static_cast<size_t>(AtomicGet(RemainingWorkerCapacities[i])));
    }
    TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_STOCKPILE_REMAINING_WORKER_CAPACITY, minSize);

}

void TStockpileTopologyListener::Dispatch() {
    NStockpile::TMetabaseShardKey metabaseShard;
    while (MetabaseShards.Dequeue(&metabaseShard)) {
        Processor->EnqueueMetabaseShard(metabaseShard);
    }
}

void TMetabaseQueue::Schedule(TLinkedList& queues) {
    ResetScheduling();
    queues.PushBack(this);
}

void TMetabaseQueue::Schedule(TTree& queues, TInstant deadline) {
    ResetScheduling();
    Deadline = deadline;
    queues.Insert(this);
}

void TMetabaseQueue::ResetScheduling() {
    if (!TIntrusiveListItem<TMetabaseQueue>::Empty() || Deadline) {
        TIntrusiveListItem<TMetabaseQueue>::Unlink();
        TAvlTreeItem<TMetabaseQueue, TCompareUsingDeadline>::Unlink();
        Deadline = TInstant::Zero();
    }
}

TStockpileShardId TStockpileQueue::GetShardId() const {
    return ShardId;
}

bool TStockpileQueue::IsBlocked(TInstant now) const {
    return BlockedFor.Defined() && BlockedSince.GetRef() + BlockedFor.GetRef() > now;
}

bool TStockpileQueue::IsReadyBySize(bool ignoreLimits) const {
    return InFlightCall.Empty() && (Records.size() >= GROW_STOCKPILE_QUEUE_UNTIL_SIZE || ignoreLimits);
}

bool TStockpileQueue::IsReadyByTime(TInstant now) const {
    return InFlightCall.Empty() && GrowUntilAge.Defined() && GrowUntilAge.GetRef() <= now;
}

void TStockpileQueue::EnqueueRecord(NStockpile::TSensorId sensorId, TNumId metabaseShardNumId, IRecordDescriptorPtr record,
                                    TInstant now)
{
    QueueSize->Inc();
    Records.push_back(TRecordInfo{
        .SensorId = sensorId,
        .MetabaseShardNumId = metabaseShardNumId,
        .Record = std::move(record)
    });
    UpdateNonEmptySince(now);
}

const TVector<TStockpileQueue::TRecordInfo>& TStockpileQueue::PrepareRecordsForCall(size_t maxRecordsToSend, TInstant now) {
    Y_VERIFY(InFlightRecords.empty());

    InFlightRecords.reserve(Min(Records.size(), maxRecordsToSend));
    i64 added = 0;
    while (InFlightRecords.size() < maxRecordsToSend && !Records.empty()) {
        InFlightRecords.emplace_back(std::move(Records.front()));
        Records.pop_front();
        added++;
    }
    QueueSize->Add(added * -1);
    GrowUntilAge.Clear();
    if (!Records.empty()) {
        UpdateNonEmptySince(now);
    }
    Sort(InFlightRecords.begin(), InFlightRecords.end(), [](const auto& left, const auto& right) {
        if (left.MetabaseShardNumId < right.MetabaseShardNumId) {
            return true;
        } else if (left.MetabaseShardNumId > right.MetabaseShardNumId) {
            return false;
        }

        if (left.SensorId.LocalId < right.SensorId.LocalId) {
            return true;
        } else if (left.SensorId.LocalId == right.SensorId.LocalId) {
            return left.Record->GetStartTime() < right.Record->GetStartTime();
        } else {
            return false;
        }
    });

    return InFlightRecords;
}

std::pair<size_t, bool> TStockpileQueue::ClearInFlightCall(bool unblock, TMap<TInstant, size_t>& startTimesToUpdate) {
    for (const auto& record: InFlightRecords) {
        --startTimesToUpdate[record.Record->GetStartTime()];
    }

    size_t requeuedCount = InFlightRecords.size();
    InFlightRecords.clear();
    InFlightCall.Clear();
    bool unblocked = false;
    if (unblock && BlockedFor.Defined()) {
        BlockedFor.Clear();
        BlockedSince.Clear();
        unblocked = true;
    }
    return std::make_pair(requeuedCount, unblocked);
}

std::pair<size_t, TDuration> TStockpileQueue::RequeueRecordsAndBlock(TInstant now, bool applyJitterToBlockInterval) {
    size_t requeued = InFlightRecords.size();
    QueueSize->Add(static_cast<i64>(requeued));
    std::move(std::begin(InFlightRecords), std::end(InFlightRecords), std::back_inserter(Records));
    if (requeued) {
        UpdateNonEmptySince(now);
    }

    InFlightRecords.clear();
    InFlightCall.Clear();

    if (!BlockedFor.Defined()) {
        BlockedFor = STOCKPILE_QUEUE_INITIAL_BLOCK_INTERVAL;
        BlockedSince = now;
    } else {
        BlockedFor = Min(BlockedFor.GetRef() * 2, STOCKPILE_QUEUE_MAX_BLOCK_INTERVAL);
        if (applyJitterToBlockInterval) {
            BlockedFor = ApplyJitter(BlockedFor.GetRef());
        }
        BlockedSince = now;
    }

    return std::make_pair(requeued, BlockedFor.GetRef());
}

bool TStockpileQueue::IsInFlight() const {
    return InFlightCall.Defined();
}

TMaybe<NStockpile::TStockpileWriteManyState>& TStockpileQueue::GetInFlightCall() {
    return InFlightCall;
}

void TStockpileQueue::UpdateNonEmptySince(TInstant timestamp) {
    if (!GrowUntilAge.Defined()) {
        GrowUntilAge = timestamp + ApplyJitter(GROW_STOCKPILE_QUEUE_UNTIL_AGE);
    }
}

TStockpileQueueProcessor::TStockpileQueueProcessor(TLog& logger, NStockpile::TStockpileState& state, const TStockpileWriterSettings& settings,
    size_t index, TStockpileDumperStats& stats)
    : Index(index)
    , Logger(logger)
    , State(state)
    , Settings(settings)
    , FiveSecondsPeriod(TRecordPeriod::Get("s5"))
    , RecordCount{}
    , RecordNumber{}
    , StockpileStats(stats)
    , LastStartTime{}
    , GrpcHandler(Logger)
    , LastDisabledLogEntryTime()
{
    TopologyListener = State.GetObserver().Register(MakeHolder<TStockpileTopologyListener>(this));
}

TStockpileQueueProcessor::~TStockpileQueueProcessor() {
    State.GetObserver().Erase(TopologyListener);
}

TMetabaseShardKey TStockpileQueueProcessor::ToShardKey(const IRecordDescriptor& desc) const {
    return TMetabaseShardKey::Make(
        desc.GetInstanceKey(),
        desc.GetSignalName(),
        State.GetShardsCount(desc.GetInstanceKey().GetItype())
    );
}

bool TStockpileQueueProcessor::IsActive() const {
    return !State.WritingIsDisabled() && RecordCount < Settings.MaxRecordsInMemory;
}

void TStockpileQueueProcessor::CheckWriting() const {
    if (State.WritingIsDisabled()) {
        throw TWritingDisabled();
    }
}

void TStockpileQueueProcessor::DumpIfReady(bool onIdle) {
    try {
        CheckWriting();
        if (RecordNumber > DUMP_EACH_NTH_RECORD) {
            DumpRecords(true);
            RecordNumber = 0;
        } else if (RecordCount >= Settings.MaxRecordsInMemory || onIdle) {
            DumpRecords();
        }
        UpdateStartTime();
    } catch (const TWritingDisabled& ) {
        auto now = TInstant::Now();
        if (LastDisabledLogEntryTime + LOG_INTERVAL_WHEN_WRITING_IS_DISABLED >= now) {
            Logger << TLOG_INFO << "StockpileQueueProcessor #" << Index << " is blocked while writing is disabled";
            LastDisabledLogEntryTime = now;
        }
    }
}

void TStockpileQueueProcessor::Update() {
    DumpIfReady(true);
}

void TStockpileQueueProcessor::AddRecord(const IRecordDescriptor& record, TInstant) {
    IRecordDescriptorPtr recordToAdd;
    if (!record.GetSignalName().IsOld()) {
        if (record.GetHostName().IsGroup()) {
            recordToAdd = record.Clone();
        } else {
            recordToAdd = record.CloneWithNewKey(record.GetInstanceKey().RemoveGroup());
        }
    }

    if (recordToAdd) {
        if (!TryAddRecordToStockpile(recordToAdd)) {
            AddRecordToMetabase(std::move(recordToAdd));
        }
        RecordNumber++;
        DumpIfReady(false);
    }
}

void TStockpileQueueProcessor::DumpRecords(bool noDelay, bool forceNotReadyQueues) {
    if (!noDelay) {
        GrpcHandler.WaitAsync(ApplyJitter(COOLDOWN_DELAY));
    } else {
        GrpcHandler.WaitAsync();
    }
    TopologyListener->Dispatch();
    ResolveRecords(TInstant::Now());
    WriteRecords(forceNotReadyQueues);
}

void TStockpileQueueProcessor::ResolveRecords(TInstant now) {
    while (!ChangedMetabaseQueues.Empty()) {
        auto& queue = *ChangedMetabaseQueues.PopFront();
        SafeResolveOneQueue(queue);
    }
    while (!DelayedMetabaseQueues.Empty() && DelayedMetabaseQueues.First()->Deadline <= now) {
        auto& queue = *DelayedMetabaseQueues.First();
        SafeResolveOneQueue(queue);
    }
}

void TStockpileQueueProcessor::WriteRecords(bool forceNotReadyQueues) {
    for (auto& [shardId, queue]: RecordsToWrite) {
        SafeWriteOneQueue(queue, forceNotReadyQueues);
    }
}

void TStockpileQueueProcessor::Finish() {
    try {
        while (RecordCount) {
            DumpRecords(false, true);
        }
    } catch (const TWritingDisabled &) {
    }
}

bool TStockpileQueueProcessor::TryAddRecordToStockpile(IRecordDescriptorPtr& record) {
    auto shardState = State.FindOneMetabaseShardForWrite(ToShardKey(*record));
    if (shardState.Defined()) {
        auto typedSeriesKey = MakeOneTypedSeriesKey(*record);
        auto sensorId = shardState->Shard->FindSensorWithTypeCheck(typedSeriesKey, ALLOWED_SENSOR_TYPE_CHANGES);
        if (sensorId.Defined()) {
            RecordCount++;
            ++StartTimes[record->GetStartTime()];

            AddRecordToStockpile(sensorId.GetRef(), shardState->Shard->GetShardNumId(), std::move(record));

            UpdateCapacityStats();
            return true;
        } else if (shardState->Shard->ContainsInRejection(typedSeriesKey.SeriesKey, TInstant::Now())) {
            StockpileStats.OnRecordsRejected(1);
            return true;
        } else if (typedSeriesKey.Type == MetricType::METRIC_TYPE_UNSPECIFIED) {
            // skip empty record
            StockpileStats.OnRecordsEmpty(1);
            return true;
        }
    }
    return false;
}

void TStockpileQueueProcessor::AddRecordToMetabase(IRecordDescriptorPtr record) {
    auto& queue = GetMetabaseQueue(ToShardKey(*record));
    RecordCount++;
    ++StartTimes[record->GetStartTime()];
    queue.QueueSize->Inc();
    queue.IncomingRecords.emplace_back(std::move(record));
    queue.Schedule(ChangedMetabaseQueues);
    UpdateCapacityStats();
}

void TStockpileQueueProcessor::AddRecordToStockpile(const TSensorId& sensorId, TNumId metabaseShardNumId,
                                                    IRecordDescriptorPtr record)
{
    GetStockpileQueue(sensorId.ShardId).EnqueueRecord(sensorId, metabaseShardNumId, std::move(record), TInstant::Now());
}

void TStockpileQueueProcessor::ClearScheduledRecords(TMetabaseQueue& queue) {
    for (const auto& record: queue.ScheduledRecords) {
        --StartTimes[record->GetStartTime()];
    }
    RecordCount -= queue.ScheduledRecords.size();
    queue.ScheduledRecords.clear();
    UpdateCapacityStats();
}

std::pair<size_t, bool> TStockpileQueueProcessor::ClearInFlightCall(TStockpileQueue& queue, bool unblock) {
    auto clearResult = queue.ClearInFlightCall(unblock, StartTimes);
    RecordCount -= clearResult.first;
    UpdateCapacityStats();
    return clearResult;
}

void TStockpileQueueProcessor::UpdateStartTime() {
    TMaybe<TInstant> lastTimeDumped;
    while (!StartTimes.empty() && StartTimes.begin()->second == 0) {
        // NOTE(rocco66): chunk was successfully dumped
        lastTimeDumped = StartTimes.begin()->first;
        StartTimes.erase(StartTimes.begin());
    }
    if (!StartTimes.empty()) {
        // NOTE(rocco66): chunk is dumping
        lastTimeDumped = StartTimes.begin()->first;
    }
    if (lastTimeDumped.Defined()) {
        ui64 prevValue = AtomicGet(LastStartTime);
        ui64 newValue = lastTimeDumped->Seconds();
        Y_ASSERT(prevValue <= newValue);
        AtomicSet(LastStartTime, newValue);
    }
}

TMaybe<TInstant> TStockpileQueueProcessor::GetLastTime() {
    auto startTime = AtomicGet(LastStartTime);
    if (startTime) {
        return TInstant::Seconds(startTime);
    }
    return Nothing();
}

void TStockpileQueueProcessor::EnqueueMetabaseShard(const NStockpile::TMetabaseShardKey& shardKey) {
    auto it = RecordsToResolve.find(shardKey);
    if (!it.IsEnd()) {
        it->second.Schedule(ChangedMetabaseQueues);
    }
}

TMetabaseQueue& TStockpileQueueProcessor::GetMetabaseQueue(NStockpile::TMetabaseShardKey shardKey) {
    TRecordsPerMetabaseShard::insert_ctx context;
    const auto it = RecordsToResolve.find(shardKey, context);
    if (it != RecordsToResolve.end()) {
        return it->second;
    } else {
        return RecordsToResolve.emplace_direct(context, shardKey, shardKey)->second;
    }
}

TStockpileQueue& TStockpileQueueProcessor::GetStockpileQueue(NStockpile::TStockpileShardId shardId) {
    TRecordsPerStockpileShard::insert_ctx context;
    const auto it = RecordsToWrite.find(shardId, context);
    if (it != RecordsToWrite.end()) {
        return it->second;
    } else {
        return RecordsToWrite.emplace_direct(context, shardId, shardId)->second;
    }
}

NStockpile::TSeriesKey TStockpileQueueProcessor::MakeOneSeriesKey(const IRecordDescriptor& record) {
    return TSeriesKey::Make(record.GetInstanceKey(), record.GetSignalName());
}

NStockpile::TTypedSeriesKey TStockpileQueueProcessor::MakeOneTypedSeriesKey(const IRecordDescriptor& record) {
    TStockpileTypeVisitor visitor(GetAggregationType(record.GetSignalName()));
    MetricType kind;
    try {
        record.Iterate(visitor);
        kind = visitor.GetType();
    } catch (const TTypeNotSupportedException&) {
        kind = MetricType::METRIC_TYPE_UNSPECIFIED;
    }
    return TTypedSeriesKey{MakeOneSeriesKey(record), kind};
}

TVector<TTypedSeriesKey> TStockpileQueueProcessor::MakeTypedSeriesKeys(const TVector<IRecordDescriptorPtr>& records) {
    TVector<TTypedSeriesKey> seriesKeys(Reserve(records.size()));
    for (const auto& record : records) {
        seriesKeys.emplace_back(MakeOneTypedSeriesKey(*record));
    }
    return seriesKeys;
}

const NStockpile::TMetabaseShardState TStockpileQueueProcessor::MoveResolvedRecordsToStockpile(TMetabaseQueue& queue) {
    const auto shardState = State.ResolveOneMetabaseShardForWrite(queue.ShardKey);
    if (queue.IncomingRecords.empty()) {
        return shardState;
    }
    auto seriesKeys = MakeTypedSeriesKeys(queue.IncomingRecords);
    auto sensors = shardState.Shard->FindSensorsWithTypeCheck(seriesKeys, ALLOWED_SENSOR_TYPE_CHANGES);

    Y_VERIFY(seriesKeys.size() == queue.IncomingRecords.size() && seriesKeys.size() == sensors.size());

    for (const auto idx : xrange(seriesKeys.size())) {
        auto& sensor = sensors[idx];
        if (sensor.Defined()) {
            queue.QueueSize->Dec();
            AddRecordToStockpile(sensor.GetRef(), shardState.Shard->GetShardNumId(), std::move(queue.IncomingRecords[idx]));
        } else {
            queue.ScheduledRecords.emplace_back(std::move(queue.IncomingRecords[idx]));
        }
    }

    queue.IncomingRecords.clear();
    return shardState;
}

bool TStockpileQueueProcessor::DispatchInFlightMetabaseCall(TMetabaseQueue& queue, const TMetabaseShardState& shardState) noexcept {
    if (!queue.InFlightCall.Defined()) {
        return true;
    }

    if (!queue.InFlightCall->IsFinished()) {
        // InFlightCall is still in progress so we are not ready to send the next one
        return false;
    }

    if (queue.InFlightCall->IsRetriable()) {
        Logger << TLOG_WARNING << "Requeue records for metabase shard " << queue.ShardKey;
        queue.QueueSize->Add(static_cast<i64>(queue.InFlightRecords.size()));
        std::move(std::begin(queue.InFlightRecords), std::end(queue.InFlightRecords), std::back_inserter(queue.ScheduledRecords));
        queue.InFlightRecords.clear();
        const bool topologyChanged = queue.InFlightCall->TopologyChanged();
        queue.InFlightCall.Clear();
        queue.Schedule(DelayedMetabaseQueues, GetDeadlineWithJitter(topologyChanged ? EXCEPTION_COOLDOWN_DELAY : RETRY_COOLDOWN_DELAY));
        return false;
    }

    // request may have failed (maybe only partially)
    auto sensors = queue.InFlightCall->GetRawResult();
    Y_VERIFY(sensors.size() == queue.InFlightRecords.size());
    size_t recordsToDrop = 0;
    for (const auto idx : xrange(queue.InFlightRecords.size())) {
        auto& sensorMaybe = sensors[idx];
        auto& record = queue.InFlightRecords[idx];
        if (sensorMaybe.Defined()) {
            AddRecordToStockpile(sensorMaybe.GetRef(), shardState.Shard->GetShardNumId(), std::move(record));
        } else {
            ++recordsToDrop;
            --StartTimes[record->GetStartTime()];
        }
    }
    if (recordsToDrop) {
        if (queue.InFlightCall->IsReadOnly()) {
            // If shard is read only -- don't pollute logs, only increment a counter.
            StockpileStats.OnRecordsRejected(recordsToDrop);
        } else if (queue.InFlightCall->IsQuotaExceeded()) {
            Logger << TLOG_WARNING << "Failed to create " << recordsToDrop
                   <<" records in metabase shard " << queue.ShardKey << ". Shard quota exceeded.";
            StockpileStats.OnRecordsRejected(recordsToDrop);
        } else {
            Logger << TLOG_WARNING << "Can't resolve " << recordsToDrop << " out of "
                   << sensors.size() << " records using metabase shard " << queue.ShardKey;
            StockpileStats.OnRecordsError(recordsToDrop);
        }
        RecordCount -= recordsToDrop;
        UpdateCapacityStats();
    }
    TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_STOCKPILE_TYPE_MISMATCH_RECORDS,
        queue.InFlightCall->GetTypeMismatchCount());
    TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_STOCKPILE_SENSOR_TYPE_CHANGE,
        queue.InFlightCall->GetSensorTypeChanges());

    queue.InFlightRecords.clear();
    queue.InFlightCall.Clear();

    return true;
}

TVector<NStockpile::TTypedSeriesKey> TStockpileQueueProcessor::PrepareScheduledRecords(TMetabaseQueue& queue) noexcept {
    size_t sizeToReserve(Min(queue.ScheduledRecords.size(), Settings.MaxRecordsInMetabaseRequest));
    TVector<TTypedSeriesKey> seriesKeys(Reserve(sizeToReserve));
    queue.InFlightRecords.reserve(sizeToReserve);

    size_t skipped = 0;
    while (queue.InFlightRecords.size() <= Settings.MaxRecordsInMetabaseRequest && !queue.ScheduledRecords.empty()) {
        auto& record = queue.ScheduledRecords.front();
        auto typedKey = MakeOneTypedSeriesKey(*record);
        if (typedKey.Type == MetricType::METRIC_TYPE_UNSPECIFIED) {
            // skip empty record
            --StartTimes[record->GetStartTime()];
            ++skipped;
        } else {
            seriesKeys.emplace_back(std::move(typedKey));
            queue.InFlightRecords.emplace_back(std::move(record));
        }
        queue.QueueSize->Dec();
        queue.ScheduledRecords.pop_front();
    }

    RecordCount -= skipped;
    StockpileStats.OnRecordsEmpty(skipped);
    UpdateCapacityStats();

    return seriesKeys;
}

bool TStockpileQueueProcessor::ResolveOneQueue(TMetabaseQueue& queue) {
    CheckWriting();
    const NStockpile::TMetabaseShardState shardState = MoveResolvedRecordsToStockpile(queue);

    const auto& shard = shardState.Shard;

    if (!DispatchInFlightMetabaseCall(queue, shardState)) {
        return true;
    }

    if (queue.ScheduledRecords.empty()) {
        // nothing to do
        return true;
    }

    Y_VERIFY(queue.InFlightRecords.empty());

    if (!shardState.IsReady()) {
        if (Settings.UnavailableMetabaseShardDropRecordsDelay.Defined() &&
            shardState.NotReadySince.GetRef() + Settings.UnavailableMetabaseShardDropRecordsDelay.GetRef() < TInstant::Now()) {
            Logger << TLOG_WARNING << "Metabase shard " << queue.ShardKey << " is not available since "
                   << shardState.NotReadySince.GetRef() << ". Dropping " << queue.ScheduledRecords.size() << " records";
            StockpileStats.OnRecordsError(queue.ScheduledRecords.size());
            ClearScheduledRecords(queue);
            return true;
        } else {
            Logger << TLOG_WARNING << "Metabase shard " << *shard << " not ready for resolving";
            return false;
        }
    }
    auto seriesKeys = PrepareScheduledRecords(queue);
    // If there are no keys, just do nothing
    if (!seriesKeys.empty()) {
        queue.InFlightCall.ConstructInPlace(seriesKeys, shard, shardState.GrpcRemoteHost,
            GrpcHandler.GetQueue(), Logger, shardState.ReadOnly, &ALLOWED_SENSOR_TYPE_CHANGES);
        queue.InFlightCall->SetCallback([this, &queue](TGrpcState&) {
            queue.Schedule(ChangedMetabaseQueues);
        });
    }
    return true;
}

void TStockpileQueueProcessor::SafeResolveOneQueue(TMetabaseQueue& queue) {
    queue.ResetScheduling();
    bool reschedule;
    try {
        reschedule = !ResolveOneQueue(queue);
    } catch (...) {
        reschedule = true;
        Logger << TLOG_ERR << "Metabase shard " << queue.ShardKey << " processing failed with: " << CurrentExceptionMessage();
    }
    if (reschedule) {
        queue.Schedule(DelayedMetabaseQueues, GetDeadlineWithJitter(EXCEPTION_COOLDOWN_DELAY));
    }
}

bool TStockpileQueueProcessor::DispatchInFlightStockpileCall(TStockpileQueue& queue) noexcept {
    if (!queue.GetInFlightCall().Defined()) {
        return true;
    }

    if (!queue.GetInFlightCall()->IsFinished()) {
        // call slot not yet available
        return false;
    }

    if (queue.GetInFlightCall()->IsRetriable()) {
        auto [requeued, blockedFor] = queue.RequeueRecordsAndBlock(TInstant::Now(), true);
        Logger << TLOG_WARNING << "Blocked shard " << queue.GetShardId() << " for " << blockedFor
                               << ". Requeue " << requeued << " records";
        TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_STOCKPILE_DELAYED_RECORDS, requeued);
        TUnistat::Instance().PushSignalUnsafe(NStockpile::NMetrics::STOCKPILE_QUEUES_BLOCKED, 1);
        return false;
    }

    auto failed = queue.GetInFlightCall()->IsFailed();
    auto [count, unblocked] = ClearInFlightCall(queue, true);
    if (failed) {
        Logger << TLOG_WARNING << "Can't write " << count << " records to stockpile shard " << queue.GetShardId();
        StockpileStats.OnRecordsError(count);
    } else {
        TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_STOCKPILE_PROCESSED_RECORDS, count);
    }
    if (unblocked) {
        Logger << TLOG_INFO << "Unblocked shard " << queue.GetShardId();
        TUnistat::Instance().PushSignalUnsafe(NStockpile::NMetrics::STOCKPILE_QUEUES_UNBLOCKED, 1);
    }
    return true;
}

void TStockpileQueueProcessor::WriteOneQueue(TStockpileQueue& queue, bool forceNotReadyQueues) {
    CheckWriting();

    auto now = TInstant::Now();
    if (!DispatchInFlightStockpileCall(queue) || queue.IsBlocked(now)) {
        return;
    }
    const auto readyBySize = queue.IsReadyBySize(forceNotReadyQueues);
    const auto readyByTime = queue.IsReadyByTime(now);
    if (!readyByTime && !readyBySize) {
        return;
    }

    auto& writeManyState = queue.GetInFlightCall().ConstructInPlace(State.ResolveOneStockpileShardForWrite(queue.GetShardId()),
        Logger, StockpileStats);
    const auto& inFlightRecords = queue.PrepareRecordsForCall(Settings.MaxRecordsInStockpileRequest, now);

    for (auto it = inFlightRecords.begin(); it != inFlightRecords.end();) {
        // collect records with same local id
        TMaybe<NStockpile::TSensorId> sensorId;
        TMaybe<NStockpile::TNumId> ownerNumId;
        TMaybe<NZoom::NAccumulators::EAccumulatorType> aggregationType;
        TVector<const IRecordDescriptor*> records;
        while (sensorId.Empty() || (it != inFlightRecords.end() && sensorId->LocalId == it->SensorId.LocalId)) {
            sensorId.ConstructInPlace(it->SensorId);
            ownerNumId = it->MetabaseShardNumId;
            aggregationType.ConstructInPlace(GetAggregationType(it->Record->GetSignalName()));
            records.emplace_back(it->Record.Get());
            ++it;
        }

        // and convert them to stockpile protobuf
        try {
            TRecordSerializeState recordSerializeState(*sensorId, *ownerNumId, *aggregationType);
            if (*aggregationType == NZoom::NAccumulators::EAccumulatorType::Hgram && sensorId->Type == MetricType::HIST) {
                TRecordUgramMergerVisitor visitor;
                for (const auto* record : records) {
                    record->Iterate(visitor);
                }
                recordSerializeState.SetUgramFreezer(visitor.GetFreezer());
            }

            writeManyState.StartRecord(std::move(recordSerializeState));
            for (const auto* record : records) {
                TStockpileRecordVisitor visitor(FiveSecondsPeriod, writeManyState, record->GetStartTime());
                record->Iterate(visitor);
            }
            writeManyState.FinishRecord();
        } catch (const TTypeNotSupportedException& exc) {
            TUnistat::Instance().PushSignalUnsafe(NStockpile::NMetrics::STOCKPILE_SENSORS_SKIPPED, records.size());
            StockpileStats.OnRecordsError(records.size());
        } catch (...) {
            Logger << TLOG_ERR << "Skip " << records.size() << " sensors: " << CurrentExceptionMessage() << ", " << MakeOneSeriesKey(*records.back());
            TUnistat::Instance().PushSignalUnsafe(NStockpile::NMetrics::STOCKPILE_SENSORS_SKIPPED, records.size());
            StockpileStats.OnRecordsError(records.size());
        }
    }

    writeManyState.FinishBuild();
    if (writeManyState.Empty()) {
        ClearInFlightCall(queue, false);
    } else {
        Metrics.PointsInRequest->Record(static_cast<i64>(writeManyState.Size()));
        Metrics.MetricsInRequest->Record(static_cast<i64>(writeManyState.Metrics()));
        Metrics.BytesInRequest->Record(static_cast<i64>(writeManyState.Bytes()));
        writeManyState.ScheduleForExecute(GrpcHandler.GetQueue());
        if (readyBySize) {
            Metrics.WriteBySize->Inc();
            TUnistat::Instance().PushSignalUnsafe(NStockpile::NMetrics::STOCKPILE_QUEUES_SENT_BY_SIZE, 1);
        } else if (readyByTime) {
            Metrics.WriteByAge->Inc();
            TUnistat::Instance().PushSignalUnsafe(NStockpile::NMetrics::STOCKPILE_QUEUES_SENT_BY_AGE, 1);
        }
    }
}

void TStockpileQueueProcessor::SafeWriteOneQueue(TStockpileQueue& queue, bool forceNotReadyQueues) {
    try {
        WriteOneQueue(queue, forceNotReadyQueues);
    } catch (...) {
        Logger << TLOG_ERR << "Stockpile shard " << queue.GetShardId() << " processing failed with: " << CurrentExceptionMessage();
    }
}

void TStockpileQueueProcessor::UpdateCapacityStats() {
    size_t remainingCapacity = (RecordCount < Settings.MaxRecordsInMemory) ? Settings.MaxRecordsInMemory - RecordCount : 0;
    StockpileStats.UpdateRemainingCapacity(Index, remainingCapacity);
}

TStockpileWriter::TStockpileWriter(TLog& log, NStockpile::TStockpileState& state, const TStockpileWriterSettings& settings,
    size_t index, TStockpileDumperStats& stats)
    : Logger(log)
    , StockpileState(state)
    , Settings(settings)
    , FiveSecondsPeriod(TRecordPeriod::Get("s5"))
    , FiveMinutesPeriod(TRecordPeriod::Get("m5"))
    , FlushOffset(0)
    , Processor(Logger, StockpileState, settings, index, stats)
{
}

void TStockpileWriter::OnRecord(const IRecordDescriptor& recordDescriptor) {
    TMeasuredMethod perf(Logger, "", NMetrics::PIPELINE_STOCKPILE_PROCESSING_TIME);

    Processor.AddRecord(recordDescriptor, TInstant::Now());

    const TInstant newFlushOffset(FiveMinutesPeriod.GetPointStartTime(recordDescriptor.GetFlushOffset()));
    AtomicSwap(&FlushOffset, newFlushOffset.GetValue());
}

bool TStockpileWriter::IsActive() const {
    return Processor.IsActive();
}

void TStockpileWriter::Update() {
    Processor.Update();
}

void TStockpileWriter::CollectStats() {
    const auto flushTime(TInstant::FromValue(AtomicGet(FlushOffset)));
    if (flushTime) {
        const TDuration delay(TInstant::Now() - TInstant::FromValue(AtomicGet(FlushOffset)) - FiveMinutesPeriod.GetResolution());
        TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_STOCKPILE_PROCESSING_DELAY, delay.SecondsFloat());
    }
}

void TStockpileWriter::Finish() {
    Processor.Finish();
}

TMaybe<TInstant> TStockpileWriter::GetLastTime() {
    return Processor.GetLastTime();
}
