#pragma once

#include <infra/yasm/histdb/components/dumper/ifaces.h>
#include <infra/yasm/histdb/components/placements/periods.h>
#include <infra/yasm/stockpile_client/points.h>
#include <infra/yasm/stockpile_client/stockpile_shard.h>
#include <infra/yasm/stockpile_client/state.h>

#include <library/cpp/containers/intrusive_avl_tree/avltree.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/thread/lfqueue.h>

//NOTE(rocco66): see https://wiki.yandex-team.ru/solomon/projects/golovan/grpc-work-flow-solomon/ for details

namespace NHistDb {
    static constexpr size_t GROW_STOCKPILE_QUEUE_UNTIL_SIZE = 30;
    static constexpr TDuration GROW_STOCKPILE_QUEUE_UNTIL_AGE = TDuration::Seconds(30);
    static constexpr TDuration STOCKPILE_QUEUE_INITIAL_BLOCK_INTERVAL = TDuration::Seconds(1);
    static constexpr TDuration STOCKPILE_QUEUE_MAX_BLOCK_INTERVAL = TDuration::Minutes(1);

    struct TStockpileWriterSettings {
        size_t MaxRecordsInMetabaseRequest = 1000;
        size_t MaxRecordsInStockpileRequest = GROW_STOCKPILE_QUEUE_UNTIL_SIZE * 2;
        size_t MaxRecordsInMemory = 130000;
        TMaybe<TDuration> UnavailableMetabaseShardDropRecordsDelay = TDuration::Minutes(30);
    };

    class TStockpileDumperStats: public NStockpile::IStockpileClientStats {
    public:
        explicit TStockpileDumperStats(size_t workerCount)
            : WorkerCount(workerCount)
            , RemainingWorkerCapacities(new TAtomic[workerCount]) {
            for (size_t i = 0; i < workerCount; ++i) {
                AtomicSet(RemainingWorkerCapacities[i], 0);
            }
        }

        void OnRecordsError(ui64 count) override;
        void OnRecordsRejected(ui64 count) override;
        void OnRecordsEmpty(ui64 count) override;
        void UpdateRemainingCapacity(size_t workerIndex, size_t remainingCapacity);
    private:
        const size_t WorkerCount;
        TArrayHolder<TAtomic> RemainingWorkerCapacities;
    };

    class TStockpileQueueProcessor;

    class TStockpileTopologyListener : public NStockpile::ITopologyListener {
    public:
        TStockpileTopologyListener(TStockpileQueueProcessor* processor)
            : Processor(processor) {
        }

        void OnChangedMetabaseShard(const NStockpile::TMetabaseShardKey& shardKey) override {
            MetabaseShards.Enqueue(shardKey);
        }

        void OnChangedStockpileShard(const NStockpile::TStockpileShardId&) override {
        };

        void Dispatch();

    private:
        TStockpileQueueProcessor* Processor;

        TLockFreeQueue<NStockpile::TMetabaseShardKey> MetabaseShards;
    };

    using IRecordDescriptorPtr = THolder<IRecordDescriptor>;
    struct TCompareUsingDeadline {
        template <class T>
        static inline bool Compare(const T& lhs, const T& rhs) {
            return (
                lhs.Deadline < rhs.Deadline
                || (
                    lhs.Deadline == rhs.Deadline
                    && &lhs < &rhs
                )
            );
        }
    };

    struct TMetabaseQueue: public TIntrusiveListItem<TMetabaseQueue>, public TAvlTreeItem<TMetabaseQueue, TCompareUsingDeadline> {
        using TLinkedList = TIntrusiveList<TMetabaseQueue>;
        using TTree = TAvlTree<TMetabaseQueue, TCompareUsingDeadline>;

        TMetabaseQueue(NStockpile::TMetabaseShardKey shardKey)
            : ShardKey(shardKey)
        {
            auto* registry = NMonitoring::TMetricRegistry::Instance();
            QueueSize = registry->IntGauge({{"sensor", "metabase.queue.size"}});
        }

        void Schedule(TLinkedList& queues);
        void Schedule(TTree& queues, TInstant deadline);
        void ResetScheduling();

        NStockpile::TMetabaseShardKey ShardKey;
        TVector<IRecordDescriptorPtr> IncomingRecords;
        TDeque<IRecordDescriptorPtr> ScheduledRecords;
        NMonitoring::TIntGauge* QueueSize;

        TMaybe<NStockpile::TMetabaseResolveSensor> InFlightCall;
        TVector<IRecordDescriptorPtr> InFlightRecords;

        TInstant Deadline;
    };

    class TStockpileQueue {
    public:
        struct TRecordInfo {
            NStockpile::TSensorId SensorId;
            NStockpile::TNumId MetabaseShardNumId;
            IRecordDescriptorPtr Record;
        };

        TStockpileQueue(NStockpile::TStockpileShardId ShardId)
            : ShardId(ShardId)
            , Records()
            , GrowUntilAge()
        {
            auto* registry = NMonitoring::TMetricRegistry::Instance();
            QueueSize = registry->IntGauge({{"sensor", "stockpile.queue.size"}});
        }

        NStockpile::TStockpileShardId GetShardId() const;
        bool IsBlocked(TInstant now) const;
        bool IsReadyBySize(bool ignoreLimits) const;
        bool IsReadyByTime(TInstant now) const;
        bool IsInFlight() const;

        void EnqueueRecord(NStockpile::TSensorId sensorId, NStockpile::TNumId MetabaseShardNumId, IRecordDescriptorPtr record,
                           TInstant now);
        const TVector<TRecordInfo>& PrepareRecordsForCall(size_t maxRecordsToSend, TInstant now);
        TMaybe<NStockpile::TStockpileWriteManyState>& GetInFlightCall();

        std::pair<size_t, bool> ClearInFlightCall(bool unblock, TMap<TInstant, size_t>& startTimesToUpdate);
        std::pair<size_t, TDuration> RequeueRecordsAndBlock(TInstant now, bool applyJitterToBlockInterval);

    private:
        void UpdateNonEmptySince(TInstant timestamp);

        const NStockpile::TStockpileShardId ShardId;

        TDeque<TRecordInfo> Records;
        NMonitoring::TIntGauge* QueueSize;
        TMaybe<TInstant> GrowUntilAge;
        TMaybe<TInstant> BlockedSince;
        TMaybe<TDuration> BlockedFor;

        TMaybe<NStockpile::TStockpileWriteManyState> InFlightCall;
        TVector<TRecordInfo> InFlightRecords;
    };

    struct TStockpileQueueProcessorMetrics {
    public:
        TStockpileQueueProcessorMetrics() {
            auto* registry = NMonitoring::TMetricRegistry::Instance();
            PointsInRequest = registry->HistogramRate(
                {{"sensor", "stockpile.request.point_count"}},
                NMonitoring::ExponentialHistogram(30, 2, 1));

            MetricsInRequest = registry->HistogramRate(
                {{"sensor", "stockpile.request.metric_count"}},
                NMonitoring::ExponentialHistogram(30, 2, 1));

            BytesInRequest = registry->HistogramRate(
                {{"sensor", "stockpile.request.byte_count"}},
                NMonitoring::ExponentialHistogram(30, 2, 1));

            WriteBySize = registry->Rate({{"sensor", "stockpile.request.by_size"}});
            WriteByAge = registry->Rate({{"sensor", "stockpile.request.by_age"}});
        }

    public:
        NMonitoring::THistogram* PointsInRequest{};
        NMonitoring::THistogram* MetricsInRequest{};
        NMonitoring::THistogram* BytesInRequest{};
        NMonitoring::IRate* WriteBySize{};
        NMonitoring::IRate* WriteByAge{};
    };

    class TStockpileQueueProcessor {
    public:
        TStockpileQueueProcessor(TLog& logger, NStockpile::TStockpileState& state, const TStockpileWriterSettings& settings,
            size_t index, TStockpileDumperStats& stats);
        ~TStockpileQueueProcessor();

        void AddRecord(const IRecordDescriptor& record, TInstant now);
        bool IsActive() const;
        void Update();
        TMaybe<TInstant> GetLastTime();

        NStockpile::TMetabaseShardKey ToShardKey(const IRecordDescriptor& desc) const;
        void EnqueueMetabaseShard(const NStockpile::TMetabaseShardKey& shardKey);

        void Finish();

    private:
        using TRecordsPerMetabaseShard = THashMap<NStockpile::TMetabaseShardKey, TMetabaseQueue>;
        using TRecordsPerStockpileShard = THashMap<NStockpile::TStockpileShardId, TStockpileQueue>;

        bool TryAddRecordToStockpile(IRecordDescriptorPtr& record);
        void AddRecordToMetabase(IRecordDescriptorPtr record);
        void AddRecordToStockpile(const NStockpile::TSensorId& sensorId, NStockpile::TNumId metabaseShardNumId,
                                  IRecordDescriptorPtr record);
        void DumpIfReady(bool onIdle);

        TMetabaseQueue& GetMetabaseQueue(NStockpile::TMetabaseShardKey shardKey);
        TStockpileQueue& GetStockpileQueue(NStockpile::TStockpileShardId shardId);

        NStockpile::TSeriesKey MakeOneSeriesKey(const IRecordDescriptor& record);
        NStockpile::TTypedSeriesKey MakeOneTypedSeriesKey(const IRecordDescriptor& record);
        TVector<NStockpile::TTypedSeriesKey> MakeTypedSeriesKeys(const TVector<IRecordDescriptorPtr>& records);

        const NStockpile::TMetabaseShardState MoveResolvedRecordsToStockpile(TMetabaseQueue& queue);
        bool DispatchInFlightMetabaseCall(TMetabaseQueue& queue, const NStockpile::TMetabaseShardState& shardState) noexcept;
        TVector<NStockpile::TTypedSeriesKey> PrepareScheduledRecords(TMetabaseQueue& queue) noexcept;
        bool ResolveOneQueue(TMetabaseQueue& queue); // return false if queue should be rescheduled
        void SafeResolveOneQueue(TMetabaseQueue& queue);

        bool DispatchInFlightStockpileCall(TStockpileQueue& queue) noexcept;
        void WriteOneQueue(TStockpileQueue& queue, bool forceNotReadyQueues);
        void SafeWriteOneQueue(TStockpileQueue& queue, bool forceNotReadyQueues);

        void CheckWriting() const;
        void DumpRecords(bool noDelay = false, bool forceNotReadyQueues = false);
        void ResolveRecords(TInstant now);
        void WriteRecords(bool forceNotReadyQueues);

        void ClearScheduledRecords(TMetabaseQueue& queue);
        std::pair<size_t, bool> ClearInFlightCall(TStockpileQueue& queue, bool unblock);

        void UpdateStartTime();

        void UpdateCapacityStats();

        size_t Index;
        TLog& Logger;
        NStockpile::TStockpileState& State;
        TStockpileTopologyListener* TopologyListener;

        const TStockpileWriterSettings Settings;
        const TRecordPeriod FiveSecondsPeriod;

        TRecordsPerMetabaseShard RecordsToResolve;
        TMetabaseQueue::TLinkedList ChangedMetabaseQueues;
        TMetabaseQueue::TTree DelayedMetabaseQueues;

        TRecordsPerStockpileShard RecordsToWrite;

        TMap<TInstant, size_t> StartTimes;
        size_t RecordCount;
        size_t RecordNumber;
        TStockpileDumperStats& StockpileStats;
        TStockpileQueueProcessorMetrics Metrics;

        TAtomic LastStartTime;
        NStockpile::TGrpcStateHandler GrpcHandler;

        TInstant LastDisabledLogEntryTime;
    };

    class TStockpileWriter: public ISnapshotVisitor {
    public:
        TStockpileWriter(TLog&, NStockpile::TStockpileState& state, const TStockpileWriterSettings& settings,  size_t index,
            TStockpileDumperStats& stats);

        void OnRecord(const IRecordDescriptor& recordDescriptor) override;

        TMaybe<TInstant> GetLastTime() override;

        void CollectStats() override;

        void Finish() override;

        bool IsActive() const override;
        void Update() override;


    private:
        TLog& Logger;
        NStockpile::TStockpileState& StockpileState;

        const TStockpileWriterSettings Settings;
        const TRecordPeriod FiveSecondsPeriod;
        const TRecordPeriod FiveMinutesPeriod;

        TAtomic FlushOffset;

        TStockpileQueueProcessor Processor;
    };
}
