#include "stockpile_writer.h"

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <infra/yasm/histdb/components/dumper/common_ut.h>

using namespace NHistDb;
using namespace NHistDb::NStockpile;

Y_UNIT_TEST_SUITE(TStockpileQueueTest) {
    Y_UNIT_TEST(NormalLifeCycle) {
        TStockpileShardId shardId = 1;
        TStockpileQueue queue(shardId);

        auto curTime = TInstant::Zero();
        for (size_t recordId: xrange(GROW_STOCKPILE_QUEUE_UNTIL_SIZE - 1)) {
            queue.EnqueueRecord(
                TSensorId(shardId, recordId, yandex::solomon::model::MetricType::DSUMMARY),
                0,
                IRecordDescriptorPtr(new TMockedRecordDescriptor()),
                curTime);

            // not ready and not blocked as there are not enough records and it is too early
            UNIT_ASSERT(!queue.IsReadyByTime(curTime));
            UNIT_ASSERT(!queue.IsReadyBySize(false));
            UNIT_ASSERT(!queue.IsBlocked(curTime));
        }

        curTime += GROW_STOCKPILE_QUEUE_UNTIL_AGE;

        // enough time passed, queue should be ready by time
        UNIT_ASSERT(queue.IsReadyByTime(curTime));
        UNIT_ASSERT(!queue.IsReadyBySize(false));
        UNIT_ASSERT(!queue.IsBlocked(curTime));

        queue.EnqueueRecord(
            TSensorId(shardId, GROW_STOCKPILE_QUEUE_UNTIL_SIZE - 1, yandex::solomon::model::MetricType::DSUMMARY),
            0,
            IRecordDescriptorPtr(new TMockedRecordDescriptor()),
            curTime);

        // size should be big enough now
        UNIT_ASSERT(queue.IsReadyByTime(curTime));
        UNIT_ASSERT(queue.IsReadyBySize(false));
        UNIT_ASSERT(!queue.IsBlocked(curTime));

        for (size_t recordId: xrange(GROW_STOCKPILE_QUEUE_UNTIL_SIZE)) {
            queue.EnqueueRecord(
                TSensorId(shardId, GROW_STOCKPILE_QUEUE_UNTIL_SIZE + recordId, yandex::solomon::model::MetricType::DSUMMARY),
                0,
                IRecordDescriptorPtr(new TMockedRecordDescriptor()),
                curTime);

            // still adding records without sending a call, queue remains ready
            UNIT_ASSERT(queue.IsReadyByTime(curTime));
            UNIT_ASSERT(queue.IsReadyBySize(false));
            UNIT_ASSERT(!queue.IsBlocked(curTime));
        }

        TStockpileDumperStats stats(2);
        TLog log;
        TPortManager portManager;
        TStringStream url;
        auto port = portManager.GetPort();
        url << "localhost:" << port;
        TAtomicSharedPtr<TGrpcRemoteHost> host(new TStockpileRemoteHost(
            TString("host1.cluster1"),
            port,
            grpc::CreateChannel(url.Str(), grpc::InsecureChannelCredentials())
        ));
        auto shard = MakeAtomicShared<TStockpileShard>(shardId, host, log);

        queue.PrepareRecordsForCall(GROW_STOCKPILE_QUEUE_UNTIL_SIZE, curTime);
        queue.GetInFlightCall().ConstructInPlace(shard, log, stats);

        // call is created, so queue should not be ready now
        UNIT_ASSERT(!queue.IsReadyByTime(curTime));
        UNIT_ASSERT(!queue.IsReadyBySize(false));
        UNIT_ASSERT(!queue.IsBlocked(curTime));
        UNIT_ASSERT(queue.IsInFlight());

        curTime += GROW_STOCKPILE_QUEUE_UNTIL_AGE;
        // imagine the call failed and is retriable
        queue.RequeueRecordsAndBlock(curTime, false);

        // queue should be ready, but blocked
        UNIT_ASSERT(!queue.IsInFlight());
        UNIT_ASSERT(queue.IsReadyByTime(curTime));
        UNIT_ASSERT(queue.IsReadyBySize(false));
        UNIT_ASSERT(queue.IsBlocked(curTime));

        // check queue get unblocked
        curTime += STOCKPILE_QUEUE_INITIAL_BLOCK_INTERVAL;
        UNIT_ASSERT(!queue.IsInFlight());
        UNIT_ASSERT(queue.IsReadyByTime(curTime));
        UNIT_ASSERT(queue.IsReadyBySize(false));
        UNIT_ASSERT(!queue.IsBlocked(curTime));

        // retry call
        queue.PrepareRecordsForCall(GROW_STOCKPILE_QUEUE_UNTIL_SIZE, curTime);
        queue.GetInFlightCall().ConstructInPlace(shard, log, stats);
        TMap<TInstant, size_t> startTimes;
        queue.ClearInFlightCall(true, startTimes);

        // queue should be still ready by size
        UNIT_ASSERT(!queue.IsInFlight());
        UNIT_ASSERT(!queue.IsReadyByTime(curTime));
        UNIT_ASSERT(queue.IsReadyBySize(false));
        UNIT_ASSERT(!queue.IsBlocked(curTime));
    }

    Y_UNIT_TEST(BlockIntervalGrows) {
        TStockpileShardId shardId = 1;
        TStockpileQueue queue(shardId);
        auto startTime = TInstant::Zero();

        for (size_t recordId: xrange(GROW_STOCKPILE_QUEUE_UNTIL_SIZE)) {
            queue.EnqueueRecord(
                TSensorId(shardId, recordId, yandex::solomon::model::MetricType::DSUMMARY),
                0,
                IRecordDescriptorPtr(new TMockedRecordDescriptor()),
                startTime);
        }

        TStockpileDumperStats stats(2);
        TLog log;
        TPortManager portManager;
        TStringStream url;
        auto port = portManager.GetPort();
        url << "localhost:" << port;
        TAtomicSharedPtr<TGrpcRemoteHost> host(new TStockpileRemoteHost(
            TString("host1.cluster1"),
            port,
            grpc::CreateChannel(url.Str(), grpc::InsecureChannelCredentials())
        ));
        auto shard = MakeAtomicShared<TStockpileShard>(shardId, host, log);
        queue.PrepareRecordsForCall(GROW_STOCKPILE_QUEUE_UNTIL_SIZE, startTime);
        queue.GetInFlightCall().ConstructInPlace(shard, log, stats);
        auto lastCallAt = startTime;
        queue.RequeueRecordsAndBlock(startTime, false);

        auto endTime = startTime + STOCKPILE_QUEUE_MAX_BLOCK_INTERVAL * 3;
        TVector<TDuration> interCallIntervals;
        for (auto curTime = startTime; curTime < endTime; curTime += TDuration::Seconds(1)) {
            UNIT_ASSERT(queue.IsReadyBySize(false));
            if (!queue.IsBlocked(curTime)) {
                interCallIntervals.push_back(curTime - lastCallAt);
                lastCallAt = curTime;
                queue.PrepareRecordsForCall(GROW_STOCKPILE_QUEUE_UNTIL_SIZE, curTime);
                queue.GetInFlightCall().ConstructInPlace(shard, log, stats);
                queue.RequeueRecordsAndBlock(curTime, false);
            }
        }

        UNIT_ASSERT(interCallIntervals.size() > 1);
        UNIT_ASSERT(interCallIntervals.back() == STOCKPILE_QUEUE_MAX_BLOCK_INTERVAL);
        for (size_t i = 0; i < interCallIntervals.size(); ++i) {
            auto curInterval = interCallIntervals[i];
            UNIT_ASSERT(curInterval >= STOCKPILE_QUEUE_INITIAL_BLOCK_INTERVAL);
            UNIT_ASSERT(curInterval <= STOCKPILE_QUEUE_MAX_BLOCK_INTERVAL);
            if (i > 0) {
                auto prevInterval = interCallIntervals[i - 1];
                if (prevInterval < STOCKPILE_QUEUE_MAX_BLOCK_INTERVAL) {
                    UNIT_ASSERT(curInterval > prevInterval);
                } else {
                    UNIT_ASSERT(curInterval == prevInterval);
                }
            }
        }
    }
}
