#pragma once

#include <library/cpp/blockcodecs/codecs.h>

#include <util/generic/buffer.h>
#include <util/generic/maybe.h>
#include <util/generic/vector.h>
#include <util/stream/direct_io.h>
#include <util/stream/output.h>
#include <util/stream/zerocopy.h>
#include <util/system/direct_io.h>
#include <util/memory/blob.h>

namespace NHistDb {
    struct TSnappyBlock {
        ui64 UncompressedSize;
        ui64 CompressedSize;
    };

    enum class ESnappyMode {
        READ = 0,
        APPEND = 1
    };

    class TSnappyOutputStream: public IOutputStream {
    public:
        TSnappyOutputStream(const TString& fileName, const TVector<TSnappyBlock>& blocks,
                            bool useDirect=true, size_t chunkSize=0, const NBlockCodecs::ICodec* codec=nullptr);
        TSnappyOutputStream(const TString& fileName, bool useDirect=true,
                            size_t chunkSize=0, const NBlockCodecs::ICodec* codec=nullptr);

        const TVector<TSnappyBlock>& Blocks() noexcept {
            return Blocks_;
        }

        size_t Position() noexcept {
            return Position_;
        }

    private:
        void DoWrite(const void* buf, size_t len) override;
        void DoFlush() override;

        void CompressBufferAndFlush();

        TDirectIOBufferedFile OutputFile_;
        TRandomAccessFileOutput OutputStream_;

        const size_t ChunkSize_;
        const NBlockCodecs::ICodec* Codec_;

        TBuffer Buffer_;
        TBuffer CompressedBuffer_;

        size_t Position_ = 0;

        TVector<TSnappyBlock> Blocks_;
    };

    class TSnappyInputStream: public IZeroCopyInput {
    public:
        TSnappyInputStream(const TString& fileName, const TVector<TSnappyBlock>& blocks,
                           size_t chunkSize=0, const NBlockCodecs::ICodec* codec=nullptr);

        void Seek(i64 offset, SeekDir origin=sSet);

        size_t Position() noexcept {
            return Position_;
        }

        size_t Size() noexcept {
            return Size_;
        }

    private:
        void ReadNextBlock(size_t len=0);
        size_t DoNext(const void** ptr, size_t len) override;

        TFile InputFile_;
        const TVector<TSnappyBlock> Blocks_;
        const size_t Size_ = 0;

        const size_t ChunkSize_;
        const NBlockCodecs::ICodec* Codec_;

        size_t BlockOffset_ = 0;
        // Index of last read block
        ssize_t BufferedBlockIndex_ = -1;
        // Index of next block
        ssize_t NextBlockIndex_ = 0;

        size_t Position_ = 0;

        TVector<size_t> CompressedPositions_;
        TVector<size_t> UncompressedPositions_;

        TBuffer RawBuffer_;
        TBuffer BlockBuffer_;
        TBuffer ResultBuffer_;
    };

    class TSnappyFile {
    public:
        TSnappyFile(const TString& fileName, ESnappyMode mode, const TVector<TSnappyBlock>& blocks, bool useDirect=true);
        TSnappyFile(const TString& fileName, bool useDirect=true);

        TSnappyInputStream& GetInputStream();
        TSnappyOutputStream& GetOutputStream();

        size_t Position();
        const TVector<TSnappyBlock>& Blocks();

        void Seek(i64 offset, SeekDir origin=sSet);
        size_t Write(TStringBuf buf);
        TStringBuf Read(size_t len, TBuffer& buf);
        TStringBuf Read(size_t len);
        void Flush();
        void Finish();

    private:
        struct TReadPolicy {
            static void OnEmpty(const std::type_info&) {
                ythrow yexception() << TStringBuf("file is opened only for reading");
            }
        };

        struct TWritePolicy {
            static void OnEmpty(const std::type_info&) {
                ythrow yexception() << TStringBuf("file is opened only for writing");
            }
        };

        TMaybe<TSnappyOutputStream, TWritePolicy> Output_;
        TMaybe<TSnappyInputStream, TReadPolicy> Input_;
        TBuffer Buffer_;
    };
}
