#include "snappy.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/xrange.h>

using namespace NHistDb;

const static TString TESTING_FILE = "testing.data";

Y_UNIT_TEST_SUITE(TSnappyStreamsTest) {

    TVector<TSnappyBlock> PrepareBlocks() {
        TVector<TSnappyBlock> blocks;

        {
            // write first line
            TSnappyOutputStream output(TESTING_FILE);
            output.Write("first\n");
            output.Finish();
            blocks = output.Blocks();
        }

        {
            // append second line
            TSnappyOutputStream output(TESTING_FILE, blocks);
            output.Write("second\n");
            output.Finish();
            blocks = output.Blocks();
        }

        return blocks;
    }

    Y_UNIT_TEST(ReadWrite) {
        TVector<TSnappyBlock> blocks(PrepareBlocks());

        TSnappyInputStream input(TESTING_FILE, blocks);
        UNIT_ASSERT_STRINGS_EQUAL(input.ReadLine(), "first");
        UNIT_ASSERT_VALUES_EQUAL(input.Position(), 6);
        UNIT_ASSERT_STRINGS_EQUAL(input.ReadLine(), "second");
        UNIT_ASSERT_VALUES_EQUAL(input.Position(), 13);
        UNIT_CHECK_GENERATED_EXCEPTION(input.ReadLine(), yexception);
        UNIT_ASSERT_VALUES_EQUAL(input.Position(), 13);
    }

    Y_UNIT_TEST(EmptyReading) {
        TVector<TSnappyBlock> blocks;
        TSnappyInputStream input(TESTING_FILE, blocks);
        UNIT_CHECK_GENERATED_EXCEPTION(input.ReadLine(), yexception);
    }

    Y_UNIT_TEST(BigChunk) {
        TString incoming;
        for (const auto index : xrange((1 << 16) + (1 << 8))) {
            incoming.append(index);
        }

        TSnappyOutputStream output(TESTING_FILE);
        output.Write(incoming);
        output.Finish();

        TSnappyInputStream input(TESTING_FILE, output.Blocks());
        UNIT_ASSERT_STRINGS_EQUAL(incoming, input.ReadAll());
    }

    Y_UNIT_TEST(Truncating) {
        TVector<TSnappyBlock> blocks;

        {
            TSnappyOutputStream output(TESTING_FILE);
            output.Write("first\n");
        }

        {
            TSnappyOutputStream output(TESTING_FILE);
            output.Write("second\n");
            output.Finish();
            blocks = output.Blocks();
        }

        TSnappyInputStream input(TESTING_FILE, blocks);
        UNIT_ASSERT_STRINGS_EQUAL(input.ReadLine(), "second");
        UNIT_CHECK_GENERATED_EXCEPTION(input.ReadLine(), yexception);
    }

    Y_UNIT_TEST(Seek) {
        TBuffer buf;
        TVector<TSnappyBlock> blocks(PrepareBlocks());
        UNIT_ASSERT_VALUES_EQUAL(blocks.size(), 2);

        TSnappyInputStream input(TESTING_FILE, blocks);

        input.Seek(blocks[0].UncompressedSize);
        buf.Resize(blocks[1].UncompressedSize);
        UNIT_ASSERT_VALUES_EQUAL(input.Read(buf.Data(), blocks[1].UncompressedSize), buf.Size());

        input.Seek(0);
        buf.Resize(blocks[0].UncompressedSize);
        UNIT_ASSERT_VALUES_EQUAL(input.Read(buf.Data(), blocks[0].UncompressedSize), buf.Size());

        UNIT_ASSERT_VALUES_EQUAL(input.Position(), blocks[0].UncompressedSize);
    }

    Y_UNIT_TEST(NonAlignedSeek) {
        TVector<TSnappyBlock> blocks(PrepareBlocks());

        TSnappyInputStream input(TESTING_FILE, blocks);
        input.Seek(9);

        TBuffer buf;
        buf.Resize(4);
        UNIT_ASSERT_VALUES_EQUAL(input.Read(buf.Data(), buf.Size()), buf.Size());
        UNIT_ASSERT_STRINGS_EQUAL(TStringBuf(buf.Data(), buf.Size()), "ond\n");
    }

    Y_UNIT_TEST(WritePositions) {
        TSnappyOutputStream output(TESTING_FILE);
        output.Write("first\n");
        UNIT_ASSERT_VALUES_EQUAL(output.Position(), 6);
        output.Write("second\n");
        UNIT_ASSERT_VALUES_EQUAL(output.Position(), 13);
    }
}

Y_UNIT_TEST_SUITE(TSnappyFileTest) {
    Y_UNIT_TEST(ReadWrite) {
        TSnappyFile firstWriter(TESTING_FILE);
        firstWriter.Write("first\n");
        UNIT_ASSERT_VALUES_EQUAL(firstWriter.Position(), 6);
        firstWriter.Flush();

        TSnappyFile secondWriter(TESTING_FILE, ESnappyMode::APPEND, firstWriter.Blocks());
        secondWriter.Write("second\n");
        UNIT_ASSERT_VALUES_EQUAL(secondWriter.Position(), 13);
        secondWriter.Flush();

        TSnappyFile reader(TESTING_FILE, ESnappyMode::READ, secondWriter.Blocks());

        reader.Seek(6);
        UNIT_ASSERT_STRINGS_EQUAL(reader.Read(7), "second\n");

        reader.Seek(0);
        UNIT_ASSERT_STRINGS_EQUAL(reader.Read(6), "first\n");
    }
}
