#include "five_minutes_writer.h"
#include "metrics.h"

#include <infra/monitoring/common/perf.h>

using namespace NHistDb;
using namespace NMonitoring;

namespace {
    TInstant ComputeStartTime(TMaybe<TInstant> startTime, const TRecordPeriod& period, TInstant now) {
        if (startTime.Defined()) {
            return *startTime + period.GetResolution();
        } else {
            return period.GetStartTime(now);
        }
    }

    template <class F>
    TVector<NZoom::NHost::THostName> FindHostNames(const TString& root, F&& predicate) {
        const TVector<NZoom::NHost::THostName> hostNames(TSecondPlacementReader::LastHostNames(root, TRecordPeriod::Get("m5")));
        TVector<NZoom::NHost::THostName> result;
        CopyIf(hostNames.begin(), hostNames.end(), std::back_inserter(result), predicate);
        return result;
    }
}

TFiveMinutesVisitor::TFiveMinutesVisitor(NZoom::NAccumulators::EAccumulatorType accumulatorType,
                                         TContinuousSeriesCursor cursor)
    : Accumulator(accumulatorType)
    , Cursor(std::move(cursor))
{
}

void TFiveMinutesVisitor::OnValue(NZoom::NValue::TValueRef value) {
    Accumulator.Mul(value);
}

void TFiveMinutesVisitor::Flush() {
    Cursor.Append(Accumulator.GetValue());
}

TFiveMinutesPerHostWriter::TFiveMinutesPerHostWriter(TLog& logger, const NZoom::NYasmConf::TYasmConf& conf,
                                                     const TString& root, NZoom::NHost::THostName hostName, TInstant now)
    : Logger(logger)
    , Conf(conf)
    , FiveMinutesPeriod(TRecordPeriod::Get("m5"))
    , FiveSecondsPeriod(TRecordPeriod::Get("s5"))
    , HostName(hostName)
    , Writer(root, HostName.GetName(), FiveMinutesPeriod)
    , StartTime(ComputeStartTime(Writer.LastRecordTime(), FiveMinutesPeriod, now))
    , FlushOffset(0)
    , LastTime((StartTime - FiveMinutesPeriod.GetResolution()).GetValue())
    , AccumulatorMapping(Conf)
{
    Logger << ELogPriority::TLOG_INFO << "Start accepting points for five minutes writer for " << HostName << " at " << StartTime;
}

void TFiveMinutesPerHostWriter::OnRecord(const IRecordDescriptor& recordDescriptor) {
    TMeasuredMethod perf(Logger, "", NMetrics::PIPELINE_FIVE_MINUTES_PROCESSING_TIME);

    const NImpl::TSignalKey signalKey(recordDescriptor.GetInstanceKey(), recordDescriptor.GetSignalName());
    const TInstant seriesStartTime = recordDescriptor.GetStartTime();
    const TInstant seriesEndTime = recordDescriptor.GetEndTime();
    Y_VERIFY(seriesEndTime - seriesStartTime < FiveMinutesPeriod.GetResolution());

    if (seriesEndTime < StartTime || seriesStartTime < StartTime) {
        return;
    }


    const auto accumulatorType(AccumulatorMapping.GetAccumulatorType(signalKey));
    if (!accumulatorType.Defined()) {
        return;
    }


    const TInstant currentPointStartTime = FiveMinutesPeriod.GetPointStartTime(seriesStartTime);
    const TInstant nextPointStartTime = currentPointStartTime + FiveMinutesPeriod.GetResolution();
    Y_VERIFY(StartTime <= seriesStartTime && currentPointStartTime <= seriesStartTime && seriesEndTime < nextPointStartTime);

    if (CurrentInstanceKey.Empty()) {
        CurrentInstanceKey = recordDescriptor.GetInstanceKey();
    } else if (recordDescriptor.GetInstanceKey() != CurrentInstanceKey) {
        CurrentInstanceKey = recordDescriptor.GetInstanceKey();
        Writer.Flush(nextPointStartTime);
        CollectStats();
    }

    try {
        TFiveMinutesVisitor visitor(*accumulatorType, Writer.CreateContinuousSeriesCursor(
            signalKey.GetInstanceKey(),
            signalKey.GetSignalName(),
            seriesStartTime));
        recordDescriptor.Iterate(visitor);
        visitor.Flush();
    } catch (...) {
        Logger << ELogPriority::TLOG_ERR
               << "Can't insert five minutes points into "
               << HostName << ";" << recordDescriptor.GetInstanceKey() << ";" << recordDescriptor.GetSignalName()
               << " during the period [" << seriesStartTime << ", " << seriesEndTime << "]: " << CurrentExceptionMessage();
    }

    const TInstant newFlushOffset(FiveMinutesPeriod.GetPointStartTime(recordDescriptor.GetFlushOffset()));
    const TInstant::TValue oldFlushOffset(AtomicSwap(&FlushOffset, newFlushOffset.GetValue()));

    if (oldFlushOffset && newFlushOffset && oldFlushOffset != newFlushOffset.GetValue()) {
        TMeasuredMethod perf(Logger, "", NMetrics::PIPELINE_FIVE_MINUTES_FLUSH_TIME);
        try {
            Writer.Flush(newFlushOffset, true);
            AtomicSet(LastTime, (newFlushOffset - FiveMinutesPeriod.GetResolution()).GetValue());
            Logger << ELogPriority::TLOG_INFO << "Flushed five minutes points for " << HostName << " created before " << newFlushOffset;
        } catch (...) {
            Logger << ELogPriority::TLOG_ERR << "Can't flush points for " << HostName << " created before " << newFlushOffset << ": " << CurrentExceptionMessage();
        }
    }

    TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_FIVE_MINUTES_PROCESSED_RECORDS, 1);
}

void TFiveMinutesPerHostWriter::CollectStats() {
    const auto flushTime(TInstant::FromValue(AtomicGet(FlushOffset)));
    if (flushTime) {
        const TDuration delay(TInstant::Now() - TInstant::FromValue(AtomicGet(FlushOffset)) - FiveMinutesPeriod.GetResolution());
        TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_FIVE_MINUTES_PROCESSING_DELAY, delay.SecondsFloat());
    }
}

TMaybe<TInstant> TFiveMinutesPerHostWriter::GetLastTime() {
    return TInstant::FromValue(AtomicGet(LastTime));
}

void TFiveMinutesPerHostWriter::Finish() {
    try {
        Writer.Finish();
        Logger << ELogPriority::TLOG_INFO << "Five minutes writer for " << HostName << " was removed";
    } catch (...) {
        Logger << ELogPriority::TLOG_ERR << "Five minutes writer for " << HostName << " was removed with error: " << CurrentExceptionMessage();
    }
}

TFiveMinutesWriter::TFiveMinutesWriter(TLog& logger, const NZoom::NYasmConf::TYasmConf& conf, const TString& root, TInstant now)
    : Logger(logger)
    , Conf(conf)
    , Root(root)
    , Now(now)
{
}

THolder<ISnapshotVisitor> TFiveMinutesWriter::CreateVisitor(NZoom::NHost::THostName hostName) {
    return MakeHolder<TFiveMinutesPerHostWriter>(Logger, Conf, Root, hostName, Now);
}

TVector<NZoom::NHost::THostName> THostFiveMinutesWriter::GetHostNames(const TString& root) {
    return FindHostNames(root, [](const auto& x) { return !x.IsGroup(); });
}

TVector<NZoom::NHost::THostName> TGroupFiveMinutesWriter::GetHostNames(const TString& root) {
    return FindHostNames(root, [](const auto& x) { return x.IsGroup(); });
}
