#include "five_minutes_writer.h"

#include <infra/yasm/histdb/components/dumper/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/folder/tempdir.h>
#include <util/generic/xrange.h>

using namespace NHistDb;
using namespace NTags;
using namespace NZoom::NSignal;
using namespace NZoom::NYasmConf;
using namespace NZoom::NValue;

namespace {
    struct TWriterState {
        TWriterState()
            : YasmConf(TYasmConf::FromString(TStringBuf("{\"conflist\": {\"common\": {\"signals\": {}, \"patterns\": {}, \"periods\": {}}}}")))
            , FiveMinutesPeriod(TRecordPeriod::Get("m5"))
            , FiveSecondsPeriod(TRecordPeriod::Get("s5"))
            , StartTime(FiveMinutesPeriod.GetStartTime(TInstant::Now()))
            , Writer(Logger, YasmConf, Root.Name(), StartTime)
        {
        }

        TLog Logger;

        const TYasmConf YasmConf;

        const TRecordPeriod FiveMinutesPeriod;
        const TRecordPeriod FiveSecondsPeriod;
        const TInstant StartTime;

        TTempDir Root;
        TGroupFiveMinutesWriter Writer;
    };
}

Y_UNIT_TEST_SUITE(TFiveMinutesWriterTest) {
    Y_UNIT_TEST(PointsAtTheStartOfChunk) {
        TMockedSnapshotDescriptor snapshot;
        TVector<TValue> values;
        values.emplace_back(42);
        values.emplace_back(42);
        auto& record(snapshot.GetRecordDescriptor().SetValues(values));

        TWriterState state;
        UNIT_ASSERT_VALUES_EQUAL(state.Writer.GetLastTime(), Nothing());

        auto startTime(state.StartTime);
        state.Writer.OnRecord(
            record
                .SetStartTime(startTime)
                .SetEndTime(startTime + state.FiveSecondsPeriod.GetResolution())
                .SetFlushOffset(startTime));
        UNIT_ASSERT_VALUES_EQUAL(TGroupFiveMinutesWriter::GetHostNames(state.Root.Name()), TVector<NZoom::NHost::THostName>{});
        UNIT_ASSERT_VALUES_EQUAL(state.Writer.GetLastTime(), MakeMaybe<TInstant>(startTime - state.FiveMinutesPeriod.GetResolution()));

        startTime += TDuration::Minutes(5);
        state.Writer.OnRecord(
            record
                .SetStartTime(startTime)
                .SetEndTime(startTime + state.FiveSecondsPeriod.GetResolution())
                .SetFlushOffset(startTime + state.FiveMinutesPeriod.GetResolution()));
        UNIT_ASSERT_VALUES_EQUAL(TGroupFiveMinutesWriter::GetHostNames(state.Root.Name()), TVector<NZoom::NHost::THostName>{record.GetHostName()});
        UNIT_ASSERT_VALUES_EQUAL(state.Writer.GetLastTime(), MakeMaybe<TInstant>(startTime));

        TSecondPlacementReader reader(state.Root.Name(), record.GetHostName().GetName(), state.FiveMinutesPeriod, state.StartTime);
        auto checker(TResponseChecker::Read(
            reader, TVector<TSomethingFormat::TTimestamp>{0}, TRequestKey::FromString("itype=testing"), TVector<TSignalName>{record.GetSignalName()}));
        checker.SeekForTimestamp(0).CheckSignal(record.GetSignalName(), TValue(84));

        TGroupFiveMinutesWriter otherWriter(state.Logger, state.YasmConf, state.Root.Name(), startTime);
        otherWriter.PreloadHostName(record.GetHostName());
        UNIT_ASSERT_VALUES_EQUAL(otherWriter.GetLastTime(), MakeMaybe<TInstant>(startTime));
    }

    Y_UNIT_TEST(PointsBeforeStartTime) {
        TWriterState state;

        TMockedSnapshotDescriptor snapshot;
        TVector<TValue> values;
        values.emplace_back(1);
        auto& record(snapshot.GetRecordDescriptor().SetValues(values));

        auto startTime(state.StartTime);
        state.Writer.OnRecord(
            record
                .SetStartTime(startTime - state.FiveSecondsPeriod.GetResolution())
                .SetEndTime(startTime - state.FiveSecondsPeriod.GetResolution())
                .SetFlushOffset(startTime));

        values.clear();
        values.emplace_back(7);
        values.emplace_back(11);
        state.Writer.OnRecord(
            record
                .SetValues(values)
                .SetStartTime(startTime)
                .SetEndTime(startTime + state.FiveSecondsPeriod.GetResolution())
                .SetFlushOffset(startTime));

        startTime += state.FiveMinutesPeriod.GetResolution();
        values.clear();
        values.emplace_back(13);
        state.Writer.OnRecord(
            record
                .SetValues(values)
                .SetStartTime(startTime)
                .SetEndTime(startTime)
                .SetFlushOffset(startTime));

        try {
            TSecondPlacementReader reader(state.Root.Name(), record.GetHostName().GetName(), state.FiveMinutesPeriod, state.StartTime - state.FiveMinutesPeriod.GetInterval());
            UNIT_FAIL("chunk before start time exists");
        } catch (...) {
        }

        TSecondPlacementReader reader(state.Root.Name(), record.GetHostName().GetName(), state.FiveMinutesPeriod, state.StartTime);
        auto checker(TResponseChecker::Read(
            reader, TVector<TSomethingFormat::TTimestamp>{0, 1}, TRequestKey::FromString("itype=testing"), TVector<TSignalName>{record.GetSignalName()}));
        checker
            .SeekForTimestamp(0)
            .CheckSignal(record.GetSignalName(), TValue(18))
            .CheckTimestampIsMissing(1);
    }

    Y_UNIT_TEST(Flush) {
        TWriterState state;

        TMockedSnapshotDescriptor snapshot;
        TVector<TValue> values;
        for (const auto idx : xrange(state.FiveMinutesPeriod.GetResolution().GetValue() / state.FiveSecondsPeriod.GetResolution().GetValue())) {
            values.emplace_back(idx + 1);
        }
        auto& record(snapshot.GetRecordDescriptor().SetValues(values));

        auto startTime(state.StartTime);
        auto endTime(startTime + state.FiveMinutesPeriod.GetResolution() - state.FiveSecondsPeriod.GetResolution());
        state.Writer.OnRecord(
            record
                .SetStartTime(startTime)
                .SetEndTime(endTime)
                .SetFlushOffset(state.StartTime));

        startTime += state.FiveMinutesPeriod.GetResolution();
        endTime += state.FiveMinutesPeriod.GetResolution();
        state.Writer.OnRecord(
            record
                .SetStartTime(startTime)
                .SetEndTime(endTime)
                .SetFlushOffset(state.StartTime + state.FiveMinutesPeriod.GetResolution()));

        TSecondPlacementReader reader(state.Root.Name(), record.GetHostName().GetName(), state.FiveMinutesPeriod, state.StartTime);
        auto checker(TResponseChecker::Read(
            reader, TVector<TSomethingFormat::TTimestamp>{0, 1}, TRequestKey::FromString("itype=testing"), TVector<TSignalName>{record.GetSignalName()}));
        checker
            .SeekForTimestamp(0)
            .CheckSignal(record.GetSignalName(), TValue(1830))
            .CheckTimestampIsMissing(1);
    }

    Y_UNIT_TEST(FlushOutdatedSeries) {
        TWriterState state;

        TMockedSnapshotDescriptor snapshot;
        TVector<TValue> values;
        values.emplace_back(42);
        auto& record(snapshot.GetRecordDescriptor().SetValues(values));
        const auto signalName(record.GetSignalName());

        auto startTime(state.StartTime);
        state.Writer.OnRecord(record.SetStartTime(startTime).SetEndTime(startTime).SetFlushOffset(startTime));

        startTime += state.FiveMinutesPeriod.GetResolution();
        state.Writer.OnRecord(
            record
                .SetSignalName(TStringBuf("network_usage_summ"))
                .SetStartTime(startTime)
                .SetEndTime(startTime)
                .SetFlushOffset(startTime));

        TSecondPlacementReader reader(state.Root.Name(), record.GetHostName().GetName(), state.FiveMinutesPeriod, state.StartTime);
        auto checker(TResponseChecker::Read(
            reader, TVector<TSomethingFormat::TTimestamp>{0}, TRequestKey::FromString("itype=testing"), TVector<TSignalName>{signalName}));
        checker.SeekForTimestamp(0).CheckSignal(signalName, TValue(42));
    }
}
