#include "five_seconds_writer.h"
#include "metrics.h"

#include <infra/monitoring/common/perf.h>

using namespace NHistDb;
using namespace NMonitoring;

namespace {

    class TFiveSecondsPerHostWriter final: public ISnapshotVisitor {
    public:
        TFiveSecondsPerHostWriter(TLog& logger, const TString& root, NZoom::NHost::THostName hostName, TInstant now);

        void OnRecord(const IRecordDescriptor& recordDescriptor) override;

        TMaybe<TInstant> GetLastTime() override;

        void CollectStats() override;

        void Finish() override;
    private:
        TLog& Logger;

        const TRecordPeriod FiveMinutesPeriod;
        const TRecordPeriod FiveSecondsPeriod;
        const NZoom::NHost::THostName HostName;

        TPlacementContinuousWriter Writer;
        const TInstant StartTime;

        TMaybe<std::pair<TInstant, NTags::TInstanceKey>> CurrentKey;
        TMaybe<TEncodedSeriesCursor> EncodedSeriesCursor;
        TAtomic FlushOffset;
        TAtomic LastTime;
    };

    TInstant ComputeStartTime(TMaybe<TInstant> lastTime, const TRecordPeriod& period, TInstant now) {
        if (lastTime.Defined()) {
            return *lastTime + period.GetResolution();
        } else {
            return period.GetStartTime(now);
        }
    }

    template <class F>
    TVector<NZoom::NHost::THostName> FindHostNames(const TString& root, F&& predicate) {
        const TVector<NZoom::NHost::THostName> hostNames(TSecondPlacementReader::LastHostNames(root, TRecordPeriod::Get("s5")));
        TVector<NZoom::NHost::THostName> result;
        CopyIf(hostNames.begin(), hostNames.end(), std::back_inserter(result), predicate);
        return result;
    }
}

TFiveSecondsPerHostWriter::TFiveSecondsPerHostWriter(TLog& logger, const TString& root, NZoom::NHost::THostName hostName, TInstant now)
    : Logger(logger)
    , FiveMinutesPeriod(TRecordPeriod::Get("m5"))
    , FiveSecondsPeriod(TRecordPeriod::Get("s5"))
    , HostName(hostName)
    , Writer(root, HostName.GetName(), FiveSecondsPeriod)
    , StartTime(ComputeStartTime(Writer.LastRecordTime(), FiveSecondsPeriod, now))
    , FlushOffset(0)
    , LastTime((StartTime - FiveSecondsPeriod.GetResolution()).GetValue())
{
    Logger << ELogPriority::TLOG_INFO << "Start accepting points for five seconds writer for " << HostName << " at " << StartTime;
}

void TFiveSecondsPerHostWriter::OnRecord(const IRecordDescriptor& recordDescriptor) {
    TMeasuredMethod perf(Logger, "", NMetrics::PIPELINE_FIVE_SECONDS_PROCESSING_TIME);

    const TInstant seriesStartTime = recordDescriptor.GetStartTime();
    const TInstant seriesEndTime = recordDescriptor.GetEndTime();
    Y_VERIFY(seriesEndTime - seriesStartTime < FiveMinutesPeriod.GetResolution());

    if (seriesEndTime < StartTime || seriesStartTime < StartTime) {
        return;
    }

    const TInstant currentPointStartTime = FiveMinutesPeriod.GetPointStartTime(seriesStartTime);
    const TInstant nextPointStartTime = currentPointStartTime + FiveMinutesPeriod.GetResolution();
    Y_VERIFY(
        StartTime <= seriesStartTime && currentPointStartTime <= seriesStartTime && seriesEndTime < nextPointStartTime && recordDescriptor.GetFlushOffset() <= currentPointStartTime);

    if (CurrentKey.Empty()) {
        CurrentKey.ConstructInPlace(currentPointStartTime, recordDescriptor.GetInstanceKey());
    } else if (currentPointStartTime != CurrentKey->first || recordDescriptor.GetInstanceKey() != CurrentKey->second) {
        if (EncodedSeriesCursor.Defined()) {
            EncodedSeriesCursor->Commit();
            EncodedSeriesCursor.Clear();
        }
        CollectStats();
        CurrentKey.ConstructInPlace(currentPointStartTime, recordDescriptor.GetInstanceKey());
    }

    try {
        if (EncodedSeriesCursor.Empty() && Writer.SwitchToCompact(seriesStartTime)) {
            EncodedSeriesCursor.ConstructInPlace(Writer.CreateEncodedSeriesCursor(recordDescriptor.GetInstanceKey(), seriesStartTime));
        }

        if (EncodedSeriesCursor.Defined()) {
            EncodedSeriesCursor->Append(
                recordDescriptor.GetSignalName(),
                seriesStartTime,
                recordDescriptor.GetValuesCount(),
                recordDescriptor.GetSeriesKind(),
                recordDescriptor.GetData());
        }

    } catch (...) {
        Logger << ELogPriority::TLOG_ERR
               << "Can't insert five seconds points into "
               << HostName << ";" << recordDescriptor.GetInstanceKey() << ";" << recordDescriptor.GetSignalName()
               << " during the period [" << seriesStartTime << ", " << seriesEndTime << "]: " << CurrentExceptionMessage();
    }

    const TInstant newFlushOffset(FiveMinutesPeriod.GetPointStartTime(recordDescriptor.GetFlushOffset()));
    const TInstant::TValue oldFlushOffset(AtomicSwap(&FlushOffset, newFlushOffset.GetValue()));

    if (oldFlushOffset && newFlushOffset && oldFlushOffset != newFlushOffset.GetValue()) {
        TMeasuredMethod perf(Logger, "", NMetrics::PIPELINE_FIVE_SECONDS_FLUSH_TIME);
        try {
            Writer.Flush(newFlushOffset, true);
            AtomicSet(LastTime, (newFlushOffset - FiveSecondsPeriod.GetResolution()).GetValue());
            Logger << ELogPriority::TLOG_INFO << "Flushed five seconds points for " << HostName << " created before " << newFlushOffset;
        } catch (...) {
            Logger << ELogPriority::TLOG_ERR << "Can't flush five seconds points for " << HostName << " created before " << newFlushOffset << ": " << CurrentExceptionMessage();
        }
    }

    TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_FIVE_SECONDS_PROCESSED_RECORDS, 1);
}

void TFiveSecondsPerHostWriter::CollectStats() {
    const auto flushTime(TInstant::FromValue(AtomicGet(FlushOffset)));
    if (flushTime) {
        const TDuration delay(TInstant::Now() - TInstant::FromValue(AtomicGet(FlushOffset)) - FiveMinutesPeriod.GetResolution());
        TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_FIVE_SECONDS_PROCESSING_DELAY, delay.SecondsFloat());
    }
}

TMaybe<TInstant> TFiveSecondsPerHostWriter::GetLastTime() {
    return TInstant::FromValue(AtomicGet(LastTime));
}

void TFiveSecondsPerHostWriter::Finish() {
    try {
        Writer.Finish();
        Logger << ELogPriority::TLOG_INFO << "Five seconds writer for " << HostName << " was removed";
    } catch (...) {
        Logger << ELogPriority::TLOG_ERR << "Five seconds writer for " << HostName << " was removed with error: " << CurrentExceptionMessage();
    }
}

TFiveSecondsWriter::TFiveSecondsWriter(TLog& logger, const TString& root, TInstant now)
    : Logger(logger)
    , Root(root)
    , Now(now)
{
}

THolder<ISnapshotVisitor> TFiveSecondsWriter::CreateVisitor(NZoom::NHost::THostName hostName) {
    return MakeHolder<TFiveSecondsPerHostWriter>(Logger, Root, hostName, Now);
}

TVector<NZoom::NHost::THostName> THostFiveSecondsWriter::GetHostNames(const TString& root) {
    return FindHostNames(root, [](const auto& x) { return !x.IsGroup(); });
}

TVector<NZoom::NHost::THostName> TGroupFiveSecondsWriter::GetHostNames(const TString& root) {
    return FindHostNames(root, [](const auto& x) { return x.IsGroup(); });
}
