#include "five_seconds_writer.h"

#include <infra/yasm/histdb/components/dumper/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/folder/tempdir.h>
#include <util/generic/xrange.h>

using namespace NHistDb;
using namespace NTags;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;

namespace {
    struct TWriterState {
        TWriterState()
            : Period(TRecordPeriod::Get("s5"))
            , StartTime(Period.GetStartTime(TInstant::Now()))
            , Writer(Logger, Root.Name(), StartTime)
        {
        }

        TLog Logger;

        const TRecordPeriod Period;
        const TInstant StartTime;

        TTempDir Root;
        TGroupFiveSecondsWriter Writer;
    };
}

Y_UNIT_TEST_SUITE(TFiveSecondsWriterTest) {
    Y_UNIT_TEST(PointsAtTheStartOfChunk) {
        TMockedSnapshotDescriptor snapshot;
        TVector<TValue> values;
        values.emplace_back(24);
        values.emplace_back(42);
        auto& record(snapshot.GetRecordDescriptor().SetValues(values));

        TWriterState state;
        UNIT_ASSERT_VALUES_EQUAL(state.Writer.GetLastTime(), Nothing());

        auto startTime(state.StartTime);
        state.Writer.OnRecord(
            record
                .SetStartTime(startTime)
                .SetEndTime(startTime + state.Period.GetResolution())
                .SetFlushOffset(startTime));
        UNIT_ASSERT_VALUES_EQUAL(TGroupFiveSecondsWriter::GetHostNames(state.Root.Name()), TVector<NZoom::NHost::THostName>{});
        UNIT_ASSERT_VALUES_EQUAL(state.Writer.GetLastTime(), MakeMaybe<TInstant>(startTime - state.Period.GetResolution()));

        startTime += TDuration::Minutes(5);
        state.Writer.OnRecord(
            record
                .SetStartTime(startTime)
                .SetEndTime(startTime + state.Period.GetResolution())
                .SetFlushOffset(startTime));
        UNIT_ASSERT_VALUES_EQUAL(TGroupFiveSecondsWriter::GetHostNames(state.Root.Name()), TVector<NZoom::NHost::THostName>{record.GetHostName()});
        UNIT_ASSERT_VALUES_EQUAL(state.Writer.GetLastTime(), MakeMaybe<TInstant>(startTime - state.Period.GetResolution()));

        startTime += TDuration::Minutes(5);
        state.Writer.OnRecord(
            record
                .SetStartTime(startTime)
                .SetEndTime(startTime + state.Period.GetResolution())
                .SetFlushOffset(startTime));

        TSecondPlacementReader reader(state.Root.Name(), record.GetHostName().GetName(), state.Period, state.StartTime);
        const TVector<TSomethingFormat::TTimestamp> timestamps = {0, 60};
        auto checker(TResponseChecker::Read(
            reader, timestamps, TRequestKey::FromString("itype=testing"), TVector<TSignalName>{record.GetSignalName()}));
        for (const auto timestamp : timestamps) {
            checker
                .SeekForTimestamp(timestamp)
                .CheckSignal(record.GetSignalName(), values.front());
        }

        // only two values was added to the series
        TGroupFiveSecondsWriter otherWriter(state.Logger, state.Root.Name(), state.StartTime);
        otherWriter.PreloadHostName(record.GetHostName());
        UNIT_ASSERT_VALUES_EQUAL(otherWriter.GetLastTime(), MakeMaybe<TInstant>(startTime - TDuration::Minutes(5) + state.Period.GetResolution()));
    }

    Y_UNIT_TEST(PointsBeforeStartTime) {
        TWriterState state;

        TMockedSnapshotDescriptor snapshot;
        TVector<TValue> values;
        values.emplace_back(42);
        auto& record(snapshot.GetRecordDescriptor().SetValues(values));

        auto startTime(state.StartTime);
        state.Writer.OnRecord(
            record
                .SetStartTime(startTime - state.Period.GetResolution())
                .SetEndTime(startTime - state.Period.GetResolution())
                .SetFlushOffset(startTime));

        state.Writer.OnRecord(
            record
                .SetStartTime(startTime)
                .SetEndTime(startTime)
                .SetFlushOffset(startTime));

        startTime += TDuration::Minutes(5);
        state.Writer.OnRecord(
            record
                .SetStartTime(startTime)
                .SetEndTime(startTime)
                .SetFlushOffset(startTime));

        try {
            TSecondPlacementReader reader(state.Root.Name(), record.GetHostName().GetName(), state.Period, state.StartTime - state.Period.GetInterval());
            UNIT_FAIL("chunk before start time exists");
        } catch (...) {
        }

        TSecondPlacementReader reader(state.Root.Name(), record.GetHostName().GetName(), state.Period, state.StartTime);
        auto checker(TResponseChecker::Read(
            reader, TVector<TSomethingFormat::TTimestamp>{0, 1}, TRequestKey::FromString("itype=testing"), TVector<TSignalName>{record.GetSignalName()}));
        checker
            .SeekForTimestamp(0)
            .CheckSignal(record.GetSignalName(), values.back())
            .CheckTimestampIsMissing(1);
    }

    Y_UNIT_TEST(Flush) {
        const TDuration fiveMinutes = TDuration::Minutes(5);

        TWriterState state;

        TMockedSnapshotDescriptor snapshot;
        TVector<TValue> values;
        for (const auto idx : xrange(fiveMinutes.GetValue() / state.Period.GetResolution().GetValue())) {
            values.emplace_back(idx + 1);
        }
        auto& record(snapshot.GetRecordDescriptor().SetValues(values));

        auto startTime(state.StartTime);
        auto endTime(startTime + fiveMinutes - state.Period.GetResolution());
        state.Writer.OnRecord(
            record
                .SetStartTime(startTime)
                .SetEndTime(endTime)
                .SetFlushOffset(state.StartTime));

        startTime += fiveMinutes;
        endTime += fiveMinutes;
        state.Writer.OnRecord(
            record
                .SetStartTime(startTime)
                .SetEndTime(endTime)
                .SetFlushOffset(state.StartTime + fiveMinutes));

        TSecondPlacementReader reader(state.Root.Name(), record.GetHostName().GetName(), state.Period, state.StartTime);
        auto checker(TResponseChecker::Read(
            reader, TVector<TSomethingFormat::TTimestamp>{0, 1, 59, 60}, TRequestKey::FromString("itype=testing"), TVector<TSignalName>{record.GetSignalName()}));
        checker
            .SeekForTimestamp(0)
            .CheckSignal(record.GetSignalName(), TValue(1))
            .SeekForTimestamp(1)
            .CheckSignal(record.GetSignalName(), TValue(2))
            .SeekForTimestamp(59)
            .CheckSignal(record.GetSignalName(), TValue(60))
            .CheckTimestampIsMissing(60);
    }
}
