#include "pipeline.h"
#include "metrics.h"

#include <library/cpp/unistat/unistat.h>

using namespace NHistDb;

TWritingPipeline::TWritingPipeline(TLog& logger, const NZoom::NYasmConf::TYasmConf& conf, const TString& root,
                                   size_t threadCount, TInstant now)
    : Logger(logger)
    , Conf(conf)
    , Root(root)
    , Now(now)
    , HostSplitter(Logger)
    , GroupSplitter(Logger)
{
    for (const auto idx : xrange(threadCount)) {
        CreateHostThread();
        CreateGroupThread();
        Y_UNUSED(idx);
    }

    if (threadCount > 0) {
        Preload();
    }
}

void TWritingPipeline::OnRecord(const IRecordDescriptor& recordDescriptor) {
    if (recordDescriptor.GetHostName().IsGroup()) {
        GroupSplitter.OnRecord(recordDescriptor);
    } else {
        HostSplitter.OnRecord(recordDescriptor);
    }
}

TMaybe<TInstant> TWritingPipeline::GetHostOrGroupLastTime(TInstant limit, bool hostChunks) {
    TMinTimeFinderWithLimit finder(limit);
    auto& writers = (hostChunks) ? HostUnitingWriters : GroupUnitingWriters;
    for (auto& writer : writers) {
        writer.AddLastTimesToFinder(finder);
    }
    if (finder.GetIgnoredCount() > finder.GetAcceptedCount()) {
        // too many ignored values, play it safe, return the limit
        return limit;
    } else {
        return finder.Result();
    }
}

TMaybe<TInstant> TWritingPipeline::GetLastTime() {
    TMinTimeFinder finder;
    AddLastTimesToFinder(finder);
    return finder.Result();
}

void TWritingPipeline::AddLastTimesToFinder(ITimeFinder& finder) {
    for (auto& writer : HostUnitingWriters) {
        writer.AddLastTimesToFinder(finder);
    }
    for (auto& writer : GroupUnitingWriters) {
        writer.AddLastTimesToFinder(finder);
    }
}

void TWritingPipeline::CreateHostThread() {
    HostFiveSecondsWriters.emplace_back(Logger, Root, Now);
    HostFiveMinutesWriters.emplace_back(Logger, Conf, Root, Now);
    HostUnitingWriters.emplace_back();
    HostUnitingWriters.back().RegisterVisitor(HostFiveSecondsWriters.back());
    HostUnitingWriters.back().RegisterVisitor(HostFiveMinutesWriters.back());
    HostSplitter.RegisterVisitor(HostUnitingWriters.back());
}

void TWritingPipeline::CreateGroupThread() {
    GroupFiveSecondsWriters.emplace_back(Logger, Root, Now);
    GroupFiveMinutesWriters.emplace_back(Logger, Conf, Root, Now);
    GroupUnitingWriters.emplace_back();
    GroupUnitingWriters.back().RegisterVisitor(GroupFiveSecondsWriters.back());
    GroupUnitingWriters.back().RegisterVisitor(GroupFiveMinutesWriters.back());
    GroupSplitter.RegisterVisitor(GroupUnitingWriters.back());
}

void TWritingPipeline::Preload() {
    for (const auto& hostName : THostFiveSecondsWriter::GetHostNames(Root)) {
        HostSplitter.PreloadHostName(hostName);
    }
    for (const auto& hostName : THostFiveMinutesWriter::GetHostNames(Root)) {
        HostSplitter.PreloadHostName(hostName);
    }

    for (const auto& hostName : TGroupFiveSecondsWriter::GetHostNames(Root)) {
        GroupSplitter.PreloadHostName(hostName);
    }
    for (const auto& hostName : TGroupFiveMinutesWriter::GetHostNames(Root)) {
        GroupSplitter.PreloadHostName(hostName);
    }
}

void TWritingPipeline::Start() {
    HostSplitter.Start();
    GroupSplitter.Start();
}

void TWritingPipeline::Stop() {
    GroupSplitter.Stop();
    HostSplitter.Stop();
}

void TWritingPipeline::Finish() {
    GroupSplitter.Finish();
    HostSplitter.Finish();
}
