#pragma once

#include <infra/monitoring/common/settings.h>

#include <infra/yasm/stockpile_client/common/base_types.h>

#include <util/generic/hash_set.h>
#include <util/generic/string.h>
#include <util/system/env.h>

namespace NHistDb {
    class TDumperSettings final: public NMonitoring::TSettings {
    public:
        inline const TFsPath& GetSnapshotsDirectory() const {
            return SnapshotsDirectory;
        }

        inline void SetSnapshotsDirectory(const TString& value) {
            SnapshotsDirectory = value;
        }

        inline void SetDumpFilePath(const TString& value) {
            DumpFilePath = value;
        }

        inline const TFsPath& GetDumpFilePath() const {
            return DumpFilePath;
        }

        inline size_t GetStockpileThreadCount() const {
            return StockpileThreadCount;
        }

        inline void SetStockpileThreadCount(size_t value) {
            StockpileThreadCount = value;
        }

        NStockpile::TSettings CreateStockpileSettings() const {
            return StockpileSettingsFactory.CreateStockpileSettings(true);
        }

        void ProcessParsedOptions(const NLastGetopt::TOptsParseResult& parsed) {
            if (parsed.Has("snapshots-dir")) {
                SetSnapshotsDirectory(parsed.Get("snapshots-dir"));
            }
            if (parsed.Has("stockpile-thread-count")) {
                SetStockpileThreadCount(parsed.Get<size_t>("stockpile-thread-count"));
            }
            if (parsed.Has("dump-state-file-path")) {
                SetDumpFilePath(parsed.Get("dump-state-file-path"));
            }
            StockpileSettingsFactory.ProcessParsedOptions(parsed);
        }

        static void FillParser(NLastGetopt::TOpts& options) {
            options.AddLongOption("snapshots-dir", "directory with tsdb snapshots").Optional().RequiredArgument();
            options.AddLongOption("stockpile-thread-count", "number of threads for stockpile dump").Optional().RequiredArgument();
            options.AddLongOption("dump-state-file-path", "where save dumper state").Optional().RequiredArgument();

            NStockpile::TStockpileSetingsFactory::FillParser(options);
        }

    protected:
        void FillFromFileConfig(const NConfig::TConfig& config) override {
            TSettings::FillFromFileConfig(config);
            SetSnapshotsDirectory(config["snapshots_dir"].As<TString>("snapshots"));
            SetStockpileThreadCount(config["stockpile_thread_count"].As<size_t>(StockpileThreadCount));
            SetDumpFilePath(config["dump_state_file_path"].As<TString>("snapshots/dumper_state.data"));

            StockpileSettingsFactory.FillFromFileConfig(config);
        }

    private:
        TFsPath SnapshotsDirectory;
        TFsPath DumpFilePath;
        size_t StockpileThreadCount = 0;

        NStockpile::TStockpileSetingsFactory StockpileSettingsFactory;
    };
}
