#include "snapshot_reader.h"
#include "metrics.h"

#include <infra/yasm/histdb/components/placements/periods.h>
#include <infra/yasm/zoom/components/compression/bit_stream.h>
#include <infra/yasm/zoom/components/compression/counted_sum.h>
#include <infra/yasm/zoom/components/compression/histograms.h>
#include <infra/yasm/zoom/components/compression/zoom_converters.h>

#include <util/generic/xrange.h>
#include <util/random/random.h>
#include <library/cpp/unistat/unistat.h>

using namespace NHistDb;
using namespace NYasmServer;
using TParsedSnapshotName = TSnapshotReader::TParsedSnapshotName;

namespace {
    static const NZoom::NValue::TValue EMPTY_VALUE;
    static constexpr auto CHUNK_DURATION = TDuration::Minutes(5);
    static constexpr auto RESOLUTION = TDuration::Seconds(5);
    static constexpr auto VALUES_COUNT = CHUNK_DURATION.GetValue() / RESOLUTION.GetValue();

    TDuration ApplyJitter(TDuration duration) {
        auto halfDuration = TDuration::FromValue(duration.GetValue() / 2);
        auto jitter = TDuration::FromValue(RandomNumber(halfDuration.GetValue()));
        return halfDuration + jitter;
    }

    class TValuesCountMismatch: public NPersistence::TExpectedSnapshotError {
    };

    class TRecordAdapter : public IRecordDescriptor {
    public:
        TRecordAdapter(NPersistence::TDataChunk dataChunk)
            : DataChunk(std::move(dataChunk))
        {
            if (dataChunk.ValuesCount != VALUES_COUNT) {
                ythrow TValuesCountMismatch() << "Unexpected values count in data chunk. Got " << dataChunk.ValuesCount << " values";
            }
        }

        TRecordAdapter(TRecordAdapter&& other)
            : DataChunk(std::move(other.DataChunk))
        {
        }

        NZoom::NHost::THostName GetHostName() const override {
            return {DataChunk.Host};
        }

        NTags::TInstanceKey GetInstanceKey() const override {
            return DataChunk.Key;
        }

        NZoom::NSignal::TSignalName GetSignalName() const override {
            return DataChunk.Signal;
        }

        TInstant GetStartTime() const override {
            return DataChunk.Start;
        }

        TInstant GetEndTime() const override {
            return DataChunk.Start + CHUNK_DURATION - RESOLUTION;
        }

        TInstant GetFlushOffset() const override {
            return DataChunk.Start;
        }

        void Iterate(IRecordVisitor& visitor) const override {
            switch (DataChunk.Kind) {
                case ESeriesKind::Double: {
                    Decode<TXorDoubleDecoder>(visitor);
                    break;
                };
                case ESeriesKind::CountedSum: {
                    Decode<TCountedSumDecoder>(visitor);
                    break;
                }
                case ESeriesKind::Histogram: {
                    Decode<THistogramDecoder>(visitor);
                    break;
                }
            }
        }

        size_t GetValuesCount() const override {
            return DataChunk.ValuesCount;
        }

        NYasmServer::ESeriesKind GetSeriesKind() const override {
            return DataChunk.Kind;
        }

        const TString& GetData() const override {
            return DataChunk.Data;
        }

        THolder<IRecordDescriptor> Clone() const override {
            return MakeHolder<TRecordAdapter>(DataChunk);
        }

        THolder<IRecordDescriptor> CloneWithNewKey(const NTags::TInstanceKey& newKey) const override {
            NPersistence::TDataChunk newChunk = DataChunk;
            newChunk.Key = newKey;
            return MakeHolder<TRecordAdapter>(std::move(newChunk));
        }

    private:
        template <class TDecoder>
        void Decode(IRecordVisitor& visitor) const {
            TDecoder decoder;
            size_t bitPosition = 0;
            for (const auto idx : xrange(DataChunk.ValuesCount)) {
                bool notNull = ReadFromBitStream(DataChunk.Data, bitPosition, 1);
                if (!notNull) {
                    visitor.OnValue(EMPTY_VALUE.GetValue());
                } else {
                    const auto value(ToZoom(decoder.Read(DataChunk.Data, bitPosition)));
                    visitor.OnValue(value.GetValue());
                }
                Y_UNUSED(idx);
            }
        }

    protected:
        NPersistence::TDataChunk DataChunk;
    };

    class TSnapshotAdapter {
    public:
        TSnapshotAdapter(TLog& logger, ISnapshotVisitor& visitor, TInstant startingOffset)
            : Visitor(visitor)
            , StartingOffset(TRecordPeriod::Get("m5").GetPointStartTime(startingOffset))
            , Reader(logger)
            , Logger(logger)
        {
        }

        virtual ~TSnapshotAdapter() = default;

        TMaybe<TParsedSnapshotName> ReadNext(const TFsPath& root, const TMaybe<TParsedSnapshotName>& lastProcessed, bool ignoreExpectedError) {
            const auto descriptor = GetNextFile(root, lastProcessed);
            if (descriptor.Defined()) {
                try {
                    auto callback = [this](NPersistence::TDataChunk&& dataChunk) { ReadCallback(std::move(dataChunk)); };
                    Reader.ReadFile(descriptor->Path, callback, NPersistence::EErrorMode::Throw);
                } catch (NPersistence::TExpectedSnapshotError&) {
                    if (ignoreExpectedError) {
                        Logger << TLOG_WARNING << "Skipping invalid snapshot file " << descriptor->Path;
                        TUnistat::Instance().PushSignalUnsafe(NHistDb::NMetrics::PIPELINE_SKIPPED_BAD_SNAPSHOTS, 1);
                    } else {
                        throw;
                    }
                }
            } else {
                Reader.NothingToRead();
            }
            return descriptor;
        }

        ISnapshotVisitor& GetVisitor() const {
            return Visitor;
        }

    protected:
        virtual void ReadCallback(NPersistence::TDataChunk&& dataChunk) = 0;

    private:
        TMaybe<TParsedSnapshotName> GetNextFile(const TFsPath& root, const TMaybe<TParsedSnapshotName>& lastProcessed) {
            for (auto& descriptor : Reader.ListDirectory(root)) {
                if (descriptor.Timestamp < StartingOffset) {
                    continue;
                }
                if (lastProcessed.Defined() && descriptor <= *lastProcessed) {
                    continue;
                }
                return descriptor;
            }
            return Nothing();
        }

        ISnapshotVisitor& Visitor;
        TInstant StartingOffset;
        NPersistence::TSnapshotReader Reader;
        TLog& Logger;
    };

    class THostSnapshotAdapter final: public TSnapshotAdapter {
    public:
        using TSnapshotAdapter::TSnapshotAdapter;

    protected:
        void ReadCallback(NPersistence::TDataChunk&& dataChunk) override {
            const TRecordAdapter adapter(std::move(dataChunk));
            if (!adapter.GetHostName().IsGroup()) {
                GetVisitor().OnRecord(adapter);
            }
        }
    };

    class TGroupSnapshotAdapter final: public TSnapshotAdapter {
    public:
        using TSnapshotAdapter::TSnapshotAdapter;

    protected:
        void ReadCallback(NPersistence::TDataChunk&& dataChunk) override {
            const TRecordAdapter adapter(std::move(dataChunk));
            if (adapter.GetHostName().IsGroup()) {
                GetVisitor().OnRecord(adapter);
            }
        }
    };

    class TStockpileSnapshotAdapter final: public TSnapshotAdapter {
    public:
        using TSnapshotAdapter::TSnapshotAdapter;

    protected:
        void ReadCallback(NPersistence::TDataChunk&& dataChunk) override {
            const TRecordAdapter adapter(std::move(dataChunk));
            GetVisitor().OnRecord(adapter);
        }
    };
}

TSnapshotReader::TSnapshotReader(TLog& logger, const TFsPath& root, TInstant startingOffset, ISnapshotVisitor& visitor)
    : Logger(logger)
    , Root(root)
    , StartingOffset(startingOffset)
    , Visitor(visitor)
    , Done(0)
{
}

TSnapshotReader::~TSnapshotReader() {
    Stop();
}

void* TSnapshotReader::DoRun(void* data) noexcept {
    TSnapshotReader* self = static_cast<TSnapshotReader*>(data);

    while (!AtomicGet(self->Done)) {
        bool hasMore = true;
        try {
            while (self->OnTick()) {
                if (AtomicGet(self->Done)) {
                    return nullptr;
                }
            }
            hasMore = false;
        } catch (const NPersistence::TExpectedSnapshotError &ex) {
            self->Attempt++;
            self->Logger << TLOG_WARNING << ex.what() << ": Retrying " << self->Attempt << " of " << self->MAX_RETRIES;
            TUnistat::Instance().PushSignalUnsafe(NHistDb::NMetrics::PIPELINE_SNAPSHOT_RETRIES, 1);
        } catch (...) {
            const auto message(CurrentExceptionMessage());
            self->Logger.Write(ELogPriority::TLOG_ERR, message.data(), message.size());
            TUnistat::Instance().PushSignalUnsafe(NHistDb::NMetrics::PIPELINE_SNAPSHOT_RETRIES, 1);
        }

        if (hasMore) {
            self->Event.WaitT(TDuration::Seconds(1));
        } else {
            self->Event.WaitT(ApplyJitter(TDuration::Minutes(3)));
        }
    }

    return nullptr;
}

void TSnapshotReader::Start() {
    auto guard = Guard(ThreadMutex);
    Y_VERIFY(!Thread);

    Thread.Reset(new TThread(TThread::TParams(DoRun, this).SetName("chunk_reader")));
    Thread->Start();
}

void TSnapshotReader::Stop() {
    auto guard = Guard(ThreadMutex);
    if (Thread) {
        AtomicSet(Done, 1);
        Event.Signal();
        Thread->Join();
        Thread.Reset();
    }
}

bool TSnapshotReader::OnTick() {
    const auto lastProcessed = ReadNext(LastProcessed);
    if (lastProcessed.Defined()) {
        Attempt = 0;
        LastProcessed = lastProcessed;
        return true;
    }
    return false;
}

TMaybe<TParsedSnapshotName> THostSnapshotReader::ReadNext(TMaybe<TParsedSnapshotName> lastProcessed) {
    THostSnapshotAdapter adapter(Logger, Visitor, StartingOffset);
    return adapter.ReadNext(Root, lastProcessed, Attempt >= MAX_RETRIES);
}

TMaybe<TParsedSnapshotName> TGroupSnapshotReader::ReadNext(TMaybe<TParsedSnapshotName> lastProcessed) {
    TGroupSnapshotAdapter adapter(Logger, Visitor, StartingOffset);
    return adapter.ReadNext(Root, lastProcessed, Attempt >= MAX_RETRIES);
}

TMaybe<TParsedSnapshotName> TStockpileSnapshotReader::ReadNext(TMaybe<TParsedSnapshotName> lastProcessed) {
    TStockpileSnapshotAdapter adapter(Logger, Visitor, StartingOffset);
    return adapter.ReadNext(Root, lastProcessed, Attempt >= MAX_RETRIES);
}
