#include "snapshot_reader.h"

#include <infra/yasm/server/persistence/snapshot_manager.h>
#include <infra/monitoring/common/collections.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/folder/path.h>
#include <util/folder/tempdir.h>
#include <util/stream/file.h>

using namespace NYasmServer::NPersistence;
using namespace NZoom::NSignal;
using namespace NTags;

namespace {
    class TMockedSnapshotVisitor final: public NHistDb::ISnapshotVisitor {
    public:
        void OnRecord(const NHistDb::IRecordDescriptor& recordDescriptor) override {
            LastRecord = recordDescriptor.Clone();
            Event.Signal();
        }

        TMaybe<TInstant> GetLastTime() override {
            return Nothing();
        }

        void WaitForRecord() {
            Event.Wait();
        }

        NHistDb::IRecordDescriptor& GetLastRecord() const {
            UNIT_ASSERT(LastRecord);
            return *LastRecord;
        }

    private:
        TAutoEvent Event;
        THolder<NHistDb::IRecordDescriptor> LastRecord;
    };
}

void CheckRecord(const NHistDb::IRecordDescriptor &record, const TDataChunk &chunk) {
    UNIT_ASSERT_VALUES_EQUAL(record.GetHostName(), chunk.Host);
    UNIT_ASSERT_VALUES_EQUAL(record.GetInstanceKey(), chunk.Key);
    UNIT_ASSERT_VALUES_EQUAL(record.GetSignalName(), chunk.Signal);
    UNIT_ASSERT_VALUES_EQUAL(record.GetStartTime(), chunk.Start);
    UNIT_ASSERT_VALUES_EQUAL(record.GetEndTime(), chunk.Start + TDuration::Seconds(295));
    UNIT_ASSERT_VALUES_EQUAL(record.GetFlushOffset(), chunk.Start);
}

Y_UNIT_TEST_SUITE(TSnapshotReaderTest) {
    Y_UNIT_TEST(ReadAndWrite) {
        TTempDir dir;
        TDataChunk chunk1{
            .Key = TInstanceKey::FromNamed("base"),
            .Signal = TSignalName(TStringBuf("signal1")),
            .Host = "host1",
            .Start = TInstant::Seconds(12345),
            .Kind = NYasmServer::ESeriesKind::Double,
            .Data = "",
            .ValuesCount = 60};

        TSnapshotManager manager(dir.Path(), TLoggerOperator<TNullLog>::Log(), 1, 1);
        manager.WriteChunks({chunk1}, ToString(TInstant::Now().Seconds()));

        TMockedSnapshotVisitor visitor;
        NHistDb::THostSnapshotReader reader(TLoggerOperator<TNullLog>::Log(), dir.Path(), TInstant::Zero(), visitor);
        reader.Start();
        visitor.WaitForRecord();
        reader.Stop();

        CheckRecord(visitor.GetLastRecord(), chunk1);
    }

    Y_UNIT_TEST(SkipBadSnapshots) {
        TTempDir dir;
        auto now = TInstant::Now().Seconds();
        TString path1 = JoinPaths(TStringBuf(dir.Name()), TStringBuf(TStringBuilder() << now - 5 << "-1.chunks"));
        TString path2 = JoinPaths(TStringBuf(dir.Name()), TStringBuf(TStringBuilder() << now - 5 << "-2.chunks"));

        TFileOutput badSnapshot1(path1);
        badSnapshot1 << "foo";
        badSnapshot1.Finish();

        TFileOutput badSnapshot2(path2);
        badSnapshot2 << "foo bar";
        badSnapshot2.Finish();

        // reader should skip first two chunks and successfully proceed next one
        TDataChunk chunk1{
                .Key = TInstanceKey::FromNamed("base"),
                .Signal = TSignalName(TStringBuf("signal1")),
                .Host = "host1",
                .Start = TInstant::Seconds(12345),
                .Kind = NYasmServer::ESeriesKind::Double,
                .Data = "",
                .ValuesCount = 60};

        TSnapshotManager manager(dir.Path(), TLoggerOperator<TNullLog>::Log(), 1, 1);
        manager.WriteChunks({chunk1}, ToString(now));

        TMockedSnapshotVisitor visitor;
        NHistDb::THostSnapshotReader reader(TLoggerOperator<TNullLog>::Log(), dir.Path(), TInstant::Zero(), visitor);

        reader.Start();
        visitor.WaitForRecord();
        reader.Stop();

        CheckRecord(visitor.GetLastRecord(), chunk1);
    }

    Y_UNIT_TEST(SkipWrongValuesCountSnapshots) {
        TTempDir dir;
        auto now = TInstant::Now().Seconds();

        TDataChunk badChunk{
                .Key = TInstanceKey::FromNamed("base"),
                .Signal = TSignalName(TStringBuf("signal1")),
                .Host = "host1",
                .Start = TInstant::Seconds(12340),
                .Kind = NYasmServer::ESeriesKind::Double,
                .Data = "",
                .ValuesCount = 61};

        TSnapshotManager manager(dir.Path(), TLoggerOperator<TNullLog>::Log(), 1, 1);
        manager.WriteChunks({badChunk}, ToString(now - 5));

        TDataChunk chunk1{
                .Key = TInstanceKey::FromNamed("base"),
                .Signal = TSignalName(TStringBuf("signal1")),
                .Host = "host1",
                .Start = TInstant::Seconds(12345),
                .Kind = NYasmServer::ESeriesKind::Double,
                .Data = "",
                .ValuesCount = 60};

        manager.WriteChunks({chunk1}, ToString(now));

        TMockedSnapshotVisitor visitor;
        NHistDb::THostSnapshotReader reader(TLoggerOperator<TNullLog>::Log(), dir.Path(), TInstant::Zero(), visitor);

        reader.Start();
        visitor.WaitForRecord();
        reader.Stop();

        CheckRecord(visitor.GetLastRecord(), chunk1);
    }
}
