#include "splitters.h"
#include "metrics.h"

#include <library/cpp/unistat/unistat.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/random/random.h>

using namespace NHistDb;
using namespace NHistDb::NImpl;
using namespace NUnistat;

namespace {
    static constexpr TDuration COOLDOWN_DURATION = TDuration::MilliSeconds(50);
    static constexpr TDuration FLUSH_INTERVAL = TDuration::Minutes(5);
    static constexpr i64 COOLDOWN_THRESHOLD = 10000;

    TDuration ApplyJitter(TDuration duration) {
        auto halfDuration = TDuration::FromValue(duration.GetValue() / 2);
        auto jitter = TDuration::FromValue(RandomNumber(halfDuration.GetValue()));
        return halfDuration + jitter;
    }
}

TVisitorExecutor::TVisitorExecutor(TLog& logger, ISnapshotVisitor& visitor)
    : Logger(logger)
    , Visitor(visitor)
    , Done(0)
{
    auto* registry = NMonitoring::TMetricRegistry::Instance();
    QueueSize = registry->IntGauge({{"sensor", "point_processor.queue.size"}});
}

TVisitorExecutor::~TVisitorExecutor() {
    Stop();
}

void* TVisitorExecutor::DoRun(void* data) noexcept {
    TVisitorExecutor* self = static_cast<TVisitorExecutor*>(data);

    while (true) {
        self->Event.WaitT(ApplyJitter(COOLDOWN_DURATION));

        if (AtomicGet(self->Done)) {
            return nullptr;
        }

        try {
            self->OnTick();
        } catch (...) {
            self->Logger << ELogPriority::TLOG_ERR << CurrentExceptionMessage();
        }
    }

    return nullptr;
}


void TVisitorExecutor::Finish() {
    Y_ASSERT(AtomicGet(Done));
    while (!Queue.IsEmpty()) {
        OnTick();
    }
}

void TVisitorExecutor::Start() {
    auto guard = Guard(ThreadMutex);
    Y_VERIFY(!Thread);

    Thread.Reset(new TThread(TThread::TParams(DoRun, this).SetName("point_processor")));
    Thread->Start();
}

void TVisitorExecutor::Stop() {
    auto guard = Guard(ThreadMutex);
    if (Thread) {
        AtomicSet(Done, 1);
        Event.Signal();
        Thread->Join();
        Thread.Reset();
    }
}

void TVisitorExecutor::OnRecord(const IRecordDescriptor& recordDescriptor) {
    while (Queue.GetCounter().Counter > COOLDOWN_THRESHOLD) {
        TUnistat::Instance().PushSignalUnsafe(NMetrics::PIPELINE_OVERLOADED, 1);
        Sleep(ApplyJitter(COOLDOWN_DURATION));
    }
    QueueSize->Inc();
    Queue.Enqueue(recordDescriptor.Clone().Release());
}

TMaybe<TInstant> TVisitorExecutor::GetLastTime() const {
    return Visitor.GetLastTime();
}

void TVisitorExecutor::PreloadHostName(NZoom::NHost::THostName hostName) {
    Visitor.PreloadHostName(hostName);
}

void TVisitorExecutor::OnTick() {
    THolder<IRecordDescriptor> record;
    while (Visitor.IsActive() && Queue.Dequeue(record)) {
        try {
            QueueSize->Dec();
            Visitor.OnRecord(*record);
        } catch (...) {
            Logger << ELogPriority::TLOG_ERR
                   << "Can't process record "
                   << record->GetHostName() << ";" << record->GetInstanceKey() << ";" << record->GetSignalName()
                   << " period [" << record->GetStartTime() << ", " << record->GetEndTime() << "]: "
                   << CurrentExceptionMessage();
        }
    }
    try {
        Visitor.Update();
        Visitor.CollectStats();
    } catch (...) {
        Logger << ELogPriority::TLOG_ERR
               << "Can't Update: "
               << CurrentExceptionMessage();
    }
}

void THostSplitter::OnRecord(const IRecordDescriptor& recordDescriptor) {
    TInstant flushTime(recordDescriptor.GetFlushOffset());
    flushTime -= TDuration::FromValue(flushTime.GetValue() % FLUSH_INTERVAL.GetValue());

    auto& state(GetState(recordDescriptor.GetHostName()));
    if (flushTime) {
        state.ValidUntil = recordDescriptor.GetFlushOffset() + FLUSH_INTERVAL;
    }
    state.Visitor->OnRecord(recordDescriptor);

    if (flushTime && LastFlush != flushTime) {
        LastFlush = flushTime;
        CleanStates();
    }
}

TMaybe<TInstant> THostSplitter::GetLastTime() {
    TMinTimeFinder finder;
    AddLastTimesToFinder(finder);
    return finder.Result();
}

void THostSplitter::AddLastTimesToFinder(ITimeFinder& finder) {
    TVector<NZoom::NHost::THostName> hostNames;
    {
        TLightReadGuard guard(Mutex);
        hostNames.reserve(Visitors.size());
        for (const auto& pair : Visitors) {
            hostNames.emplace_back(pair.first);
        }
    }
    for (const auto& hostName : hostNames) {
        GetState(hostName).Visitor->AddLastTimesToFinder(finder);
    }
}

void THostSplitter::PreloadHostName(NZoom::NHost::THostName hostName) {
    InsertState(hostName);
}

void THostSplitter::CollectStats() {
    TLightReadGuard guard(Mutex);
    for (const auto& pair : Visitors) {
        pair.second.Visitor->CollectStats();
    }
}

THostSplitter::TVisitorState* THostSplitter::FindState(NZoom::NHost::THostName hostName) {
    TLightReadGuard guard(Mutex);
    const auto it = Visitors.find(hostName);
    if (it != Visitors.end()) {
        return &it->second;
    } else {
        return nullptr;
    }
}

THostSplitter::TVisitorState* THostSplitter::InsertState(NZoom::NHost::THostName hostName) {
    TLightWriteGuard guard(Mutex);
    typename THashMap<NZoom::NHost::THostName, TVisitorState>::insert_ctx context;
    const auto it = Visitors.find(hostName, context);
    if (it != Visitors.end()) {
        return &it->second;
    }
    TVisitorState state{
        .Visitor = CreateVisitor(hostName),
        .ValidUntil = TInstant::Now() + FLUSH_INTERVAL};
    Y_ASSERT(state.Visitor);
    return &Visitors.emplace_direct(context, hostName, std::move(state))->second;
}

THostSplitter::TVisitorState& THostSplitter::GetState(NZoom::NHost::THostName hostName) {
    TVisitorState* state = FindState(hostName);
    if (state == nullptr) {
        state = InsertState(hostName);
    }
    Y_ASSERT(state != nullptr);
    return *state;
}

void THostSplitter::CleanStates() {
    TVector<NZoom::NHost::THostName> expiredHosts;
    {
        TLightReadGuard guard(Mutex);
        for (const auto& pair : Visitors) {
            if (pair.second.ValidUntil < LastFlush) {
                expiredHosts.emplace_back(pair.first);
            }
        }
    }

    TVector<THolder<ISnapshotVisitor>> visitorsToFinish;
    if (expiredHosts) {
        TLightWriteGuard guard(Mutex);
        for (const auto& host : expiredHosts) {
            const auto it(Visitors.find(host));
            if (it != Visitors.end()) {
                visitorsToFinish.emplace_back(std::move(it->second.Visitor));
                Visitors.erase(it);
            }
        }
    }

    for (auto& visitor : visitorsToFinish) {
        visitor->Finish();
    }
}

TBaseQueueSplitter::TBaseQueueSplitter(TLog& logger)
    : Logger(logger)
{
}

void TBaseQueueSplitter::OnRecord(const IRecordDescriptor& recordDescriptor) {
    SelectExecutor(recordDescriptor).OnRecord(recordDescriptor);
}

void TBaseQueueSplitter::Start() {
    for (const auto& executor : Executors) {
        executor->Start();
    }
}

void TBaseQueueSplitter::Finish() {
    for (const auto& executor : Executors) {
        executor->Finish();
    }
}

void TBaseQueueSplitter::Stop() {
    for (const auto& executor : Executors) {
        executor->Stop();
    }
}

TMaybe<TInstant> TBaseQueueSplitter::GetLastTime() {
    TMinTimeFinder finder;
    for (const auto& executor : Executors) {
        finder.Add(executor->GetLastTime());
    }
    return finder.Result();
}

void TBaseQueueSplitter::RegisterVisitor(ISnapshotVisitor& visitor) {
    InsertExecutor(visitor);
}

void TBaseQueueSplitter::InsertExecutor(ISnapshotVisitor& visitor) {
    Executors.emplace_back(MakeHolder<TVisitorExecutor>(Logger, visitor));
}

TVisitorExecutor& TBaseQueueSplitter::GetExecutor(size_t position) {
    return *Executors.at(position);
}

size_t TBaseQueueSplitter::ExecutorCount() const {
    Y_VERIFY(Executors.size());
    return Executors.size();
}

TVisitorExecutor& THostQueueSplitter::SelectExecutor(const IRecordDescriptor& recordDescriptor) {
    return SelectExecutorByHost(recordDescriptor.GetHostName());
}

TVisitorExecutor& THostQueueSplitter::SelectExecutorByHost(NZoom::NHost::THostName hostName) {
    const auto it(ExecutorMap.find(hostName));
    if (it != ExecutorMap.end()) {
        return GetExecutor(it->second);
    } else {
        const auto current(ExecutorMap.size() % ExecutorCount());
        ExecutorMap[hostName] = current;
        return GetExecutor(current);
    }
}

void THostQueueSplitter::PreloadHostName(NZoom::NHost::THostName hostName) {
    SelectExecutorByHost(hostName).PreloadHostName(hostName);
}

NImpl::TVisitorExecutor& TInstanceKeyQueueSplitter::SelectExecutor(const IRecordDescriptor& recordDescriptor) {
    const auto keyHash(MultiHash(recordDescriptor.GetHostName(), recordDescriptor.GetInstanceKey(), recordDescriptor.GetSignalName()));
    return GetExecutor(keyHash % ExecutorCount());
}

void TUnitingVisitor::OnRecord(const IRecordDescriptor& recordDescriptor) {
    for (auto* visitor : Visitors) {
        visitor->OnRecord(recordDescriptor);
    }
}

TMaybe<TInstant> TUnitingVisitor::GetLastTime() {
    TMinTimeFinder finder;
    AddLastTimesToFinder(finder);
    return finder.Result();
}

void TUnitingVisitor::AddLastTimesToFinder(ITimeFinder& finder) {
    for (auto* visitor : Visitors) {
        visitor->AddLastTimesToFinder(finder);
    }
}

void TUnitingVisitor::PreloadHostName(NZoom::NHost::THostName hostName) {
    for (auto* visitor : Visitors) {
        visitor->PreloadHostName(hostName);
    }
}

void TUnitingVisitor::CollectStats() {
    for (auto* visitor : Visitors) {
        visitor->CollectStats();
    }
}

void TUnitingVisitor::RegisterVisitor(ISnapshotVisitor& visitor) {
    Visitors.emplace_back(&visitor);
}
