#pragma once

#include <infra/yasm/histdb/components/dumper/ifaces.h>

#include <library/cpp/deprecated/atomic/atomic.h>
#include <library/cpp/threading/light_rw_lock/lightrwlock.h>
#include <library/cpp/logger/log.h>
#include <library/cpp/monlib/metrics/metric.h>

#include <util/thread/lfqueue.h>
#include <util/system/thread.h>
#include <util/system/event.h>
#include <util/system/mutex.h>

namespace NHistDb {
    namespace NImpl {
        struct TRecordCounter {
            void IncCount(const IRecordDescriptor*) {
                AtomicIncrement(Counter);
            }

            void DecCount(const IRecordDescriptor*) {
                AtomicDecrement(Counter);
            }

            TAtomic Counter = 0;
        };

        class TRecordQueue: public TLockFreeQueue<IRecordDescriptor*, TRecordCounter> {
        public:
            ~TRecordQueue() {
                THolder<IRecordDescriptor> record;
                while (this->Dequeue(record)) {
                }
            }

            inline bool Dequeue(THolder<IRecordDescriptor>& t) {
                IRecordDescriptor* res = nullptr;

                if (TLockFreeQueue<IRecordDescriptor*, TRecordCounter>::Dequeue(&res)) {
                    t.Reset(res);
                    return true;
                }

                return false;
            }
        };

        class TVisitorExecutor {
        public:
            TVisitorExecutor(TLog& logger, ISnapshotVisitor& visitor);
            ~TVisitorExecutor();

            static void* DoRun(void* self) noexcept;

            void Start();
            void Stop();

            void OnRecord(const IRecordDescriptor& recordDescriptor);

            TMaybe<TInstant> GetLastTime() const;
            void PreloadHostName(NZoom::NHost::THostName hostName);
            void Finish();

        private:
            void OnTick();

            TLog& Logger;

            ISnapshotVisitor& Visitor;
            NImpl::TRecordQueue Queue;
            NMonitoring::TIntGauge* QueueSize;

            TAutoEvent Event;
            TAtomic Done;
            TMutex ThreadMutex;
            THolder<TThread> Thread;
        };
    }

    class THostSplitter: public ISnapshotVisitor {
    public:
        void OnRecord(const IRecordDescriptor& recordDescriptor) override;

        TMaybe<TInstant> GetLastTime() override;
        void AddLastTimesToFinder(ITimeFinder& finder) override;
        void CollectStats() override;
        void PreloadHostName(NZoom::NHost::THostName hostName) override;

        virtual THolder<ISnapshotVisitor> CreateVisitor(NZoom::NHost::THostName hostName) = 0;

    private:
        struct TVisitorState {
            THolder<ISnapshotVisitor> Visitor;
            TInstant ValidUntil;
        };

        TVisitorState* FindState(NZoom::NHost::THostName hostName);
        TVisitorState* InsertState(NZoom::NHost::THostName hostName);
        TVisitorState& GetState(NZoom::NHost::THostName hostName);

        void CleanStates();

        THashMap<NZoom::NHost::THostName, TVisitorState> Visitors;

        TLightRWLock Mutex;
        TInstant LastFlush;
    };

    class TBaseQueueSplitter: public ISnapshotVisitor {
    public:
        TBaseQueueSplitter(TLog& logger);

        void OnRecord(const IRecordDescriptor& recordDescriptor) override;

        TMaybe<TInstant> GetLastTime() override;

        void Start();
        void Stop();
        void Finish() override;

        void RegisterVisitor(ISnapshotVisitor& visitor);

    protected:
        virtual NImpl::TVisitorExecutor& SelectExecutor(const IRecordDescriptor& recordDescriptor) = 0;

        NImpl::TVisitorExecutor& GetExecutor(size_t position);
        void InsertExecutor(ISnapshotVisitor& visitor);
        size_t ExecutorCount() const;

    private:
        TLog& Logger;
        TVector<THolder<NImpl::TVisitorExecutor>> Executors;
    };

    class THostQueueSplitter: public TBaseQueueSplitter {
    public:
        using TBaseQueueSplitter::TBaseQueueSplitter;

        void PreloadHostName(NZoom::NHost::THostName hostName) override;

    protected:
        NImpl::TVisitorExecutor& SelectExecutor(const IRecordDescriptor& recordDescriptor) override;

    private:
        NImpl::TVisitorExecutor& SelectExecutorByHost(NZoom::NHost::THostName hostName);

        THashMap<NZoom::NHost::THostName, size_t> ExecutorMap;
    };

    class TInstanceKeyQueueSplitter: public TBaseQueueSplitter {
    public:
        using TBaseQueueSplitter::TBaseQueueSplitter;

    protected:
        NImpl::TVisitorExecutor& SelectExecutor(const IRecordDescriptor& recordDescriptor) override;
    };

    class TUnitingVisitor final: public ISnapshotVisitor {
    public:
        void OnRecord(const IRecordDescriptor& recordDescriptor) override;

        TMaybe<TInstant> GetLastTime() override;
        void AddLastTimesToFinder(ITimeFinder& finder) override;

        void PreloadHostName(NZoom::NHost::THostName hostName) override;

        void CollectStats() override;

        void RegisterVisitor(ISnapshotVisitor& visitor);

    private:
        TVector<ISnapshotVisitor*> Visitors;
    };
}
