#include "splitters.h"

#include <infra/yasm/histdb/components/dumper/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NHistDb;
using namespace NTags;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;

namespace {
    static constexpr TInstant LAST_TIME = TInstant::Seconds(1530890486);

    class TCounter: public TNonCopyable {
    public:
        void Increment() {
            AtomicIncrement(Counter);
        }

        i64 Get() const {
            return AtomicGet(Counter);
        }

    private:
        TAtomic Counter = 0;
    };

    class TMockedSnapshotVisitor final: public ISnapshotVisitor {
    public:
        TMockedSnapshotVisitor(TCounter& destructions)
            : Destructions(destructions)
        {
        }

        ~TMockedSnapshotVisitor() {
            Destructions.Increment();
        }

        void OnRecord(const IRecordDescriptor&) override {
            AtomicIncrement(VisitCount);
        }

        TMaybe<TInstant> GetLastTime() override {
            return LAST_TIME;
        }

        size_t GetVisitCount() const {
            return AtomicGet(VisitCount);
        }

    private:
        TAtomic VisitCount = 0;
        TCounter& Destructions;
    };

    class TMockedHostSplitter final: public THostSplitter {
    public:
        THolder<ISnapshotVisitor> CreateVisitor(NZoom::NHost::THostName hostName) override {
            HostNames.emplace(hostName);
            return MakeHolder<TMockedSnapshotVisitor>(Destructions);
        }

        const THashSet<NZoom::NHost::THostName>& GetInsertedNames() const {
            return HostNames;
        }

        i64 GetDestructionCount() const {
            return Destructions.Get();
        }

    private:
        THashSet<NZoom::NHost::THostName> HostNames;
        TCounter Destructions;
    };

    template <typename F>
    void Wait(F&& checker) {
        TInstant deadline = TInstant::Now() + TDuration::Seconds(5);
        while (!checker() && deadline > TInstant::Now()) {
            Sleep(TDuration::MilliSeconds(10));
        }
    }
}

Y_UNIT_TEST_SUITE(TUnitingVisitorTest) {
    Y_UNIT_TEST(Visiting) {
        TCounter destructions;
        TMockedSnapshotVisitor mockedVisitor(destructions);

        TUnitingVisitor unitingVisitor;
        unitingVisitor.RegisterVisitor(mockedVisitor);
        unitingVisitor.RegisterVisitor(mockedVisitor);

        TMockedSnapshotDescriptor snapshot;
        unitingVisitor.OnRecord(snapshot.GetRecordDescriptor());

        UNIT_ASSERT_VALUES_EQUAL(mockedVisitor.GetVisitCount(), 2);
        UNIT_ASSERT_VALUES_EQUAL(unitingVisitor.GetLastTime(), MakeMaybe<TInstant>(LAST_TIME));
    }
}

Y_UNIT_TEST_SUITE(THostSplitterTest) {
    Y_UNIT_TEST(Visiting) {
        TMockedHostSplitter splitter;
        splitter.PreloadHostName(TStringBuf("SAS.000"));
        UNIT_ASSERT_VALUES_EQUAL(splitter.GetLastTime(), MakeMaybe<TInstant>(LAST_TIME));
        UNIT_ASSERT(splitter.GetInsertedNames().contains(TStringBuf("SAS.000")));

        TMockedSnapshotDescriptor snapshot;
        splitter.OnRecord(snapshot.GetRecordDescriptor().SetHostName("SAS.001"));
        UNIT_ASSERT(splitter.GetInsertedNames().contains(TStringBuf("SAS.001")));
    }

    Y_UNIT_TEST(Flushing) {
        auto flushInterval(TDuration::Minutes(5));
        auto flushTime(TInstant::Now());
        flushTime = flushTime - TDuration::FromValue(flushTime.GetValue() % flushInterval.GetValue());

        TMockedHostSplitter splitter;
        TMockedSnapshotDescriptor snapshot;
        auto& record(snapshot.GetRecordDescriptor());

        splitter.OnRecord(record.SetHostName("SAS.000").SetFlushOffset(flushTime));
        UNIT_ASSERT_VALUES_EQUAL(splitter.GetDestructionCount(), 0);

        splitter.OnRecord(record.SetHostName("SAS.001").SetFlushOffset(flushTime + flushInterval));
        UNIT_ASSERT_VALUES_EQUAL(splitter.GetDestructionCount(), 0);

        splitter.OnRecord(record.SetHostName("SAS.001").SetFlushOffset(flushTime + flushInterval * 2));
        UNIT_ASSERT_VALUES_EQUAL(splitter.GetDestructionCount(), 1);
    }
}

Y_UNIT_TEST_SUITE(TQueueSplitterTest) {
    Y_UNIT_TEST(HostVisiting) {
        TCounter destructions;
        TMockedSnapshotVisitor visitor(destructions);

        TLog logger;
        THostQueueSplitter splitter(logger);
        splitter.RegisterVisitor(visitor);
        UNIT_ASSERT_VALUES_EQUAL(splitter.GetLastTime(), MakeMaybe<TInstant>(LAST_TIME));
        splitter.Start();

        TMockedSnapshotDescriptor snapshot;
        splitter.OnRecord(snapshot.GetRecordDescriptor().SetHostName("SAS.000"));
        splitter.OnRecord(snapshot.GetRecordDescriptor().SetHostName("sas1-1234.search.yandex.net"));

        Wait([&]() {
            return visitor.GetVisitCount() == 2;
        });

        UNIT_ASSERT_VALUES_EQUAL(visitor.GetVisitCount(), 2);
    }

    Y_UNIT_TEST(InstanceKeyVisiting) {
        TCounter destructions;
        TMockedSnapshotVisitor visitor(destructions);

        TLog logger;
        TInstanceKeyQueueSplitter splitter(logger);
        splitter.RegisterVisitor(visitor);
        UNIT_ASSERT_VALUES_EQUAL(splitter.GetLastTime(), MakeMaybe<TInstant>(LAST_TIME));
        splitter.Start();

        TMockedSnapshotDescriptor snapshot;
        splitter.OnRecord(snapshot.GetRecordDescriptor().SetInstanceKey(NTags::TInstanceKey::FromNamed("itype=first")));
        splitter.OnRecord(snapshot.GetRecordDescriptor().SetInstanceKey(NTags::TInstanceKey::FromNamed("itype=second")));

        Wait([&]() {
            return visitor.GetVisitCount() == 2;
        });

        UNIT_ASSERT_VALUES_EQUAL(visitor.GetVisitCount(), 2);
    }
}
